%GEOMETRY class definition
%
%   Usage:
%      geometry=geometry();

classdef geometry
	properties (SetAccess=public) 
		surface           = NaN;
		thickness         = NaN;
		base              = NaN;
		bed               = NaN;
		hydrostatic_ratio = NaN;
	end
	methods (Static)
		function self = loadobj(self) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. Update old properties here

			%2014 March 26th
			if isstruct(self),
				disp('WARNING: updating geometry');
				disp('         md.geometry.bed        is now md.geometry.base');
				disp('         md.geometry.bathymetry is now md.geometry.bed');
				obj2 = self;
				self = geometry();
				self.surface    = obj2.surface;
				self.thickness  = obj2.thickness;
				self.base       = obj2.bed;
				self.bed        = obj2.bathymetry;
			end

		end% }}}
	end
	methods
		function createxml(self, fid)% {{{
			fprintf(fid, '\n\n');
			fprintf(fid, '%s\n', '<!-- geometry -->');
			fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="Geometry parameters">','<section name="geometry" />'); 
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="surface" type="','path','" optional="','false','">','     <section name="geometry" />','     <help> surface elevation [m] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="thickness" type="','path','" optional="','false','">','     <section name="geometry" />','     <help> ice thickness [m] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="bed" type="','path','" optional="','false','">','     <section name="geometry" />','     <help> bed elevation [m] </help>','</parameter>');
			%fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="bathymetry" type="',class(self.bathymetry),'" default="',convert2str(self.bathymetry),'">','     <section name="geometry" />','     <help> bathymetry elevation [m] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','<parameter key ="hydrostatic_ratio" type="',class(self.hydrostatic_ratio),'" default="',convert2str(self.hydrostatic_ratio),'">','     <section name="geometry" />','     <help> coefficient for ice shelves; thickness correction: hydrostatic_ratio H_obs+ (1-hydrostatic_ratio) H_hydro </help>','</parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');
		end % }}}
		function self = extrude(self,md) % {{{
			self.surface=project3d(md,'vector',self.surface,'type','node');
			self.thickness=project3d(md,'vector',self.thickness,'type','node');
			self.hydrostatic_ratio=project3d(md,'vector',self.hydrostatic_ratio,'type','node');
			self.base=project3d(md,'vector',self.base,'type','node');
			self.bed=project3d(md,'vector',self.bed,'type','node');
		end % }}}
		function self = geometry(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if (solution==TransientSolutionEnum() & md.transient.isgia) | (solution==GiaSolutionEnum()),
				md = checkfield(md,'fieldname','geometry.thickness','timeseries',1,'NaN',1,'>=',0);
			else
				md = checkfield(md,'fieldname','geometry.surface'  ,'NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','geometry.base'      ,'NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','geometry.thickness','NaN',1,'size',[md.mesh.numberofvertices 1],'>',0);
				if any((self.thickness-self.surface+self.base)>10^-9),
					md = checkmessage(md,['equality thickness=surface-base violated']);
				end 
				if solution==TransientSolutionEnum() & md.transient.isgroundingline,
					md = checkfield(md,'fieldname','geometry.bed','NaN',1,'size',[md.mesh.numberofvertices 1]);
				end
			end
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   geometry parameters:'));

			fielddisplay(self,'surface','ice upper surface elevation [m]');
			fielddisplay(self,'thickness','ice thickness [m]');
			fielddisplay(self,'base','ice base elevation [m]');
			fielddisplay(self,'bed','bed elevation [m]');

		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'data',self.surface,'format','DoubleMat','mattype',1,'enum',SurfaceEnum());
			WriteData(fid,'data',self.thickness,'format','DoubleMat','mattype',1,'enum',ThicknessEnum(),'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'data',self.base,'format','DoubleMat','mattype',1,'enum',BaseEnum());
			WriteData(fid,'data',self.bed,'format','DoubleMat','mattype',1,'enum',BedEnum());
			WriteData(fid,'object',self,'fieldname','hydrostatic_ratio','format','DoubleMat','mattype',1);
		end % }}}
	end
end
