%MISFIT class definition
%
%   Usage:
%      misfit=misfit();
%      misfit=misfit('name','SurfaceAltimetry',...
%                    'definitionenum',Outputdefinition1Enum, 'model_enum',SurfaceEnum,...
%                    'observation_enum',SurfaceObservationsEnum,...
%                    'observation',md.geometry.surface,...
%                    'timeinterpolation','nearestneighbor',...
%                    'local',1,...
%                    'weights',ones(md.mesh.numberofvertices,1),...
%                    'weights_enum',WeightsSurfaceObservationsEnum);
%
%

classdef misfit
	properties (SetAccess=public)
		%misfit
		name              = '';
		definitionenum   = NaN; %enum that identifies this output definition uniquely, from Outputdefinition[1-10]Enum
		model_enum        = NaN; %enum for field that is modeled
		observation       = NaN; %observed field that we compare the model against
		observation_enum  = NaN; %enum for observed field.
		timeinterpolation = '';
		local             = 1;
		weights           = NaN; %weight coefficients for every vertex
		weights_enum      = NaN; %enum to identify this particular set of weights
		cumulated         = NaN; %do we cumulate misfit through time?
	end
	
	methods
		function self = misfit(varargin) % {{{
			if nargin==0,
				self=setdefaultparameters(self);
			else
				%use provided options to change fields
				options=pairoptions(varargin{:});

				%get name
				self.name=getfieldvalue(options,'name','');
				self.definitionenum=getfieldvalue(options,'definitionenum');
				self.model_enum=getfieldvalue(options,'model_enum');
				self.observation=getfieldvalue(options,'observation',NaN);
				self.observation_enum=getfieldvalue(options,'observation_enum');
				self.local=getfieldvalue(options,'local',1);
				self.timeinterpolation=getfieldvalue(options,'timeinterpolation','nearestneighbor');
				self.weights=getfieldvalue(options,'weights',NaN);
				self.weights_enum=getfieldvalue(options,'weights_enum',NaN);

			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
			self.local=1;
			self.timeinterpolation='nearestneighbor';
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ischar(self.name),
				error('misfit error message: ''name'' field should be a string!');
			end
			md = checkfield(md,'fieldname','self.definitionenum','field',self.definitionenum,'values',[Outputdefinition1Enum:Outputdefinition100Enum]);

			if ~ischar(self.timeinterpolation),
				error('misfit error message: ''timeinterpolation'' field should be a string!');
			end
			md = checkfield(md,'fieldname','self.observation','field',self.observation,'timeseries',1,'NaN',1);
			md = checkfield(md,'fieldname','self.timeinterpolation','field',self.timeinterpolation,'values',{'nearestneighbor'});
			md = checkfield(md,'fieldname','self.weights','field',self.weights,'timeseries',1,'NaN',1);

		end % }}}
		function md = disp(self) % {{{
		
			disp(sprintf('   Misfit:\n'));

			fielddisplay(self,'name','identifier for this misfit response');
			fielddisplay(self,'definitionenum','enum that identifies this output definition uniquely, from Outputdefinition[1-10]Enum');
			fielddisplay(self,'model_enum','enum for field that is modeled');
			fielddisplay(self,'observation','observed field that we compare the model against');
			fielddisplay(self,'observation_enum','observation enum');
			fielddisplay(self,'local','is the response local to the elements, or global? (default is 1)''');
			fielddisplay(self,'timeinterpolation','interpolation routine used to interpolate misfit between two time steps (default is ''nearestneighbor''');
			fielddisplay(self,'weights','weights (at vertices) to apply to the misfit');
			fielddisplay(self,'weights_enum','enum for weights for identification purposes');

		end % }}}
		function md = marshall(self,md,fid) % {{{

		WriteData(fid,'object',self,'fieldname','name','format','String');
		WriteData(fid,'object',self,'fieldname','definitionenum','format','Integer');
		WriteData(fid,'object',self,'fieldname','model_enum','format','Integer');
		WriteData(fid,'object',self,'fieldname','observation','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
		WriteData(fid,'object',self,'fieldname','observation_enum','format','Integer');
		WriteData(fid,'object',self,'fieldname','local','format','Integer');
		WriteData(fid,'object',self,'fieldname','timeinterpolation','format','String');
		WriteData(fid,'object',self,'fieldname','weights','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
		WriteData(fid,'object',self,'fieldname','weights_enum','format','Integer');

		end % }}}
	end
end
