%MESH3DPRISMS class definition
%
%   Usage:
%      mesh=mesh3dprisms();

classdef mesh3dprisms
	properties (SetAccess=public) 
		x                           = NaN;
		y                           = NaN;
		z                           = NaN;
		elements                    = NaN;
		numberoflayers              = 0;
		numberofelements            = 0;
		numberofvertices            = 0;

		lat                         = NaN;
		long                        = NaN;
		epsg                        = 0;

		vertexonbase                = NaN;
		vertexonsurface             = NaN;
		lowerelements               = NaN;
		lowervertex                 = NaN;
		upperelements               = NaN;
		uppervertex                 = NaN;
		vertexonboundary            = NaN;

		vertexconnectivity          = NaN;
		elementconnectivity         = NaN;
		average_vertex_connectivity = 0;

		x2d                         = NaN;
		y2d                         = NaN;
		elements2d                  = NaN;
		numberofvertices2d          = 0;
		numberofelements2d          = 0;

		extractedvertices           = NaN;
		extractedelements           = NaN;
	end
	methods (Static)
		function self = loadobj(self) % {{{
			% This function is directly called by matlab when a model selfect is
			% loaded. Update old properties here

			%2014 Oct. 1st
			if isstruct(self),
				oldself=self;
				%Assign property values from struct
				self=structtoobj(mesh3dprisms(),oldself);
				if isfield(oldself,'hemisphere'),
					disp('md.mesh.hemisphere has been automatically converted to EPSG code');
					if strcmpi(oldself.hemisphere,'n'),
						self.epsg=3413;
					else
						self.epsg=3031;
					end
				end
			end

		end% }}}
	end
	methods
		function createxml(self,fid) % {{{
			fprintf(fid, '<!-- 3D prism Mesh -->\n');

			% Elements and verticies of the original 2d mesh
			fprintf(fid,'%s\n%s\n%s\n','       <frame key="1" label="Elements and vertices of the orginal 2d mesh">','       <section name="mesh3dprisms" />');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="numberofelements2d" type="',class(self.numberofelements2d),'" default="',convert2str(self.numberofelements2d),'">','              <help> number of elements </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="numberofvertices2d" type="',class(self.numberofvertices2d),'" default="',convert2str(self.numberofvertices2d),'">','              <help> number of vertices </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="elements2d" type="',class(self.elements2d),'" default="',convert2str(self.elements2d),'">','              <help> vertex indices of the mesh elements </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="x2d" type="',class(self.x2d),'" default="',convert2str(self.x2d),'">','     <section name="mesh3dprisms" />','     <help> vertices x coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="y2d" type="',class(self.y2d),'" default="',convert2str(self.y2d),'">','     <section name="mesh3dprisms" />','     <help> vertices y coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','       </frame>');

			% Elements and vertices of the extruded 3d mesh
			fprintf(fid,'%s\n%s\n%s\n','       <frame key="2" label="Elements and vertices of the orginal 3d mesh">','       <section name="mesh3dprisms" />');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="numberofelements" type="',class(self.numberofelements),'" default="',convert2str(self.numberofelements),'">','              <help> number of elements </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="numberofvertices" type="',class(self.numberofvertices),'" default="',convert2str(self.numberofvertices),'">','              <help> number of vertices </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','          <parameter key ="elements" type="',class(self.elements),'" default="',convert2str(self.elements),'">','              <help> vertex indices of the mesh elements </help>','          </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="x" type="',class(self.x),'" default="',convert2str(self.x),'">','     <section name="mesh3dprisms" />','     <help> vertices x coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="y" type="',class(self.y),'" default="',convert2str(self.y),'">','     <section name="mesh3dprisms" />','     <help> vertices y coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="z" type="',class(self.y),'" default="',convert2str(self.y),'">','     <section name="mesh3dprisms" />','     <help> vertices z coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','       </frame>');

			% properties
			fprintf(fid,'%s\n%s\n%s\n','<frame key="3" label="Properties">','<section name="mesh3dprisms" />');             
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="numberoflayers" type="',class(self.numberoflayers),'" default="',convert2str(self.numberoflayers),'">','     <section name="mesh3dprisms" />','     <help> number of extrusion layers </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexonbase" type="',class(self.vertexonbase),'" default="',convert2str(self.vertexonbase),'">','     <section name="mesh3dprisms" />','     <help> lower vertices flags list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="elementontbase" type="',class(self.elementontbase),'" default="',convert2str(self.elementontbase),'">','     <section name="mesh3dprisms" />','     <help> lower elements flags list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexonsurface" type="',class(self.vertexonsurface),'" default="',convert2str(self.vertexonsurface),'">','     <section name="mesh3dprisms" />','     <help> upper vertices flags list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="elementontsurface" type="',class(self.elementontsurface),'" default="',convert2str(self.elementontsurface),'">','     <section name="mesh3dprisms" />','     <help> upper elements flags list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="uppervertex" type="',class(self.uppervertex),'" default="',convert2str(self.uppervertex),'">','     <section name="mesh3dprisms" />','     <help> upper vertex list (NaN for vertex on the upper surface) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="upperelements" type="',class(self.upperelements),'" default="',convert2str(self.upperelements),'">','     <section name="mesh3dprisms" />','     <help> upper element list (NaN for element on the upper layer) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="lowervertex" type="',class(self.lowervertex),'" default="',convert2str(self.lowervertex),'">','     <section name="mesh3dprisms" />','     <help> lower vertex list (NaN for vertex on the lower surface) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="lowerelements" type="',class(self.lowerelements),'" default="',convert2str(self.lowerelements),'">','     <section name="mesh3dprisms" />','     <help> element list (NaN for element on the lower layer) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexonboundary" type="',class(self.vertexonboundary),'" default="',convert2str(self.vertexonboundary),'">','     <section name="mesh3dprisms" />','     <help> vertices on the boundary of the domain flag list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexconnectivity" type="',class(self.vertexconnectivity),'" default="',convert2str(self.vertexconnectivity),'">','     <section name="mesh3dprisms" />','     <help> list of vertices connected to vertex_i </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="elementconnectivity" type="',class(self.elementconnectivity),'" default="',convert2str(self.elementconnectivity),'">','     <section name="mesh3dprisms" />','     <help> average number of vertices connected to one vertex </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="average_vertex_connectivity" type="',class(self.average_vertex_connectivity),'" default="',convert2str(self.average_vertex_connectivity),'">','     <section name="mesh3dprisms" />','     <help> average number of vertices connected to one vertex </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

			% Extracted model
			fprintf(fid,'%s\n%s\n%s\n','<frame key="4" label="Extracted Model">','<section name="mesh3dprisms" />'); 
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="extractedvertices" type="',class(self.extractedvertices),'" default="',convert2str(self.extractedvertices),'">','     <section name="mesh3dprisms" />','     <help> vertices extracted from the model </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="extractedelements" type="',class(self.extractedelements),'" default="',convert2str(self.extractedelements),'">','     <section name="mesh3dprisms" />','     <help> elements extracted from the model </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

			% Projection
			fprintf(fid,'%s\n%s\n%s\n','<frame key="5" label="Projection">','<section name="mesh3dprisms" />'); 
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="lat" type="',class(self.lat),'" default="',convert2str(self.lat),'">','     <section name="mesh3dprisms" />','     <help> vertices latitude [degrees] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="long" type="',class(self.long),'" default="',convert2str(self.long),'">','     <section name="mesh3dprisms" />','     <help> verticies longitude [degrees] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','  <parameter key ="epsg" type="',class(self.epsg),'" default="',convert2str(self.epsg),'">','     <section name="mesh3dprisms" />','     <help> Indicate epsg ''n'' or ''s'' </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

		end % }}}cd
		function self = mesh3dprisms(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					self=mesh3dprisms();
					object=varargin{1};
					fields=fieldnames(object);
					for i=1:length(fields)
						field=fields{i};
						if ismember(field,properties('mesh3dprisms')),
							self.(field)=object.(field);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%the connectivity is the averaged number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			self.average_vertex_connectivity=25;
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','mesh.x','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','mesh.y','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','mesh.z','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','mesh.elements','NaN',1,'>',0,'values',1:md.mesh.numberofvertices);
			md = checkfield(md,'fieldname','mesh.elements','size',[md.mesh.numberofelements 6]);
			if any(~ismember(1:md.mesh.numberofvertices,sort(unique(md.mesh.elements(:)))));
				md = checkmessage(md,'orphan nodes have been found. Check the mesh outline');
			end
			md = checkfield(md,'fieldname','mesh.numberoflayers','>=',0);
			md = checkfield(md,'fieldname','mesh.numberofelements','>',0);
			md = checkfield(md,'fieldname','mesh.numberofvertices','>',0);
			md = checkfield(md,'fieldname','mesh.vertexonbase','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			md = checkfield(md,'fieldname','mesh.vertexonsurface','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			md = checkfield(md,'fieldname','mesh.z','>=',md.geometry.base-10^-10,'message','''mesh.z'' lower than bedrock');
			md = checkfield(md,'fieldname','mesh.z','<=',md.geometry.surface+10^-10,'message','''mesh.z'' higher than surface elevation');
			md = checkfield(md,'fieldname','mesh.average_vertex_connectivity','>=',24,'message','''mesh.average_vertex_connectivity'' should be at least 24 in 3d');
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   3D prism Mesh:')); 

			disp(sprintf('\n      Elements and vertices of the original 2d mesh:'));
			fielddisplay(self,'numberofelements2d','number of elements');
			fielddisplay(self,'numberofvertices2d','number of vertices');
			fielddisplay(self,'elements2d','vertex indices of the mesh elements');
			fielddisplay(self,'x2d','vertices x coordinate [m]');
			fielddisplay(self,'y2d','vertices y coordinate [m]');

			disp(sprintf('\n      Elements and vertices of the extruded 3d mesh:'));
			fielddisplay(self,'numberofelements','number of elements');
			fielddisplay(self,'numberofvertices','number of vertices');
			fielddisplay(self,'elements','vertex indices of the mesh elements');
			fielddisplay(self,'x','vertices x coordinate [m]');
			fielddisplay(self,'y','vertices y coordinate [m]');
			fielddisplay(self,'z','vertices z coordinate [m]');

			disp(sprintf('\n      Properties:'));
			fielddisplay(self,'numberoflayers','number of extrusion layers');
			fielddisplay(self,'vertexonbase','lower vertices flags list');
			fielddisplay(self,'vertexonsurface','upper vertices flags list');
			fielddisplay(self,'uppervertex','upper vertex list (NaN for vertex on the upper surface)');
			fielddisplay(self,'upperelements','upper element list (NaN for element on the upper layer)');
			fielddisplay(self,'lowervertex','lower vertex list (NaN for vertex on the lower surface)');
			fielddisplay(self,'lowerelements','lower element list (NaN for element on the lower layer');
			fielddisplay(self,'vertexonboundary','vertices on the boundary of the domain flag list');

			fielddisplay(self,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(self,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(self,'average_vertex_connectivity','average number of vertices connected to one vertex');

			disp(sprintf('\n      Extracted model:'));
			fielddisplay(self,'extractedvertices','vertices extracted from the model');
			fielddisplay(self,'extractedelements','elements extracted from the model');

			disp(sprintf('\n      Projection:'));
			fielddisplay(self,'lat','vertices latitude [degrees]');
			fielddisplay(self,'long','vertices longitude [degrees]');
			fielddisplay(self,'epsg','EPSG code (ex: 3413 for UPS Greenland, 3031 for UPS Antarctica)');
		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'enum',DomainTypeEnum(),'data',StringToEnum(['Domain' domaintype(self)]),'format','Integer');
			WriteData(fid,'enum',DomainDimensionEnum(),'data',dimension(self),'format','Integer');
			WriteData(fid,'enum',MeshElementtypeEnum(),'data',StringToEnum(elementtype(self)),'format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','z','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',self,'class','mesh','fieldname','numberoflayers','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','vertexonbase','format','BooleanMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','vertexonsurface','format','BooleanMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','lowerelements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',self,'class','mesh','fieldname','upperelements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',self,'class','mesh','fieldname','average_vertex_connectivity','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','elements2d','format','DoubleMat','mattype',3);
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofvertices2d','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofelements2d','format','Integer');
		end % }}}
		function type = domaintype(self) % {{{
			type = '3D';
		end % }}}
		function d = dimension(self) % {{{
			d = 3;
		end % }}}
		function s = elementtype(self) % {{{
			s = 'Penta';
		end % }}}
	end
end
