%MATSEAICE class definition
%
%   Usage:
%      matseaice=matseaice();

classdef matseaice
	properties (SetAccess=public) 
		rho_ice                = 0.;
		poisson                = 0.;
		young_modulus          = 0.;
		ridging_exponent       = 0.;
		cohesion               = 0.;
		internal_friction_coef = 0.;
		compression_coef       = 0.;
		traction_coef          = 0.;
		time_relaxation_stress  = 0.;
		time_relaxation_damage  = 0.;
	end
	methods
		function self = matseaice(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					inputstruct=varargin{1};
					list1 = properties('matseaice');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							self.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%Ice density
			self.rho_ice = 917.; %kg/m^3

			%poisson modulus
			self.poisson=0.3;

			%Young modulus
			self.young_modulus=9.e+9;

			%ridging exponent
			self.ridging_exponent=-20.;

			%Cohesion (Weiss 2007)
			self.cohesion = 40.e+3; %40 kPa

			%internal_friction_coef 
			self.internal_friction_coef=0.7;

			%compression_coef (Bouillon and Rampal 2014)
			self.compression_coef=5./2;

			%traction_coef (Bouillon and Rampal 2014)
			self.traction_coef=5./6.;

			%Time relaxation stress
			%1e20 for the elastic case (perfect memory of the stress), equal to the timestep for the viscous case (no memory of the stress)
			self.time_relaxation_stress=1.e+20;

			%Time relaxation damage
			%1e20 for the brittle case (perfect memory of the damage), equal to the timestep for the plastic case (no memory of the damage)
			self.time_relaxation_damage=1.e+20;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			md = checkfield(md,'fieldname','materials.rho_ice','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.poisson','NaN',1,'>',0,'<',.5,'numel',1);
			md = checkfield(md,'fieldname','materials.young_modulus','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.ridging_exponent','NaN',1,'<',0,'numel',1);
			md = checkfield(md,'fieldname','materials.cohesion','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.internal_friction_coef','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.compression_coef','NaN',1,'>',1,'numel',1);
			md = checkfield(md,'fieldname','materials.traction_coef','NaN',1,'>',0,'<',1,'numel',1);
			md = checkfield(md,'fieldname','materials.time_relaxation_stress','NaN',1,'>',md.timestepping.time_step,'numel',1);
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Sea Ice Material:'));
			fielddisplay(self,'rho_ice','ice density [kg/m^3]');
			fielddisplay(self,'poisson','poisson ratio for undamaged ice [no unit]');
			fielddisplay(self,'young_modulus','Young modulus for undamaged ice [Pa]');
			fielddisplay(self,'ridging_exponent','Riging exponent (c, Hibler parameter) [no unit]');
			fielddisplay(self,'cohesion','cohesion (C) [Pa]');
			fielddisplay(self,'internal_friction_coef','Mohr-Coulomb internal friction coefficient (mu=tan(phi)) [no unit]');
			fielddisplay(self,'compression_coef','Ratio between cutoff compressive strength and the cohesion [no unit]');
			fielddisplay(self,'traction_coef','Ratio between cutoff tensile strength and Mohr-Coulomb tensile strength [no unit]');
			fielddisplay(self,'time_relaxation_stress','Relaxation time for stress (1e+20: elastic, dt: viscous) [s]');
			fielddisplay(self,'time_relaxation_damage','Relaxation time for damage (1e+20: brittle, dt: plastic) [s]');
		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'enum',MaterialsEnum(),'data',MatseaiceEnum(),'format','Integer');
			WriteData(fid,'object',self,'class','materials','fieldname','rho_ice','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','poisson','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','young_modulus','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','ridging_exponent','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','cohesion','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','internal_friction_coef','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','compression_coef','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','traction_coef','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','time_relaxation_stress','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','time_relaxation_damage','format','Double');
		end % }}}
	end
end
