%SEAICEOCEAN Class definition
%
%   Usage:
%      seaiceocean=seaiceocean();

classdef seaiceocean
	properties (SetAccess=public) 
		rho_ocean            = 0.;
		ocean_coef           = 0.;
		ocean_lin_drag_coef  = 0.;
		ocean_quad_drag_coef = 0.;
		ocean_turning_angle  = 0.;
		ocean_ssh            = NaN;
		ocean_vx             = NaN;
		ocean_vy             = NaN;
	end
	methods
		function self = seaiceocean(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%ocean density (kg/m^3)
			self.rho_ocean=1025.;

			%By default, we apply the full stress
			self.ocean_coef = 1.;

			%drag coefficients (different from Hibler 0.0055)
			self.ocean_lin_drag_coef  = 0.;
			self.ocean_quad_drag_coef = 0.004;

			%Turning angle in degrees (McPhee 1998)
			self.ocean_turning_angle = deg2rad(25.);

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			md = checkfield(md,'fieldname','basalforcings.rho_ocean','NaN',1,'>',0.,'numel',1);
			md = checkfield(md,'fieldname','basalforcings.ocean_coef','NaN',1,'>=',0.,'<=',1.,'numel',1);
			md = checkfield(md,'fieldname','basalforcings.ocean_lin_drag_coef','NaN',1,'>=',0.,'numel',1);
			md = checkfield(md,'fieldname','basalforcings.ocean_quad_drag_coef','NaN',1,'>=',0.,'numel',1);
			md = checkfield(md,'fieldname','basalforcings.ocean_turning_angle','NaN',1,'>=',0.,'<',90,'numel',1);
			md = checkfield(md,'fieldname','basalforcings.ocean_ssh','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','basalforcings.ocean_vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','basalforcings.ocean_vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Ocean forcings for sea ice:'));
			%disp(sprintf('\n     tau_b = ocean_coef*rho_ocean*...'));
			fielddisplay(self,'rho_ocean','ocean density [kg/m^3]');
			fielddisplay(self,'ocean_coef','ocean stress spin up coefficient (between 0 and 1)');
			fielddisplay(self,'ocean_lin_drag_coef','ocean linear drag coefficient [Pa/(m/s)]');
			fielddisplay(self,'ocean_quad_drag_coef','ocean quadratic drag coefficient [Pa/(m/s)^2]');
			fielddisplay(self,'ocean_turning_angle','ocean turning angle [rad]');
			fielddisplay(self,'ocean_ssh','ocean sea surface height [m]');
			fielddisplay(self,'ocean_vx','ocean speed x-component [m/s]');
			fielddisplay(self,'ocean_vy','ocean speed y-component [m/s]');
		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'enum',BasalforcingsEnum(),'data',SeaiceoceanEnum(),'format','Integer');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','rho_ocean','format','Double');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_coef','format','Double');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_lin_drag_coef','format','Double');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_quad_drag_coef','format','Double');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_turning_angle','format','Double');
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_ssh','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_vx','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','basalforcings','fieldname','ocean_vy','format','DoubleMat','mattype',1);
		end % }}}
	end
end
