%LINEAR BASAL FORCINGS class definition
%
%   Usage:
%      linearbasalforcings=linearbasalforcings();

classdef linearbasalforcings
	properties (SetAccess=public) 
		groundedice_melting_rate  = NaN;
		deepwater_melting_rate    = NaN;
		deepwater_elevation       = NaN;
		upperwater_elevation      = NaN;
		geothermalflux            = NaN;
	end
	methods
     function createxml(obj,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- basalforcings -->');
			 fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="melting_rate" type="',            class(obj.melting_rate),'" default="',              num2str(obj.melting_rate),'">',              '     <section name="basalforcings" />','     <help> basal melting rate (positive if melting) [m/yr] </help>','</parameter>');
             fprintf(fid,'%s%s%s%s%s\n%s\n%s\n',        '<parameter key ="geothermalflux" type="',          class(obj.geothermalflux),'" default="',            num2str(obj.geothermalflux),'">',            '     <section name="basalforcings" />','     <help> geothermal heat flux [W/m^2] </help>','</parameter>');
             
        end % }}}
		function obj = linearbasalforcings(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					obj=structtoobj(linearbasalforcings(),varargin{1});
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = initialize(self,md) % {{{

			if isnan(self.groundedice_melting_rate),
				self.groundedice_melting_rate=zeros(md.mesh.numberofvertices,1);
				disp('      no basalforcings.groundedice_melting_rate specified: values set as zero');
			end

		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%default values for melting parameterization
			obj.deepwater_melting_rate = 50;
			obj.deepwater_elevation    = -800;
			obj.upperwater_elevation   = -400;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			if ismember(MasstransportAnalysisEnum(),analyses) & ~(solution==TransientSolutionEnum() & md.transient.ismasstransport==0),
				md = checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'timeseries',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_melting_rate','>=',0,'numel',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_elevation','<','basalforcings.upperwater_elevation','numel',1);
				md = checkfield(md,'fieldname','basalforcings.upperwater_elevation','<',0,'numel',1);
			end
			if ismember(BalancethicknessAnalysisEnum(),analyses),
				md = checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','basalforcings.deepwater_melting_rate','>=',0,'numel',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_elevation','<','basalforcings.upperwater_elevation','numel',1);
				md = checkfield(md,'fieldname','basalforcings.upperwater_elevation','<',0,'numel',1);
			end
			if ismember(ThermalAnalysisEnum(),analyses) & ~(solution==TransientSolutionEnum() & md.transient.isthermal==0),
				md = checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'timeseries',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_melting_rate','>=',0,'numel',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_elevation','<','basalforcings.upperwater_elevation','numel',1);
				md = checkfield(md,'fieldname','basalforcings.upperwater_elevation','<',0,'numel',1);
				md = checkfield(md,'fieldname','basalforcings.geothermalflux','NaN',1,'timeseries',1,'>=',0);
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   basal forcings parameters:'));

			fielddisplay(obj,'groundedice_melting_rate','basal melting rate (positive if melting) [m/yr]');
			fielddisplay(obj,'deepwater_melting_rate','basal melting rate (positive if melting applied for floating ice whith base < deepwater_elevation) [m/yr]');
			fielddisplay(obj,'deepwater_elevation','elevation of ocean deepwater [m]');
			fielddisplay(obj,'upperwater_elevation','elevation of ocean upperwater [m]');
			fielddisplay(obj,'geothermalflux','geothermal heat flux [W/m^2]');

		end % }}}
		function marshall(obj,md,fid) % {{{

			yts=365.0*24.0*3600.0;

			floatingice_melting_rate=zeros(md.mesh.numberofvertices,1);
			floatingice_melting_rate(find(md.geometry.base<=md.basalforcings.deepwater_elevation))=md.basalforcings.deepwater_melting_rate;
			pos=find(md.geometry.base>md.basalforcings.deepwater_elevation & md.geometry.base<md.basalforcings.upperwater_elevation);
			floatingice_melting_rate(pos)=md.basalforcings.deepwater_melting_rate*(md.geometry.base(pos)-md.basalforcings.upperwater_elevation)/(md.basalforcings.deepwater_elevation-md.basalforcings.upperwater_elevation);
			WriteData(fid,'enum',BasalforcingsEnum(),'data',LinearFloatingMeltRateEnum(),'format','Integer');
			WriteData(fid,'data',floatingice_melting_rate,'format','DoubleMat','enum',BasalforcingsFloatingiceMeltingRateEnum(),'mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1)
			WriteData(fid,'object',obj,'fieldname','groundedice_melting_rate','format','DoubleMat','enum',BasalforcingsGroundediceMeltingRateEnum(),'mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1)
			WriteData(fid,'object',obj,'fieldname','geothermalflux','enum',BasalforcingsGeothermalfluxEnum(),'format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'fieldname','deepwater_melting_rate','format','Double','enum',BasalforcingsDeepwaterMeltingRateEnum(),'scale',1./yts)
			WriteData(fid,'object',obj,'fieldname','deepwater_elevation','format','Double','enum',BasalforcingsDeepwaterElevationEnum())
			WriteData(fid,'object',obj,'fieldname','upperwater_elevation','format','Double','enum',BasalforcingsUpperwaterElevationEnum())
		end % }}}
	end
end
