/*!\file:  control.cpp
 * \brief: control solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "control"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;
	char* control_type=NULL;

	/*Fem models : */
	Model* model=NULL;

	/*Results: */
	DataSet* results=NULL;
	Result*  result=NULL;

	ParameterInputs* inputs=NULL;
	int waitonlock=0;

	/*Intermediary: */
	double* u_g_initial=NULL;
	double* u_g_obs=NULL;
	Param*  param=NULL;
	int      count;
	DataSet* parameters=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Initialize model structure: */
	model=new Model();

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading diagnostic horiz model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	_printf_("\n   reading diagnostic vert model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),VertAnalysisEnum());
	
	_printf_("\n   reading diagnostic stokes model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
	_printf_("\n   reading diagnostic hutter model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	
	_printf_("\n   reading surface and bed slope computation model data:\n");
	model->AddFormulation(fid,SlopeComputeAnalysisEnum());
	
	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	_printf_("initialize inputs:\n");
	model->FindParam(&u_g_initial,"u_g",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	model->FindParam(&u_g_obs,"u_g_obs",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	model->FindParam(&numberofnodes,"numberofnodes");

	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	inputs->Add("velocity_obs",u_g_obs,2,numberofnodes);
	
	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	//Add output file name to parameters of femmodels[0]
	parameters=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum())->get_parameters();
	count=parameters->Size()+1;
	param= new Param(count,"outputfilename",STRING);
	param->SetString(outputfilename);
	parameters->AddObject(param);

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run control analysis: */
		_printf_("call computational core:\n");
		control_core(results,model,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on control core:\n");

#ifdef _HAVE_DAKOTA_ 
		Qmux(model,inputs,DiagnosticAnalysisEnum(),NoneAnalysisEnum());
#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
#endif
	}

	/*Add analysis_type and control_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","control");
	results->AddObject(result);
	
	model->FindParam(&control_type,"control_type");
	result=new Result(results->Size()+1,0,1,"control_type",control_type);
	results->AddObject(result);

	_printf_("process results:\n");
	ProcessResults(&results,model,ControlAnalysisEnum());

	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	if (waitonlock){
		WriteLockFile(lockname);
	}

	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	/*Free ressources */
	xfree((void**)&u_g_initial);
	xfree((void**)&u_g_obs);

	return 0; //unix success return;
}
