/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution 
 */ 

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <float.h>
#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void transient_core(FemModel* femmodel){

	/*parameters: */
	IssmDouble starttime,finaltime,dt,yts;
	bool       isstressbalance,ismasstransport,isFS,isthermal,isgroundingline,isgia,islevelset,isdamageevolution,ishydrology,iscalving;
	bool       save_results,dakota_analysis;
	bool       time_adapt=false;
	int        output_frequency;
	int        domaintype,groundingline_migration,calvinglaw;
	int        numoutputs         = 0;
	Analysis  *analysis = NULL;
	char**     requested_outputs = NULL;


	/*intermediary: */
	int    step;
	IssmDouble time;

	//first recover parameters common to all solutions
	femmodel->parameters->FindParam(&domaintype,DomainTypeEnum);
	femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
	femmodel->parameters->FindParam(&finaltime,TimesteppingFinalTimeEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);
	femmodel->parameters->FindParam(&dakota_analysis,QmuIsdakotaEnum);
	femmodel->parameters->FindParam(&output_frequency,SettingsOutputFrequencyEnum);
	femmodel->parameters->FindParam(&time_adapt,TimesteppingTimeAdaptEnum);
	femmodel->parameters->FindParam(&isstressbalance,TransientIsstressbalanceEnum);
	femmodel->parameters->FindParam(&ismasstransport,TransientIsmasstransportEnum);
	femmodel->parameters->FindParam(&isthermal,TransientIsthermalEnum);
	femmodel->parameters->FindParam(&isgia,TransientIsgiaEnum);
	femmodel->parameters->FindParam(&isgroundingline,TransientIsgroundinglineEnum);
	femmodel->parameters->FindParam(&islevelset,TransientIslevelsetEnum);
	femmodel->parameters->FindParam(&isdamageevolution,TransientIsdamageevolutionEnum);
	femmodel->parameters->FindParam(&ishydrology,TransientIshydrologyEnum);
	femmodel->parameters->FindParam(&isFS,FlowequationIsFSEnum);
	femmodel->parameters->FindParam(&iscalving,TransientIscalvingEnum);
	femmodel->parameters->FindParam(&calvinglaw,CalvingLawEnum);
	if(isgroundingline) femmodel->parameters->FindParam(&groundingline_migration,GroundinglineMigrationEnum);
	femmodel->parameters->FindParam(&numoutputs,TransientNumRequestedOutputsEnum);
	if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,TransientRequestedOutputsEnum);

	/*initialize: */
	step=0;
	time=starttime;

	while(time < finaltime - (yts*DBL_EPSILON)){ //make sure we run up to finaltime.
		/*Increment*/
		if(time_adapt){
			femmodel->TimeAdaptx(&dt);
			if(time+dt>finaltime) dt=finaltime-time;
			femmodel->parameters->SetParam(dt,TimesteppingTimeStepEnum);
		}
		step+=1;
		time+=dt;
		femmodel->parameters->SetParam(time,TimeEnum);
		femmodel->parameters->SetParam(step,StepEnum);

		if(VerboseSolution()) _printf0_("iteration " << step << "/" << floor((finaltime-time)/dt)+step << "  time [yr]: " << time/yts << " (time step: " << dt/yts << ")\n");
		if(step%output_frequency==0 || (time >= finaltime - (yts*DBL_EPSILON)) || step==1)
		 save_results=true;
		else
		 save_results=false;
		femmodel->parameters->SetParam(save_results,SaveResultsEnum);

		if(isthermal && domaintype==Domain3DEnum){
			if(VerboseSolution()) _printf0_("   computing thermal regime\n");
			thermal_core(femmodel);
		}

		if(ishydrology){
			if(VerboseSolution()) _printf0_("   computing water heads\n");
			hydrology_core(femmodel);
		}

		if(isstressbalance){
			if(VerboseSolution()) _printf0_("   computing new velocity\n");
			stressbalance_core(femmodel);
		}

		if(isdamageevolution){
			if(VerboseSolution()) _printf0_("   computing damage\n");
			damage_core(femmodel);
		}

		if(islevelset){
			if(iscalving && calvinglaw==CalvingLevermannEnum){
				if(VerboseSolution()) _printf0_("   computing calving rate\n");
				femmodel->StrainRateparallelx();
				femmodel->StrainRateperpendicularx();
				femmodel->CalvingRateLevermannx();
			}
			if(VerboseSolution()) _printf0_("   computing movement of ice boundaries\n");
			/* smoothen slope of lsf for computation of normal on ice domain*/
			levelsetfunctionslope_core(femmodel);

			/* extrapolate velocities onto domain with no ice */
			Analysis* extanalysis = new ExtrapolationAnalysis();
			const int nvars=2;
			int vars[nvars] = {VxEnum, VyEnum};
			for(int iv=0;iv<nvars;iv++){
				femmodel->parameters->SetParam(vars[iv],ExtrapolationVariableEnum); 
				extanalysis->Core(femmodel);
			}
			delete extanalysis;	

			/* solve level set equation */
			analysis = new LevelsetAnalysis();
			analysis->Core(femmodel);
			delete analysis;

			/* update vertices included for next calculation */
			GetMaskOfIceVerticesLSMx(femmodel);

			/* add computation domain mask to outputs */
			int outputs[1] = {IceMaskNodeActivationEnum};
			femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],1);
		}

		if(ismasstransport){
			if(VerboseSolution()) _printf0_("   computing new thickness\n");
			masstransport_core(femmodel);
			if(VerboseSolution()) _printf0_("   updating vertices positions\n");
			femmodel->UpdateVertexPositionsx();
		}
		
		if(isgroundingline){
			if(VerboseSolution()) _printf0_("   computing new grounding line position\n");
			GroundinglineMigrationx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			femmodel->parameters->SetParam(MaskGroundediceLevelsetEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
			femmodel->parameters->SetParam(BaseEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
			femmodel->parameters->SetParam(SurfaceEnum,InputToExtrudeEnum);
			extrudefrombase_core(femmodel);
				
			if(save_results){
				int outputs[3] = {SurfaceEnum,BaseEnum,MaskGroundediceLevelsetEnum};
				femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],3);
			}
		}

		/*Calculate new Basal melting on Floating ice*/
		FloatingiceMeltingRatex(femmodel);
		
		if(isgia){
			if(VerboseSolution()) _printf0_("   computing glacial isostatic adjustment\n");
			#ifdef _HAVE_GIA_
			gia_core(femmodel);
			#else
			_error_("ISSM was not compiled with gia capabilities. Exiting");
			#endif
		}

		/*unload results*/
		if(VerboseSolution()) _printf0_("   computing requested outputs\n");
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs,save_results);
		if(isgroundingline && (groundingline_migration==SubelementMigrationEnum || groundingline_migration==SubelementMigration2Enum)){
			int outputs[1] = {MaskGroundediceLevelsetEnum};
			femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],1,save_results);
		}

		if(save_results){
			if(VerboseSolution()) _printf0_("   saving temporary results\n");
			OutputResultsx(femmodel);
		}
	}

	femmodel->RequestedDependentsx();

	/*Free ressources:*/	
	if(numoutputs){for(int i=0;i<numoutputs;i++){xDelete<char>(requested_outputs[i]);} xDelete<char*>(requested_outputs);}
}
