%PFE class definition
%
%   Usage:
%      cluster=pfe();
%      cluster=pfe('np',3);
%      cluster=pfe('np',3,'login','username');

classdef pfe
    properties (SetAccess=public)  
		 % {{{
		 name           = 'pfe'
		 login          = '';
		 numnodes       = 20;
		 cpuspernode    = 8;
		 port           = 1025;
		 queue          = 'long';
		 time           = 12*60;
		 processor      = 'wes';
		 codepath       = '';
		 executionpath  = '';
		 grouplist     = 's1010';
		 interactive    = 0;
		 bbftp          = 0;
		 numstreams     = 8;
		 hyperthreading = 0;
	 end
	 properties (SetAccess=private) 
		 np=20*8;
		 % }}}
	 end
	 methods
		 function cluster=pfe(varargin) % {{{

			 %initialize cluster using default settings if provided
			 if (exist('pfe_settings')==2), pfe_settings; end

			 %use provided options to change fields
			 cluster=AssignObjectFields(pairoptions(varargin{:}),cluster);
		 end
		 %}}}
		 function disp(cluster) % {{{
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    numnodes: %i',cluster.numnodes));
			 disp(sprintf('    cpuspernode: %i',cluster.cpuspernode));
			 disp(sprintf('    np: %i',cluster.cpuspernode*cluster.numnodes));
			 disp(sprintf('    queue: %s',cluster.queue));
			 disp(sprintf('    time: %i',cluster.time));
			 disp(sprintf('    processor: %s',cluster.processor));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    grouplist: %s',cluster.grouplist));
			 disp(sprintf('    interactive: %i',cluster.interactive));
			 disp(sprintf('    hyperthreading: %i',cluster.hyperthreading));
		 end
		 %}}}
		 function md = checkconsistency(cluster,md,solution,analyses) % {{{

			 available_queues={'long','normal','debug','devel','alphatst@pbspl233'};
			 queue_requirements_time=[5*24*60 8*60 2*60 2*60 24*60];
			 queue_requirements_np=[2048 2048 150 150 2048];

			 QueueRequirements(available_queues,queue_requirements_time,queue_requirements_np,cluster.queue,cluster.np,cluster.time)

			 %now, check cluster.cpuspernode according to processor type
			 if (strcmpi(cluster.processor,'har') | strcmpi(cluster.processor,'neh')),
				 if cluster.hyperthreading,
					 if ((cluster.cpuspernode>16 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 16 for ''neh'' and ''har'' processors in hyperthreading mode');
					 end
				 else
					 if ((cluster.cpuspernode>8 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 8 for ''neh'' and ''har'' processors');
					 end
				 end
			 elseif strcmpi(cluster.processor,'wes'),
				 if cluster.hyperthreading,
					 if ((cluster.cpuspernode>24 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 24 for ''wes'' processors in hyperthreading mode');
					 end
				 else
					 if ((cluster.cpuspernode>12 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 12 for ''wes'' processors');
					 end
				 end
			 elseif strcmpi(cluster.processor,'ivy'),
				 if cluster.hyperthreading,
					 if ((cluster.cpuspernode>40 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 40 for ''ivy'' processors in hyperthreading mode');
					 end
				 else
					 if ((cluster.cpuspernode>20 ) | (cluster.cpuspernode<1)),
						 md = checkmessage(md,'cpuspernode should be between 1 and 20 for ''ivy'' processors');
					 end
				 end

			 else
				 md = checkmessage(md,'unknown processor type, should be ''neh'',''wes'' or ''har'' or ''ivy''');
			 end

			 %Miscelaneous
			 if isempty(cluster.login), md = checkmessage(md,'login empty'); end
			 if isempty(cluster.codepath), md = checkmessage(md,'codepath empty'); end
			 if isempty(cluster.executionpath), md = checkmessage(md,'executionpath empty'); end
			 if isempty(cluster.grouplist), md = checkmessage(md,'grouplist empty'); end

		 end
		 %}}}
		 function BuildQueueScript(cluster,dirname,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			 if(isgprof),    disp('gprof not supported by cluster, ignoring...'); end

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %write queuing script 
			 fid=fopen([modelname '.queue'],'w');
			 fprintf(fid,'#PBS -S /bin/bash\n');
%			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -l select=%i:ncpus=%i:model=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 fprintf(fid,'#PBS -W group_list=%s\n',cluster.grouplist);
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',[cluster.executionpath '/' dirname '/' modelname]);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',[cluster.executionpath '/' dirname '/' modelname]);
			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			 fprintf(fid,'module load comp-intel/11.1.046\n');
			 fprintf(fid,'module load math/intel_mkl_64_10.0.011\n');
			 fprintf(fid,'module load mpi-sgi/mpt.2.06rp16\n');
			 fprintf(fid,'export PATH="$PATH:."\n\n');
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			 fprintf(fid,'export ISSM_DIR="%s/../"\n',cluster.codepath); %FIXME
			 fprintf(fid,'source $ISSM_DIR/etc/environment.sh\n');       %FIXME
			 fprintf(fid,'cd %s/%s/\n\n',cluster.executionpath,dirname);
			 if ~isvalgrind,
				 fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
			 else
				 fprintf(fid,'mpiexec -np %i valgrind --leak-check=full %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
			 end
			 if ~io_gather, %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 if ~isvalgrind,
					 fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/Interactive' num2str(cluster.interactive)],modelname);
				 else
					 fprintf(fid,'mpiexec -np %i valgrind --leak-check=full %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/Interactive' num2str(cluster.interactive)],modelname);
				 end
				 if ~io_gather, %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function BuildKrigingQueueScript(cluster,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			 if(isgprof),    disp('gprof not supported by cluster, ignoring...'); end

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %write queuing script 
			 fid=fopen([modelname '.queue'],'w');
			 fprintf(fid,'#PBS -S /bin/bash\n');
			 %			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -l select=%i:ncpus=%i:model=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 fprintf(fid,'#PBS -W group_list=%s\n',cluster.grouplist);
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',modelname);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',modelname);
			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			 fprintf(fid,'module load comp-intel/2013.1.117\n');
			 fprintf(fid,'module load mpi-sgi/mpt.2.06rp16\n');
			 fprintf(fid,'export PATH="$PATH:."\n');
			 fprintf(fid,'export ISSM_DIR="%s/../"\n',cluster.codepath); %FIXME
			 fprintf(fid,'source $ISSM_DIR/etc/environment.sh\n');       %FIXME
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			 fprintf(fid,'cd %s/%s/\n\n',cluster.executionpath,modelname);
			 fprintf(fid,'mpiexec -np %i %s/kriging.exe %s %s\n',cluster.np,cluster.codepath,[cluster.executionpath '/' modelname],modelname); %FIXME
			 if ~io_gather, %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 if ~isvalgrind,
					 fprintf(fid,'mpiexec -np %i %s/kriging.exe %s %s\n',cluster.np,cluster.codepath,[cluster.executionpath '/' modelname],modelname);
				 else
					 fprintf(fid,'mpiexec -np %i valgrind --leak-check=full %s/kriging.exe %s %s\n',cluster.np,cluster.codepath,[cluster.executionpath '/' modelname],modelname);
				 end
				 if ~io_gather, %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function BuildOceanQueueScript(cluster,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %write queuing script 
			 fid=fopen([modelname '.queue'],'w');
			 fprintf(fid,'#PBS -S /bin/bash\n');
			 fprintf(fid,'#PBS -l select=%i:ncpus=%i:model=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 fprintf(fid,'#PBS -W group_list=%s\n',cluster.grouplist);
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',modelname);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',modelname);
			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			 fprintf(fid,'module load comp-intel/2015.0.090\n');
			 fprintf(fid,'module load test/mpt.2.11r8\n');
			 fprintf(fid,'module load netcdf/4.0\n');
			 fprintf(fid,'module load mpi-mvapich2/1.4.1/gcc\n');
			 fprintf(fid,'module load gcc/4.4.4\n');
			 fprintf(fid,'export PATH="$PATH:."\n');
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			 fprintf(fid,'cd %s/%s/\n\n',cluster.executionpath,modelname);
			 fprintf(fid,'mpiexec -np %i ./mitgcm\n',cluster.np); 
		%	 if ~io_gather, %concatenate the output files:
		%		 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
		%	 end
			 fclose(fid);

		 end %}}}
		 function LaunchQueueJob(cluster,modelname,dirname,filelist)% {{{

			 %compress the files into one zip.
			 compressstring=['tar -zcf ' dirname '.tar.gz '];
			 for i=1:numel(filelist),
				 compressstring = [compressstring ' ' filelist{i}];
			 end
			 if cluster.interactive,
				 compressstring = [compressstring ' ' modelname '.run '  modelname '.errlog ' modelname '.outlog '];
			 end
			 system(compressstring);

			 disp('uploading input file and queueing script');
			 if cluster.interactive,
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
			 else 
				 directory=cluster.executionpath;
			 end

			 if ~cluster.bbftp,
				 issmscpout(cluster.name,directory,cluster.login,cluster.port,{[dirname '.tar.gz']});
			 else
				 issmbbftpout(cluster.name,directory,cluster.login,cluster.port,cluster.numstreams,{[dirname '.tar.gz']});
			 end

			 %lauch command, to be executed via ssh
			 if ~cluster.interactive, 
				 launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' dirname ' && mkdir ' dirname ...
					 ' && cd ' dirname ' && mv ../' dirname '.tar.gz ./ && tar -zxf ' dirname '.tar.gz  && qsub ' modelname '.queue '];
			 else
				 launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive) ' && tar -zxf ' dirname '.tar.gz'];
			 end

			 disp('launching solution sequence on remote cluster');
			 issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
		 end
		 %}}}
		 function Download(cluster,dirname,filelist)% {{{

			 %copy files from cluster to current directory
			 if ~cluster.interactive,
				 directory=[cluster.executionpath '/' dirname '/'];
			 else
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive) '/'];
			 end

			 if ~cluster.bbftp,
				 issmscpin(cluster.name,cluster.login,cluster.port,directory,filelist);
			 else
				 issmbbftpin(cluster.name, cluster.login, cluster.port, cluster.numstreams, directory, filelist);
			 end

		 end %}}}
	end
end
