%MASSTRANSPORT class definition
%
%   Usage:
%      masstransport=masstransport();

classdef masstransport
	properties (SetAccess=public) 
		 spcthickness           = NaN;
		 iscalvingrate          = 0;
		 calvingrate            = NaN;
		 levermann_calving_coeff= 0;
		 isfreesurface          = 0;
		 min_thickness          = 0;
		 hydrostatic_adjustment = 0;
		 stabilization          = 0;
		 vertex_pairing         = NaN;
		 penalty_factor         = 0;
		 requested_outputs      = {};
	end
	methods (Static)
		function obj = loadobj(obj) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. If the input is a struct it is an old version of this class and
			% old fields must be recovered (make sure they are in the deprecated
			% model properties)

			if verLessThan('matlab','7.9'),
				disp('Warning: your matlab version is old and there is a risk that load does not work correctly');
				disp('         if the model is not loaded correctly, rename temporarily loadobj so that matlab does not use it');

				% This is a Matlab bug: all the fields of md have their default value
				% Example of error message:
				% Warning: Error loading an object of class 'model':
				% Undefined function or method 'exist' for input arguments of type 'cell'
				%
				% This has been fixed in MATLAB 7.9 (R2009b) and later versions
			end

			if isstruct(obj)
				disp('Recovering masstransport from older version');
				objstruct = obj;
				obj = structtoobj(masstransport(),objstruct);
			end
		end% }}}
	end
	methods
         function createxml(obj,fid) % {{{
            fprintf(fid, '<!-- masstransport -->\n');            
                    
            % Masstransport solution parameters
            fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="Masstransport solution parameters">','<section name="masstransport" />');                    
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="spcthickness" type="',class(obj.spcthickness),'" default="',convert2str(obj.spcthickness),'">','     <section name="masstransport" />','     <help> thickness constraints (NaN means no constraint) [m] </help>','  </parameter>');
				fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="iscalvingrate" type="',class(obj.iscalvingrate),'" default="',convert2str(obj.iscalvingrate),'">','     <section name="masstransport" />','     <help> do we use calving (loss of ice) </help>','  </parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="calvingrate" type="',class(obj.calvingrate),'" default="',convert2str(obj.calvingrate),'">','     <section name="masstransport" />','     <help> calving rate at given location [m/a] </help>','  </parameter>');
				fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="levermann_calving_coeff" type="',class(obj.levermann_calving_coeff),'" default="',convert2str(obj.levermann_calving_coeff),'">','     <section name="masstransport" />','     <help> Proportionality coefficient in Levermann model  </help>','  </parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isfreesurface" type="',class(obj.isfreesurface),'" default="',convert2str(obj.isfreesurface),'">','     <section name="masstransport" />','     <help> do we use free surfaces (FS only) are mass conservation </help>','  </parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="min_thickness" type="',class(obj.min_thickness),'" default="',convert2str(obj.min_thickness),'">','     <section name="masstransport" />','     <help> minimum ice thickness allowed [m] </help>','  </parameter>');
            
            % hydrostatic_adjustment drop-down (incremental or absolute )
            fprintf(fid,'%s\n%s\n%s\n%s\n','  <parameter key ="hydrostatic_adjustment" type="alternative" optional ="false">','     <section name="masstransport" />','     <help> adjustment of ice shelves surface and bed elevations: ''Incremental'' or ''Absolute''  </help>');
            fprintf(fid,'%s\n','       <option value="Incremental" type="string" default="true"> </option>');
            fprintf(fid,'%s\n%s\n','       <option value="Absolute" type="string" default="false"> </option>','</parameter>');
            
            %stabilization drop-down (0, 1, 2, or 3)
            fprintf(fid,'%s\n%s\n%s\n%s\n','  <parameter key ="stabilization" type="alternative" optional="false">','     <section name="masstransport" />','     <help> 0: no, 1:  artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin </help>');
            fprintf(fid,'%s\n','       <option value="0" type="string" default="true"> </option>');
            fprintf(fid,'%s\n','       <option value="1" type="string" default="false"> </option>');
            fprintf(fid,'%s\n','       <option value="2" type="string" default="false"> </option>');
            fprintf(fid,'%s\n%s\n','       <option value="3" type="string" default="false"> </option>','</parameter>');

            fprintf(fid,'%s\n%s\n','</frame>');    
            
            %Penalty options
            fprintf(fid,'%s\n%s\n%s\n','<frame key="2" label="Penalty options">','<section name="masstransport" />'); 
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertex_pairing" type="',class(obj.vertex_pairing),'" default="',convert2str(obj.vertex_pairing),'">','     <section name="masstransport" />','     <help> offset used by penalties: penalty = Kmax*10^offset </help>','  </parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="penalty_factor" type="',class(obj.penalty_factor),'" default="',convert2str(obj.penalty_factor),'">','     <section name="masstransport" />','     <help> pairs of vertices that are penalized </help>','  </parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="requested_outputs" type="',class(obj.requested_outputs),'" default="',convert2str(obj.requested_outputs),'">','     <section name="masstransport" />','     <help> additional outputs requested </help>','  </parameter>');
            fprintf(fid,'%s\n%s\n','</frame>');
        
        end % }}}
		function obj = masstransport(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('masstransport');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function list = defaultoutputs(self,md) % {{{

			list = {'Thickness','Surface','Base'};

		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%Proportionality coefficient in Levermann model
			obj.levermann_calving_coeff=8e21;
			
			%Type of stabilization to use 0:nothing 1:artificial_diffusivity 3:Discontinuous Galerkin
			obj.stabilization=1;

			%Factor applied to compute the penalties kappa=max(stiffness matrix)*10^penalty_factor
			obj.penalty_factor=3;

			%Minimum ice thickness that can be used
			obj.min_thickness=1;

			%Hydrostatic adjustment
			obj.hydrostatic_adjustment='Absolute';

			%default output
			obj.requested_outputs={'default'};
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(MasstransportAnalysisEnum(),analyses) |  (solution==TransientSolutionEnum() & md.transient.ismasstransport==0), return; end

			md = checkfield(md,'fieldname','masstransport.spcthickness','forcing',1);
			if(ismember(LevelsetAnalysisEnum(), analyses) & md.transient.islevelset)
				md = checkfield(md,'fieldname','masstransport.iscalvingrate','values',[0 1]);
				md = checkfield(md,'fieldname','masstransport.calvingrate','NaN',1,'size',[md.mesh.numberofvertices 1],'>=',0);
				md = checkfield(md,'fieldname','masstransport.levermann_calving_coeff','>',0);
			end

			md = checkfield(md,'fieldname','masstransport.isfreesurface','values',[0 1]);
			md = checkfield(md,'fieldname','masstransport.hydrostatic_adjustment','values',{'Absolute' 'Incremental'});
			md = checkfield(md,'fieldname','masstransport.stabilization','values',[0 1 2 3 4]);
			md = checkfield(md,'fieldname','masstransport.min_thickness','>',0);
			md = checkfield(md,'fieldname','masstransport.requested_outputs','stringrow',1);

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Masstransport solution parameters:'));
			fielddisplay(obj,'spcthickness','thickness constraints (NaN means no constraint) [m]');
			fielddisplay(obj,'iscalvingrate','do we use calving (loss of ice)');
			fielddisplay(obj,'calvingrate','calving rate at given location [m/a]');
			fielddisplay(obj,'levermann_calving_coeff','proportionality coefficient in Levermann model');
			fielddisplay(obj,'isfreesurface','do we use free surfaces (FS only) are mass conservation');
			fielddisplay(obj,'min_thickness','minimum ice thickness allowed [m]');
			fielddisplay(obj,'hydrostatic_adjustment','adjustment of ice shelves surface and bed elevations: ''Incremental'' or ''Absolute'' ');
			fielddisplay(obj,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin, 4: Flux Correction Transport');

			disp(sprintf('\n      %s','Penalty options:'));
			fielddisplay(obj,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
			fielddisplay(obj,'vertex_pairing','pairs of vertices that are penalized');
			fielddisplay(obj,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(obj,md,fid) % {{{

			yts=365.*24.*3600.;

			WriteData(fid,'object',obj,'fieldname','spcthickness','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'fieldname','calvingrate','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1,'scale',1./yts);
			WriteData(fid,'object',obj,'fieldname','iscalvingrate','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isfreesurface','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','levermann_calving_coeff','format','Double');
			WriteData(fid,'object',obj,'fieldname','min_thickness','format','Double');
			WriteData(fid,'data',StringToEnum(obj.hydrostatic_adjustment),'format','Integer','enum',MasstransportHydrostaticAdjustmentEnum());
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',obj,'fieldname','vertex_pairing','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','penalty_factor','format','Double');
			
			%process requested outputs
			outputs = obj.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(obj,md)]; %add defaults
			end
			WriteData(fid,'data',outputs,'enum',MasstransportRequestedOutputsEnum,'format','StringArray');
		end % }}}
	end
end
