function results=steadystate_core(models,inputs);
%STEADYSTATE_CORE - compute the core temperature and velocity field  at thermal steady state.
%
%   Usage:
%      results=steadystate_core(models,inputs);
%

%recover models
m_dh=models.dh;
m_dv=models.dv;
m_ds=models.ds;
m_dhu=models.dhu;
m_sl=models.sl;
m_t=models.t;
m_m=models.m;

%recover parameters common to all solutions
debug=m_dhu.parameters.debug;
dim=m_dhu.parameters.dim;
eps_rel=m_dhu.parameters.eps_rel;
ishutter=m_dhu.parameters.ishutter;
ismacayealpattyn=m_dh.parameters.ismacayealpattyn;
isstokes=m_ds.parameters.isstokes;

%convergence
converged=0;

step=1;

%initialize: 
t_g_old=0;
u_g_old=0;

if isstokes, ndof=4; else ndof=3; end

while(~converged),
	
	displaystring(debug,'%s%i','computing temperature and velocity for step ',step);
	
	%first compute temperature at steady state.
	if (step>1),
		inputs=add(inputs,'velocity',results_diagnostic.u_g,'doublevec',ndof,m_t.parameters.numberofnodes);
	end
	results_thermal=thermal_core(models,inputs);
	
	%add temperature to inputs.
	%Compute depth averaged temperature
	temperature_average=FieldDepthAverage(m_t.elements,m_t.nodes,m_t.loads,m_t.materials,results_thermal.t_g,'temperature');
	inputs=add(inputs,'temperature_average',temperature_average,'doublevec',1,m_t.parameters.numberofnodes);
	inputs=add(inputs,'temperature',results_thermal.t_g,'doublevec',1,m_t.parameters.numberofnodes);
	
	%now compute diagnostic velocity using the steady state temperature.
	results_diagnostic=diagnostic_core(models,inputs);
	
	%convergence? 
	du_g=results_diagnostic.u_g-u_g_old;
	ndu=norm(du_g,2); 
	nu=norm(u_g_old,2);
	
	dt_g=results_thermal.t_g-t_g_old;
	ndt=norm(dt_g,2); 
	nt=norm(t_g_old,2); 

	u_g_old=results_diagnostic.u_g;
	t_g_old=results_thermal.t_g;
	
	displaystring(debug,'%-60s%g\n                                     %s%g\n                                     %s%g%s',...
	              '      relative convergence criterion: velocity -> norm(du)/norm(u)=   ',ndu/nu*100,' temperature -> norm(dt)/norm(t)=',ndt/nt*100,' eps_rel:                        ',eps_rel*100,' %');
	if (ndu/nu<=eps_rel)  & (ndt/nt<=eps_rel),
		converged=1;
	else
		if step>1,
			converged=1;
		end
	end

	step=step+1;
	if converged,
		break;
	end
end

%save results from thermal and diagnostic
results.u_g=results_diagnostic.u_g;
results.p_g=results_diagnostic.p_g;
results.t_g=results_thermal.t_g;
results.m_g=results_thermal.m_g;
results.step=step;
results.time=0;
