#include "./SeaiceAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  SeaiceAnalysis::DofsPerNode(int** doflist,int domaintype,int approximation){/*{{{*/
	return 2;
}/*}}}*/
void SeaiceAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/
	parameters->AddObject(iomodel->CopyConstantObject(SeaiceMinConcentrationEnum));
	parameters->AddObject(iomodel->CopyConstantObject(SeaiceMinThicknessEnum));
	parameters->AddObject(iomodel->CopyConstantObject(SeaiceMaxThicknessEnum));
}/*}}}*/
void SeaiceAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	/*Make sure we are in 2D horizontal*/
	if(iomodel->domaintype!=Domain2DhorizontalEnum) _error_("Only 2D model are supported");

	/*Update the elements so that they are aware of this analysis*/
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	/*Push to the elements all the inputs that we need*/
	iomodel->FetchDataToInput(elements,SeaiceThicknessEnum);
	iomodel->FetchDataToInput(elements,SeaiceConcentrationEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanSshEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanVxEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanVyEnum);
	iomodel->FetchDataToInput(elements,SurfaceforcingsWindVxEnum);
	iomodel->FetchDataToInput(elements,SurfaceforcingsWindVyEnum);
	iomodel->FetchDataToInput(elements,DamageEnum);
	iomodel->FetchDataToInput(elements,VxStarEnum);
	iomodel->FetchDataToInput(elements,VyStarEnum);
	iomodel->FetchDataToInput(elements,VxEnum);
	iomodel->FetchDataToInput(elements,VyEnum);
	iomodel->FetchDataToInput(elements,SeaiceCoriolisFactorEnum);
	iomodel->FetchDataToInput(elements,StressTensorPredictorxxEnum);
	iomodel->FetchDataToInput(elements,StressTensorPredictoryyEnum);
	iomodel->FetchDataToInput(elements,StressTensorPredictorxyEnum);
	iomodel->FetchDataToInput(elements,MeshXEnum);
	iomodel->FetchDataToInput(elements,MeshYEnum);
}/*}}}*/
void SeaiceAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	::CreateNodes(nodes,iomodel,SeaiceAnalysisEnum,P1Enum);

}/*}}}*/
void SeaiceAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	const int finiteelement = P1Enum;
	IoModelToConstraintsx(constraints,iomodel,SeaiceSpcvxEnum,SeaiceAnalysisEnum,finiteelement,0);
	IoModelToConstraintsx(constraints,iomodel,SeaiceSpcvyEnum,SeaiceAnalysisEnum,finiteelement,1);
}/*}}}*/
void SeaiceAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/
	return;
}/*}}}*/

/*Finite Element Analysis*/
void           SeaiceAnalysis::Core(FemModel* femmodel){/*{{{*/
	_error_("not implemented");
}/*}}}*/
ElementVector* SeaiceAnalysis::CreateDVector(Element* element){/*{{{*/
	/*Default, return NULL*/
	return NULL;
}/*}}}*/
ElementMatrix* SeaiceAnalysis::CreateJacobianMatrix(Element* element){/*{{{*/
_error_("Not implemented");
}/*}}}*/
ElementMatrix* SeaiceAnalysis::CreateKMatrix(Element* element){/*{{{*/

	/* Check if there is ice in this element */
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	IssmDouble c_min; element->FindParam(&c_min,SeaiceMinConcentrationEnum);
	if(concentration_input->Max()<=c_min) return NULL;

	/*Intermediaries */
	IssmDouble  Jdet,D_scalar,dt,thickness,concentration;
	IssmDouble  ocean_coef,rho_ocean,ocean_lin_drag_coef,ocean_quad_drag_coef;
	IssmDouble  vx,vy,oceanvx,oceanvy,vnorm,ocean_turning_angle;
	IssmDouble  C[3][3];
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = numnodes*2;

	/*Initialize Element vector and other vectors*/
	ElementMatrix* Ke      = element->NewElementMatrix();
	IssmDouble*    B       = xNew<IssmDouble>(3*numdof);
	IssmDouble*    M       = xNew<IssmDouble>(numdof);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	element->FindParam(&ocean_coef,BasalforcingsOceanCoefEnum);
	element->FindParam(&ocean_lin_drag_coef,BasalforcingsOceanLinDragCoefEnum);
	element->FindParam(&ocean_quad_drag_coef,BasalforcingsOceanQuadDragCoefEnum);
	element->FindParam(&ocean_turning_angle,BasalforcingsOceanTurningAngleEnum);
	IssmDouble rho_i                  = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble time_relaxation_stress = element->GetMaterialParameter(MaterialsTimeRelaxationStressEnum);
	Input*     thickness_input = element->GetInput(SeaiceThicknessEnum);       _assert_(thickness_input);
	Input*     vx_input        = element->GetInput(VxEnum);                    _assert_(vx_input);
	Input*     vy_input        = element->GetInput(VyEnum);                    _assert_(vy_input);
	Input*     oceanvx_input   = element->GetInput(BasalforcingsOceanVxEnum);  _assert_(oceanvx_input);
	Input*     oceanvy_input   = element->GetInput(BasalforcingsOceanVyEnum);  _assert_(oceanvy_input);

	/*Get minimum inertia to avoid 0 in the line, and time_ratio*/
	IssmDouble minimum_inertia = rho_i*0.01;
	IssmDouble time_ratio=(1-dt/time_relaxation_stress);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Get Jacobian Determinant and thickness*/
		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		thickness_input->GetInputValue(&thickness,gauss);
		concentration_input->GetInputValue(&concentration,gauss);

		/*Create first part of the stiffness matrix*/
		this->GetM(M,element,xyz_list,gauss);
		D_scalar=(rho_i*thickness+minimum_inertia)/dt*gauss->weight*Jdet;
		TripleMultiply(M,1,numdof,1,
					&D_scalar,1,1,0,
					M,1,numdof,0,
					&Ke->values[0],1);

		/*Create Elastic part*/
		this->GetB(B,element,xyz_list,gauss);
		this->CreateCTensor(&C[0][0],element,gauss);
		for(int i=0;i<3;i++){
			for(int j=0;j<3;j++){
				C[i][j] = dt*thickness*time_ratio*gauss->weight*Jdet*C[i][j];
			}
		}
		TripleMultiply(B,3,numdof,1,
					&C[0][0],3,3,0,
					B,3,numdof,0,
					&Ke->values[0],1);

		/*Create Ocean forcing part*/
		oceanvx_input->GetInputValue(&oceanvx,gauss);
		oceanvy_input->GetInputValue(&oceanvy,gauss);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		vnorm = sqrt(pow(oceanvx-vx,2) + pow(oceanvy-vy,2));
		D_scalar = concentration*ocean_coef*rho_ocean*(ocean_lin_drag_coef+ocean_quad_drag_coef*vnorm)*cos(ocean_turning_angle)*gauss->weight*Jdet;
		TripleMultiply(M,1,numdof,1,
					&D_scalar,1,1,0,
					M,1,numdof,0,
					&Ke->values[0],1);
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(M);
	delete gauss;
	return Ke;
}
/*}}}*/
ElementVector* SeaiceAnalysis::CreatePVector(Element* element){/*{{{*/

	/* Check if there is ice in this element */
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	IssmDouble c_min; element->FindParam(&c_min,SeaiceMinConcentrationEnum);
	if(concentration_input->Max()<=c_min) return NULL;

	/*Intermediaries */
	IssmDouble  air_coef,ocean_coef,constant_part,time_relaxation_stress;
	IssmDouble  rho_ice,rho_air,rho_ocean,gravity,omega;
	IssmDouble  vx,vy,vxstar,vystar,windvx,windvy,oceanvx,oceanvy,vnorm;
	IssmDouble  air_lin_drag_coef,air_quad_drag_coef;
	IssmDouble  ocean_lin_drag_coef,ocean_quad_drag_coef;
	IssmDouble  concentration,thickness,coriolis_factor,dt,Jdet,D_scalar;
	IssmDouble  sigma_xx,sigma_yy,sigma_xy,sigma_vec[3];
	IssmDouble  ocean_turning_angle,dssh[2];
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = 2*numnodes;

	/*Initialize Element vector and vectors*/
	ElementVector* pe      = element->NewElementVector();
	IssmDouble*    B       = xNew<IssmDouble>(3*numdof);
	IssmDouble*    basis   = xNew<IssmDouble>(numnodes);
	IssmDouble     D[3][3] = {0.};

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	element->FindParam(&rho_air,SurfaceforcingsRhoAirEnum);
	element->FindParam(&air_coef,SurfaceforcingsAirCoefEnum);
	element->FindParam(&air_lin_drag_coef,SurfaceforcingsAirLinDragCoefEnum);
	element->FindParam(&air_quad_drag_coef,SurfaceforcingsAirQuadDragCoefEnum);
	element->FindParam(&rho_ocean,BasalforcingsRhoOceanEnum);
	element->FindParam(&ocean_coef,BasalforcingsOceanCoefEnum);
	element->FindParam(&ocean_lin_drag_coef,BasalforcingsOceanLinDragCoefEnum);
	element->FindParam(&ocean_quad_drag_coef,BasalforcingsOceanQuadDragCoefEnum);
	element->FindParam(&ocean_turning_angle,BasalforcingsOceanTurningAngleEnum);
	rho_ice                = element->GetMaterialParameter(MaterialsRhoIceEnum);
	gravity                = element->GetMaterialParameter(ConstantsGEnum);
	omega                  = element->GetMaterialParameter(ConstantsOmegaEnum);
	time_relaxation_stress = element->GetMaterialParameter(MaterialsTimeRelaxationStressEnum);
	Input* thickness_input     = element->GetInput(SeaiceThicknessEnum);       _assert_(thickness_input);
	Input* coriolis_fact_input = element->GetInput(SeaiceCoriolisFactorEnum);  _assert_(coriolis_fact_input);
	Input* vx_input            = element->GetInput(VxEnum);                    _assert_(vx_input);
	Input* vy_input            = element->GetInput(VyEnum);                    _assert_(vy_input);
	Input* vxstar_input        = element->GetInput(VxStarEnum);                _assert_(vxstar_input);
	Input* vystar_input        = element->GetInput(VyStarEnum);                _assert_(vystar_input);
	Input* windvx_input        = element->GetInput(SurfaceforcingsWindVxEnum); _assert_(windvx_input);
	Input* windvy_input        = element->GetInput(SurfaceforcingsWindVyEnum); _assert_(windvy_input);
	Input* oceanvx_input       = element->GetInput(BasalforcingsOceanVxEnum);  _assert_(oceanvx_input);
	Input* oceanvy_input       = element->GetInput(BasalforcingsOceanVyEnum);  _assert_(oceanvy_input);
	Input* oceanssh_input      = element->GetInput(BasalforcingsOceanSshEnum); _assert_(oceanssh_input);
	Input* sigma_xx_input      = element->GetInput(StressTensorxxEnum);        _assert_(sigma_xx_input);
	Input* sigma_yy_input      = element->GetInput(StressTensoryyEnum);        _assert_(sigma_yy_input);
	Input* sigma_xy_input      = element->GetInput(StressTensorxyEnum);        _assert_(sigma_xy_input);

	/*Calculate time_ratio*/
	IssmDouble time_ratio=(1-dt/time_relaxation_stress);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		element->NodalFunctions(basis, gauss);
		this->GetB(B,element,xyz_list,gauss);

		/*Get all inputs on gauss point*/
		thickness_input->GetInputValue(&thickness,gauss);
		concentration_input->GetInputValue(&concentration,gauss);
		coriolis_fact_input->GetInputValue(&coriolis_factor,gauss);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		vxstar_input->GetInputValue(&vxstar,gauss);
		vystar_input->GetInputValue(&vystar,gauss);
		windvx_input->GetInputValue(&windvx,gauss);
		windvy_input->GetInputValue(&windvy,gauss);
		oceanvx_input->GetInputValue(&oceanvx,gauss);
		oceanvy_input->GetInputValue(&oceanvy,gauss);
		sigma_xx_input->GetInputValue(&sigma_xx,gauss);
		sigma_yy_input->GetInputValue(&sigma_yy,gauss);
		sigma_xy_input->GetInputValue(&sigma_xy,gauss);
		oceanssh_input->GetInputDerivativeValue(&dssh[0],xyz_list,gauss);

		/*Previous speed (finite differencing)*/
		for(int i=0;i<numnodes;i++){
			pe->values[i*2+0]+=Jdet*gauss->weight*rho_ice*thickness*vx*basis[i];
			pe->values[i*2+1]+=Jdet*gauss->weight*rho_ice*thickness*vy*basis[i];
		}

		/*Atmospheric forcing*/
		vnorm    = sqrt(windvx*windvx + windvy*windvy);
		constant_part = concentration*air_coef*rho_air*(air_lin_drag_coef+air_quad_drag_coef*vnorm);
		for(int i=0;i<numnodes;i++){
			pe->values[i*2+0] += constant_part*windvx*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += constant_part*windvy*Jdet*gauss->weight*basis[i];
		}

		/*Ocean forcing*/
		vnorm = sqrt(pow(oceanvx-vx,2) + pow(oceanvy-vy,2));
		constant_part = concentration*ocean_coef*rho_ocean*(ocean_lin_drag_coef+ocean_quad_drag_coef*vnorm);
		for(int i=0;i<numnodes;i++){
			pe->values[i*2+0] += constant_part*omega*(oceanvy-vy)*sin(ocean_turning_angle)*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += constant_part*omega*(vx-oceanvx)*sin(ocean_turning_angle)*Jdet*gauss->weight*basis[i];

			pe->values[i*2+0] += constant_part*oceanvx*cos(ocean_turning_angle)*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += constant_part*oceanvy*cos(ocean_turning_angle)*Jdet*gauss->weight*basis[i];

			pe->values[i*2+0] += (-rho_ice*thickness*gravity*omega*dssh[0])*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += (-rho_ice*thickness*gravity*omega*dssh[1])*Jdet*gauss->weight*basis[i];
		}

		/*Coriolis forces (use ustar)*/
		for(int i=0;i<numnodes;i++){
			pe->values[i*2+0] += (-rho_ice*thickness*coriolis_factor*vystar)*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += (+rho_ice*thickness*coriolis_factor*vxstar)*Jdet*gauss->weight*basis[i];
		}
		
		/*Add elastic part of previous time step*/
		sigma_vec[0] = sigma_xx;
		sigma_vec[1] = sigma_yy;
		sigma_vec[2] = sigma_xy;
		D[0][0] = D[1][1] = D[2][2] = time_ratio*thickness*Jdet*gauss->weight;
		TripleMultiply(B,3,numdof,1,
					&D[0][0],3,3,0,
					&sigma_vec[0],3,1,0,
					&pe->values[0],1);
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(B);
	delete gauss;
	return pe;
}/*}}}*/
void SeaiceAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	   _error_("not implemented yet");
}/*}}}*/
void SeaiceAnalysis::GradientJ(Vector<IssmDouble>* gradient,Element* element,int control_type,int control_index){/*{{{*/
	_error_("Not implemented yet");
}/*}}}*/
void SeaiceAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	/*Intermediaries*/
	int* doflist=NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = numnodes*2;

	/*Fetch dof list and allocate solution vectors*/
	element->GetDofList(&doflist,SSAApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numdof);
	IssmDouble* vx        = xNew<IssmDouble>(numnodes);
	IssmDouble* vy        = xNew<IssmDouble>(numnodes);
	IssmDouble* vel       = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(int i=0;i<numnodes;i++){
		vx[i]=values[i*2+0];
		vy[i]=values[i*2+1];
		if(xIsNan<IssmDouble>(vx[i])) _error_("NaN found in solution vector");
		if(xIsNan<IssmDouble>(vy[i])) _error_("NaN found in solution vector");
	}

	/*Compute vel*/
	for(int i=0;i<numnodes;i++) vel[i]=sqrt(vx[i]*vx[i] + vy[i]*vy[i]);

	/*Add vx and vy as inputs to the tria element: */
	element->AddInput(VxEnum,vx,element->GetElementType());
	element->AddInput(VyEnum,vy,element->GetElementType());
	element->AddInput(VelEnum,vel,element->GetElementType());

	/*Free ressources:*/
	xDelete<IssmDouble>(vel);
	xDelete<IssmDouble>(vy);
	xDelete<IssmDouble>(vx);
	xDelete<IssmDouble>(values);
	xDelete<int>(doflist);
}/*}}}*/
void SeaiceAnalysis::UpdateConstraints(FemModel* femmodel){/*{{{*/

	/*Intermediaries*/
	IssmDouble          c_min;
	Vector<IssmDouble>* mask        = NULL;
	IssmDouble*         serial_mask = NULL;

	/*Get minimum concentration allowed*/
	femmodel->parameters->FindParam(&c_min,SeaiceMinConcentrationEnum);

	/*Step 1: update mask of active nodes*/
	mask=new Vector<IssmDouble>(femmodel->nodes->NumberOfNodes(SeaiceAnalysisEnum));
	for (int i=0;i<femmodel->elements->Size();i++){
		Element* element=dynamic_cast<Element*>(femmodel->elements->GetObjectByOffset(i));

		/*Get current concentration of element and decide whether it is an active element*/
		int    numnodes            = element->GetNumberOfNodes();
		Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
		if(concentration_input->Max()>c_min){
			for(int in=0;in<numnodes;in++) mask->SetValue(element->nodes[in]->Sid(),1.,INS_VAL);
		}
	}

	/*Assemble and serialize*/
	mask->Assemble();
	serial_mask=mask->ToMPISerial();
	delete mask;

	/*Update node activation accordingly*/
	int counter =0;
	for(int i=0;i<femmodel->nodes->Size();i++){
		Node* node=dynamic_cast<Node*>(femmodel->nodes->GetObjectByOffset(i));
		if(node->InAnalysis(SeaiceAnalysisEnum)){
			if(serial_mask[node->Sid()]==1.){
				node->Activate();
				counter++;
			}
			else{
				node->Deactivate();
			}
		}
	}
	xDelete<IssmDouble>(serial_mask);

	/*Display number of active nodes*/
	int sum_counter;
	ISSM_MPI_Reduce(&counter,&sum_counter,1,ISSM_MPI_INT,ISSM_MPI_SUM,0,IssmComm::GetComm() );
	if(VerboseSolution()) _printf0_("   Number of active nodes: "<< sum_counter <<"\n");
}/*}}}*/

/*Intermediary functions*/
void SeaiceAnalysis::CreateCTensor(IssmDouble* C,Element* element,Gauss* gauss){/*{{{*/

	/*Intermediaries*/
	IssmDouble D[3][3];
	IssmDouble damage,concentration;

	/*Get Material parameters*/
	IssmDouble poisson          = element->GetMaterialParameter(MaterialsPoissonEnum);
	IssmDouble young_modulus    = element->GetMaterialParameter(MaterialsYoungModulusEnum);
	IssmDouble ridging_exponent = element->GetMaterialParameter(MaterialsRidgingExponentEnum);

	/*Get damage input at this location*/
	Input* damage_input        = element->GetInput(DamageEnum);              _assert_(damage_input);
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	damage_input->GetInputValue(&damage,gauss);
	concentration_input->GetInputValue(&concentration,gauss);

	/*Build D tensor once for all*/
	D[0][0] = 1.;      D[0][1] = poisson; D[0][2] = 0.;
	D[1][0] = poisson; D[1][1] = 1.;      D[1][2] = 0.;
	D[2][0] = 0.;      D[2][1] = 0;       D[2][2] = (1.-poisson)/2.;
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			D[i][j] = D[i][j]/(1-poisson*poisson);
		}
	}

	/*Get f(A), E*/
	IssmDouble f = exp(ridging_exponent*(1.-concentration));
	IssmDouble E = young_modulus*(1.-damage)*f;

	/*Let's build C = hE*D */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			C[i*3+j] = E*D[i][j];
		}
	}
}/*}}}*/
void SeaiceAnalysis::GetB(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For node i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dN/dx           0    ] 
	 *          [   0           dN/dy  ]      
	 *          [ 1/2*dN/dy  1/2*dN/dx ]     
	 * where N is the basis function for node i.
	 *
	 * We assume B has been allocated already           */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* dbasis=xNew<IssmDouble>(2*numnodes);
	element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[2*numnodes*0+2*i+0] = dbasis[0*numnodes+i];
		B[2*numnodes*0+2*i+1] = 0.;
		B[2*numnodes*1+2*i+0] = 0.;
		B[2*numnodes*1+2*i+1] = dbasis[1*numnodes+i];
		B[2*numnodes*2+2*i+0] = .5*dbasis[1*numnodes+i];
		B[2*numnodes*2+2*i+1] = .5*dbasis[0*numnodes+i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(dbasis);
}/*}}}*/
void SeaiceAnalysis::GetM(IssmDouble* M,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute M  matrix. M=[M1 M2 M3] where Mi is of size 3*NDOF2. 
	 * For node i, Mi can be expressed in the actual coordinate system
	 * by: 
	 *       Mi=[ N  N ] 
	 * where N is the basis function for node i.
	 *
	 * We assume M has been allocated already           */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* basis=xNew<IssmDouble>(numnodes);
	element->NodalFunctions(basis,gauss);

	/*Build M: */
	for(int i=0;i<numnodes;i++){
		M[2*i+0] = basis[i];
		M[2*i+1] = basis[i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(basis);
}/*}}}*/
void SeaiceAnalysis::UpdateDamageAndStress(FemModel* femmodel){/*{{{*/
	/* The damage variable is updated as a function of the actual elastic deformation
	 * In both cases, a Coulombic enveloppe is used, define by the cohesion C, tan(phi) and tract_coef.
	 * In both cases, a maximal compressive strength is fixed at compr_max
	 * The enveloppe is defined in N/m^2.
	 * The coeficients of the internal stress are then multiplied by the ice thickness to be used in the vertical integrated momentiom equation.
	 */

	/* Mohr-Coulomb criterion calculation
	 * %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	 *                                                                   
	 *                            sigma_s                                
	 *        Mohr.Coulomb branch     |                                  
	 *                     \          |                                  
	 *                      * |       |                                  
	 *                        *       |   cohesion (C=cfix+calea)        
	 *                        | *     |  /                               
	 *                        |   *   | /    tract                       
	 *                        |     * |/    /                            
	 *             -comp_max  |       *    /                             
	 *                      \ |       | * /                              
	 *                       \|      0| | *                              
	 *             -------------------------*------------ sigma_n        
	 *                        |       | | *                              
	 *                        |       | *                                
	 *                        |       *                                  
	 *                        |     * |                                  
	 *                        |   *   |                                  
	 *                        | *     |                                  
	 *                        *       |                                  
	 *                      * |       |                                  
	 *                    *           |                                  
	 *                                |                                  
	 *                                |                                  
	 *                                                                   
	 * %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	 */

	/*Intermediaties*/
	IssmDouble sigma_xx,sigma_yy,sigma_xy,sigma_s,sigma_n,sigma_target;
	IssmDouble compression_coef,traction_coef,cohesion,tan_phi;
	IssmDouble traction,compression_max;
	IssmDouble damage_test,damage_new,damage,tmp;

	/*Loop over the elements of this partition and update accordingly*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=dynamic_cast<Element*>(femmodel->elements->GetObjectByOffset(i));

		/*Get Mohr-Coulomb parameters*/
		compression_coef = element->GetMaterialParameter(MaterialsCompressionCoefEnum);
		traction_coef    = element->GetMaterialParameter(MaterialsTractionCoefEnum);
		cohesion         = element->GetMaterialParameter(MaterialsCohesionEnum); //C
		tan_phi          = element->GetMaterialParameter(MaterialsInternalFrictionCoefEnum); //mu = tan(phi)

		/*Get current stress state*/
		Input*   sigma_xx_input = element->GetInput(StressTensorPredictorxxEnum); _assert_(sigma_xx_input);
		Input*   sigma_yy_input = element->GetInput(StressTensorPredictoryyEnum); _assert_(sigma_yy_input);
		Input*   sigma_xy_input = element->GetInput(StressTensorPredictorxyEnum); _assert_(sigma_xy_input);
		sigma_xx_input->GetInputAverage(&sigma_xx);
		sigma_yy_input->GetInputAverage(&sigma_yy);
		sigma_xy_input->GetInputAverage(&sigma_xy);

		/* Compute the invariants of the elastic deformation and instantaneous deformation rate */
		sigma_s=sqrt(pow((sigma_xx-sigma_yy)/2.,2)+pow(sigma_xy,2));
		sigma_n=(sigma_xx+sigma_yy)/2.;

		/* same maximal tensile strength */
		traction=traction_coef*cohesion/tan_phi;
		compression_max=compression_coef*cohesion;

		/* estimate the internal constraints using the current elastic deformation */
		Input* damage_input    = element->GetInput(DamageEnum); _assert_(damage_input);
		damage_input->GetInputAverage(&damage);
		damage_test = damage;
		damage_new  = damage;
		if(sigma_n>traction || sigma_n<-compression_max){
			if(sigma_n>traction){
				sigma_target=traction;
			}
			else{
				sigma_target=-compression_max;
			}

			tmp=1.-sigma_target/sigma_n*(1.-damage);
			if(tmp<1. && tmp>damage_new){
				damage_test=((damage_test>tmp)?(damage_test):(tmp)); /* max(damage_test,tmp); */
			}
		}
		if(sigma_s>cohesion-sigma_n*tan_phi){
			tmp=1.0-cohesion/(sigma_s+sigma_n*tan_phi)*(1-damage);
			if(tmp<1. && tmp>damage_new){
				damage_test=((damage_test>tmp)?(damage_test):(tmp)); /*max(damage_test,tmp); */
			}
		}

		/* The damage variable is changed */
		damage_new=damage_test;
		element->AddInput(DamageEnum,&damage_new,P0Enum);

		/* Recompute the internal stress*/
		if(damage<1.){
			sigma_xx = (1.-damage_new)/(1.-damage)*sigma_xx;
			sigma_yy = (1.-damage_new)/(1.-damage)*sigma_yy;
			sigma_xy = (1.-damage_new)/(1.-damage)*sigma_xy;
		}
		else{
			sigma_xx = 0.;
			sigma_yy = 0.;
			sigma_xy = 0.;
		}
		element->AddInput(StressTensorxxEnum,&sigma_xx,P0Enum);
		element->AddInput(StressTensoryyEnum,&sigma_yy,P0Enum);
		element->AddInput(StressTensorxyEnum,&sigma_xy,P0Enum);
	}
}/*}}}*/
void SeaiceAnalysis::PostProcess(FemModel* femmodel){/*{{{*/

	/*Intermediaties*/
	IssmDouble *xyz_list = NULL;
	IssmDouble *xyz_list_old = NULL;
	IssmDouble *xyz_list_new = NULL;
	IssmDouble  dt,meshx,meshy,vx,vy,eps_xx,eps_yy,eps_zz,area_old,area_new;
	IssmDouble  time_relaxation_stress,time_relaxation_damage,damage,concentration,thickness;
	IssmDouble  sigma_dot_xx,sigma_dot_yy,sigma_dot_xy;
	IssmDouble  sigma_pred_xx,sigma_pred_yy,sigma_pred_xy;
	IssmDouble  sigma_xx,sigma_yy,sigma_xy;
	IssmDouble  min_c,min_h,max_h;
	IssmDouble  epsilon[3]; /*eps_xx,eps_yy,eps_xy*/
	IssmDouble  C[3][3];

	/*Fetch the parameters once for all*/
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&min_c,SeaiceMinConcentrationEnum);
	femmodel->parameters->FindParam(&min_h,SeaiceMinThicknessEnum);
	femmodel->parameters->FindParam(&max_h,SeaiceMaxThicknessEnum);

	/*Loop over the elements of this partition and update accordingly*/
	for(int i=0;i<femmodel->elements->Size();i++){
		Element* element=dynamic_cast<Element*>(femmodel->elements->GetObjectByOffset(i));

		/*Get some inputs needed for the update*/
		element->GetVerticesCoordinates(&xyz_list);
		time_relaxation_stress = element->GetMaterialParameter(MaterialsTimeRelaxationStressEnum);
		time_relaxation_damage = element->GetMaterialParameter(MaterialsTimeRelaxationDamageEnum);
		Input* meshx_input = element->GetInput(MeshXEnum); _assert_(meshx_input);
		Input* meshy_input = element->GetInput(MeshYEnum); _assert_(meshy_input);
		Input* vx_input    = element->GetInput(VxEnum);    _assert_(vx_input);
		Input* vy_input    = element->GetInput(VyEnum);    _assert_(vy_input);

		/*Preallocate future inputs*/
		int numvertices=element->GetNumberOfVertices();
		IssmDouble* meshx_new_list = xNew<IssmDouble>(numvertices);
		IssmDouble* meshy_new_list = xNew<IssmDouble>(numvertices);
		xyz_list_old = xNew<IssmDouble>(numvertices*3);
		xyz_list_new = xNew<IssmDouble>(numvertices*3);

		/*1. update vertex positions (using a Gauss object for convenience)*/
		Gauss* gauss=element->NewGauss();
		for (int iv=0;iv<numvertices;iv++){
			gauss->GaussVertex(iv);

			meshx_input->GetInputValue(&meshx,gauss);
			meshy_input->GetInputValue(&meshy,gauss);
			vx_input->GetInputValue(&vx,gauss);
			vy_input->GetInputValue(&vy,gauss);

			meshx_new_list[iv] = meshx + vx*dt;
			meshy_new_list[iv] = meshy + vy*dt;

			xyz_list_old[iv*3+0] = meshx;
			xyz_list_old[iv*3+1] = meshy;
			xyz_list_old[iv*3+2] = 0.;

			xyz_list_new[iv*3+0] = meshx_new_list[iv];
			xyz_list_new[iv*3+1] = meshy_new_list[iv];
			xyz_list_new[iv*3+2] = 0.;
		}
		element->AddInput(MeshXEnum,meshx_new_list,P1Enum);
		element->AddInput(MeshYEnum,meshy_new_list,P1Enum);

		/*Now we are going to use a point in the center of the element, because we are P0 for the stress*/
		gauss->GaussNode(P0Enum,0);

		/*Calculate sigma_dot*/
		sigma_dot_xx=sigma_dot_yy=sigma_dot_xy=0.;
		this->CreateCTensor(&C[0][0],element,gauss);
		element->StrainRateSSA(&epsilon[0],xyz_list,gauss,vx_input,vy_input);
		for(int j=0;j<3;j++){
			sigma_dot_xx += C[0][j]*epsilon[0];
			sigma_dot_yy += C[1][j]*epsilon[1];
			sigma_dot_xy += C[2][j]*epsilon[2];
		}

		/*Get predicted stress state*/
		Input* sigma_xx_input = element->GetInput(StressTensorPredictorxxEnum); _assert_(sigma_xx_input);
		Input* sigma_yy_input = element->GetInput(StressTensorPredictoryyEnum); _assert_(sigma_yy_input);
		Input* sigma_xy_input = element->GetInput(StressTensorPredictorxyEnum); _assert_(sigma_xy_input);
		sigma_xx_input->GetInputAverage(&sigma_xx);
		sigma_yy_input->GetInputAverage(&sigma_yy);
		sigma_xy_input->GetInputAverage(&sigma_xy);

		/*Calculate new stress and push to element*/
		sigma_xx = sigma_xx + dt*(sigma_dot_xx - sigma_xx/time_relaxation_stress);
		sigma_yy = sigma_yy + dt*(sigma_dot_yy - sigma_yy/time_relaxation_stress);
		sigma_xy = sigma_xy + dt*(sigma_dot_xy - sigma_xy/time_relaxation_stress);
		element->AddInput(StressTensorxxEnum,&sigma_xx,P0Enum);
		element->AddInput(StressTensoryyEnum,&sigma_yy,P0Enum);
		element->AddInput(StressTensorxyEnum,&sigma_xy,P0Enum);

		/*Update Damage According*/
		Input* damage_input = element->GetInput(DamageEnum); _assert_(damage_input);
		damage_input->GetInputAverage(&damage);
		damage = damage*(1.-dt/time_relaxation_damage);
		element->AddInput(DamageEnum,&damage,P0Enum);

		/*Prepare new predictor*/
		this->CreateCTensor(&C[0][0],element,gauss); /*C is updated now that there is a new damage in inputs*/
		for(int j=0;j<3;j++){
			sigma_pred_xx += C[0][j]*epsilon[0];
			sigma_pred_yy += C[1][j]*epsilon[1];
			sigma_pred_xy += C[2][j]*epsilon[2];
		}
		element->AddInput(StressTensorPredictorxxEnum,&sigma_pred_xx,P0Enum);
		element->AddInput(StressTensorPredictoryyEnum,&sigma_pred_yy,P0Enum);
		element->AddInput(StressTensorPredictorxyEnum,&sigma_pred_xy,P0Enum);

		/*Calculate Old and new area (FIXME: for now we assume trianlges...)*/
		element->JacobianDeterminant(&area_old,xyz_list_old,gauss);
		element->JacobianDeterminant(&area_new,xyz_list_new,gauss);
		area_old = area_old/SQRT3;
		area_new = area_new/SQRT3;

		/*Update ice thickness and concentration using element distortion*/
		Input* thickness_input     = element->GetInput(SeaiceThicknessEnum);     _assert_(thickness_input);
		Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
		thickness_input->GetInputAverage(&thickness);
		concentration_input->GetInputAverage(&concentration);
		IssmDouble ice_area   = concentration*area_old;
		IssmDouble ice_volume = thickness*area_old;
		if(concentration>min_c){
			concentration = ice_area  /area_new;
			thickness     = ice_volume/area_new;

			/* lower bounds */
			concentration = ((concentration>min_c)?(concentration):(min_c));
			thickness     = ((thickness    >min_h)?(thickness    ):(min_h));

			/* upper bounds (only for the concentration) */
			concentration    = ((concentration<1.   )?(concentration):(1));
			thickness        = ((thickness    <max_h)?(thickness    ):(max_h));
		}
		element->AddInput(SeaiceConcentrationEnum,&concentration,P0Enum);
		element->AddInput(SeaiceThicknessEnum,&thickness,P0Enum);

		/*Clean up*/
		xDelete<IssmDouble>(meshx_new_list);
		xDelete<IssmDouble>(meshy_new_list);
		xDelete<IssmDouble>(xyz_list);
		xDelete<IssmDouble>(xyz_list_old);
		xDelete<IssmDouble>(xyz_list_new);
	}

}/*}}}*/
