%SEAICEATM Class definition
%
%   Usage:
%      seaiceatm=seaiceatm();

classdef seaiceatm
	properties (SetAccess=public) 
		rho_air            = 0.;
		air_coef           = 0.;
		air_lin_drag_coef  = 0.;
		air_quad_drag_coef = 0.;
		wind_vx            = NaN;
		wind_vy            = NaN;
	end
	methods
		function obj = seaiceatm(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%air density (kg/m^3)
			obj.rho_air=1.3;

			%By default, we apply the full stress
			obj.air_coef = 1.;

			%drag coefficients (different from Hibler 0.0012)
			obj.air_lin_drag_coef  = 0.;
			obj.air_quad_drag_coef = 0.003;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			md = checkfield(md,'fieldname','surfaceforcings.rho_air','NaN',1,'>',0.,'numel',1);
			md = checkfield(md,'fieldname','surfaceforcings.air_coef','NaN',1,'>=',0.,'<=',1,'numel',1);
			md = checkfield(md,'fieldname','surfaceforcings.air_lin_drag_coef','NaN',1,'>=',0.,'numel',1);
			md = checkfield(md,'fieldname','surfaceforcings.air_quad_drag_coef','NaN',1,'>=',0.,'numel',1);
			md = checkfield(md,'fieldname','surfaceforcings.wind_vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','surfaceforcings.wind_vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Atmospheric forcings for sea ice:'));
			disp(sprintf('\n     tau_a = air_coef*rho_air*(air_lin_drag_coef+air_quad_drag_coef*|v_wind|)*v_wind\n'));
			fielddisplay(obj,'rho_air','air density [kg/m^3]');
			fielddisplay(obj,'air_coef','atmospheric stress spin up coefficient (between 0 and 1)');
			fielddisplay(obj,'air_lin_drag_coef','air linear drag coefficient (no unit)');
			fielddisplay(obj,'air_quad_drag_coef','air quadratic drag coefficient [s/m]');
			fielddisplay(obj,'wind_vx','wind speed x-component at 10 m elevation [m/s]');
			fielddisplay(obj,'wind_vy','wind speed y-component at 10 m elevation [m/s]');
		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'enum',SurfaceforcingsEnum(),'data',SeaiceatmEnum(),'format','Integer');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','rho_air','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','air_coef','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','air_lin_drag_coef','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','air_quad_drag_coef','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','wind_vx','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','wind_vy','format','DoubleMat','mattype',1);
		end % }}}
	end
end
