%MATSEAICE class definition
%
%   Usage:
%      matseaice=matseaice();

classdef matseaice
	properties (SetAccess=public) 
		poisson                = 0.;
		young_modulus          = 0.;
		damage                 = NaN;
		ridging_exponent       = 0.;
		cohesion               = 0.;
		internal_friction_coef = 0.;
		compression_coef       = 0.;
		traction_coef          = 0.;
	end
	methods
		function obj = matseaice(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('matseaice');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%poisson modulus
			obj.poisson=0.3;

			%Young modulus
			obj.young_modulus=9.e+9;

			%ridging exponent
			obj.ridging_exponent=-20.;

			%Cohesion (Weiss 2007)
			obj.cohesion = 40.e+3; %40 kPa

			%internal_friction_coef 
			obj.internal_friction_coef=0.7;

			%compression_coef (Bouillon and Rampal 2014)
			obj.compression_coef=5./2;

			%traction_coef (Bouillon and Rampal 2014)
			obj.traction_coef=5./6.;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			md = checkfield(md,'fieldname','materials.poisson','NaN',1,'>',0,'<',.5,'numel',1);
			md = checkfield(md,'fieldname','materials.young_modulus','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.damage','NaN',1,'>=',0,'<',1,'size',[md.mesh.numberofelements 1]);
			md = checkfield(md,'fieldname','materials.ridging_exponent','NaN',1,'<',0,'numel',1);
			md = checkfield(md,'fieldname','materials.cohesion','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.internal_friction_coef','NaN',1,'>',0,'numel',1);
			md = checkfield(md,'fieldname','materials.compression_coef','NaN',1,'>',1,'numel',1);
			md = checkfield(md,'fieldname','materials.traction_coef','NaN',1,'>',0,'<',1,'numel',1);
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Sea Ice Material:'));

			fielddisplay(obj,'poisson','poisson ratio for undamaged ice [no unit]');
			fielddisplay(obj,'young_modulus','Young modulus for undamaged ice [Pa]');
			fielddisplay(obj,'damage','damage, between 0 (no damage) and 1 (fully damaged)');
			fielddisplay(obj,'ridging_exponent','Riging exponent (c, Hibler parameter) [no unit]');
			fielddisplay(obj,'cohesion','cohesion (C) [Pa]');
			fielddisplay(obj,'internal_friction_coef','Mohr-Coulomb internal friction coefficient (mu=tan(phi)) [no unit]');
			fielddisplay(obj,'compression_coef','Ratio between cutoff compressive strength and the cohesion [no unit]');
			fielddisplay(obj,'traction_coef','Ratio between cutoff tensile strength and Mohr-Coulomb tensile strength [no unit]');
		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'enum',MaterialsEnum(),'data',MatseaiceEnum(),'format','Integer');
			WriteData(fid,'object',obj,'class','materials','fieldname','poisson','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','young_modulus','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','damage','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'class','materials','fieldname','ridging_exponent','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','cohesion','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','internal_friction_coef','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','compression_coef','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','traction_coef','format','Double');
		end % }}}
	end
end
