#include "./SeaiceAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  SeaiceAnalysis::DofsPerNode(int** doflist,int domaintype,int approximation){/*{{{*/
	return 2;
}/*}}}*/
void SeaiceAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/
}/*}}}*/
void SeaiceAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	/*Make sure we are in 2D horizontal*/
	if(iomodel->domaintype!=Domain2DhorizontalEnum) _error_("Only 2D model are supported");

	/*Update the elements so that they are aware of this analysis*/
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	/*Push to the elements all the inputs that we need*/
	iomodel->FetchDataToInput(elements,SeaiceThicknessEnum);
	iomodel->FetchDataToInput(elements,SeaiceConcentrationEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanSshEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanVxEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsOceanVyEnum);
	iomodel->FetchDataToInput(elements,SurfaceforcingsWindVxEnum);
	iomodel->FetchDataToInput(elements,SurfaceforcingsWindVyEnum);
	iomodel->FetchDataToInput(elements,MaterialsDamageEnum);
	iomodel->FetchDataToInput(elements,VxStarEnum);
	iomodel->FetchDataToInput(elements,VyStarEnum);
	iomodel->FetchDataToInput(elements,VxEnum);
	iomodel->FetchDataToInput(elements,VyEnum);
}/*}}}*/
void SeaiceAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	::CreateNodes(nodes,iomodel,SeaiceAnalysisEnum,P1Enum);

}/*}}}*/
void SeaiceAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	const int finiteelement = P1Enum;
	IoModelToConstraintsx(constraints,iomodel,SeaiceSpcvxEnum,SeaiceAnalysisEnum,finiteelement,0);
	IoModelToConstraintsx(constraints,iomodel,SeaiceSpcvyEnum,SeaiceAnalysisEnum,finiteelement,1);
}/*}}}*/
void SeaiceAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/
	return;
}/*}}}*/

/*Finite Element Analysis*/
void           SeaiceAnalysis::Core(FemModel* femmodel){/*{{{*/
	_error_("not implemented");
}/*}}}*/
ElementVector* SeaiceAnalysis::CreateDVector(Element* element){/*{{{*/
	/*Default, return NULL*/
	return NULL;
}/*}}}*/
ElementMatrix* SeaiceAnalysis::CreateJacobianMatrix(Element* element){/*{{{*/
_error_("Not implemented");
}/*}}}*/
ElementMatrix* SeaiceAnalysis::CreateKMatrix(Element* element){/*{{{*/

	/* Check if there is ice in this element */
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	if(concentration_input->Max()==0.) return NULL;

	/*Intermediaries */
	IssmDouble  Jdet,D_scalar,dt;
	IssmDouble  C[3][3];
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = numnodes*2;

	/*Initialize Element vector and other vectors*/
	ElementMatrix* Ke      = element->NewElementMatrix();
	IssmDouble*    B       = xNew<IssmDouble>(3*numdof);
	IssmDouble*    M       = xNew<IssmDouble>(numdof);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		element->JacobianDeterminant(&Jdet,xyz_list,gauss);

		/*Create first part of the stiffness matrix*/
		this->GetM(M,element,xyz_list,gauss);
		D_scalar=gauss->weight*Jdet;
		TripleMultiply(M,1,numdof,1,
					&D_scalar,1,1,0,
					M,1,numdof,0,
					&Ke->values[0],1);

		/*Create Elasitc part*/
		this->GetB(B,element,xyz_list,gauss);
		this->CreateCTensor(&C[0][0],element,gauss);
		for(int i=0;i<3;i++){
			for(int j=0;j<3;j++){
				C[i][j] = gauss->weight*Jdet*dt*C[i][j];
			}
		}
		TripleMultiply(B,3,numdof,1,
					&C[0][0],3,3,0,
					B,3,numdof,0,
					&Ke->values[0],1);
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(B);
	delete gauss;
	return Ke;
}
/*}}}*/
ElementVector* SeaiceAnalysis::CreatePVector(Element* element){/*{{{*/

	/* Check if there is ice in this element */
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	if(concentration_input->Max()==0.) return NULL;

	/*Intermediaries */
	IssmDouble  concentration,thickness,dt,Jdet,vx,vy;
	IssmDouble  rho_air,air_coef,air_lin_drag_coef,air_quad_drag_coef;
	IssmDouble  windvx,windvy,wind_vnorm;
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = 2*numnodes;

	/*Initialize Element vector and vectors*/
	ElementVector* pe    = element->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	element->FindParam(&rho_air,SurfaceforcingsRhoAirEnum);
	element->FindParam(&air_coef,SurfaceforcingsAirCoefEnum);
	element->FindParam(&air_lin_drag_coef,SurfaceforcingsAirLinDragCoefEnum);
	element->FindParam(&air_quad_drag_coef,SurfaceforcingsAirQuadDragCoefEnum);
	Input* vx_input            = element->GetInput(VxEnum);                    _assert_(vx_input);
	Input* vy_input            = element->GetInput(VyEnum);                    _assert_(vy_input);
	Input* windvx_input        = element->GetInput(SurfaceforcingsWindVxEnum); _assert_(windvx_input);
	Input* windvy_input        = element->GetInput(SurfaceforcingsWindVyEnum); _assert_(windvy_input);
	Input* thickness_input     = element->GetInput(SeaiceThicknessEnum);       _assert_(thickness_input);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		element->NodalFunctions(basis, gauss);

		thickness_input->GetInputValue(&thickness,gauss);
		concentration_input->GetInputValue(&concentration,gauss);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		windvx_input->GetInputValue(&windvx,gauss);
		windvy_input->GetInputValue(&windvy,gauss);

		/*Atmespheric forcing*/
		wind_vnorm = sqrt(windvx*windvx + windvy*windvy);

		for(int i=0;i<numnodes;i++){
			pe->values[i*2+0] += dt*concentration*air_coef*rho_air*(air_lin_drag_coef+air_quad_drag_coef*wind_vnorm)*windvx*Jdet*gauss->weight*basis[i];
			pe->values[i*2+1] += dt*concentration*air_coef*rho_air*(air_lin_drag_coef+air_quad_drag_coef*wind_vnorm)*windvy*Jdet*gauss->weight*basis[i];
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	delete gauss;
	return pe;
}/*}}}*/
void SeaiceAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	   _error_("not implemented yet");
}/*}}}*/
void SeaiceAnalysis::GradientJ(Vector<IssmDouble>* gradient,Element* element,int control_type,int control_index){/*{{{*/
	_error_("Not implemented yet");
}/*}}}*/
void SeaiceAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	/*Intermediaries*/
	int*        doflist=NULL;
	IssmDouble* xyz_list=NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = numnodes*2;

	/*Fetch dof list and allocate solution vectors*/
	element->GetDofList(&doflist,SSAApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numdof);
	IssmDouble* vx        = xNew<IssmDouble>(numnodes);
	IssmDouble* vy        = xNew<IssmDouble>(numnodes);
	IssmDouble* vel       = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(int i=0;i<numnodes;i++){
		vx[i]=values[i*2+0];
		vy[i]=values[i*2+1];
		if(xIsNan<IssmDouble>(vx[i])) _error_("NaN found in solution vector");
		if(xIsNan<IssmDouble>(vy[i])) _error_("NaN found in solution vector");
	}

	/*Compute vel*/
	for(int i=0;i<numnodes;i++) vel[i]=sqrt(vx[i]*vx[i] + vy[i]*vy[i]);

	/*Add vx and vy as inputs to the tria element: */
	element->AddInput(VxEnum,vx,element->GetElementType());
	element->AddInput(VyEnum,vy,element->GetElementType());
	element->AddInput(VelEnum,vel,element->GetElementType());

	/*Free ressources:*/
	xDelete<IssmDouble>(vel);
	xDelete<IssmDouble>(vy);
	xDelete<IssmDouble>(vx);
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(xyz_list);
	xDelete<int>(doflist);
}/*}}}*/
void SeaiceAnalysis::UpdateConstraints(FemModel* femmodel){/*{{{*/
	/*Default, do nothing*/
	return;
}/*}}}*/

/*Intermediary functions*/
void SeaiceAnalysis::CreateCTensor(IssmDouble* C,Element* element,Gauss* gauss){/*{{{*/

	/*Intermediaries*/
	IssmDouble D[3][3];
	IssmDouble damage,thickness,concentration;

	/*Get Material parameters*/
	IssmDouble poisson          = element->GetMaterialParameter(MaterialsPoissonEnum);
	IssmDouble young_modulus    = element->GetMaterialParameter(MaterialsYoungModulusEnum);
	IssmDouble ridging_exponent = element->GetMaterialParameter(MaterialsRidgingExponentEnum);

	/*Get damage input at this location*/
	Input* damage_input        = element->GetInput(MaterialsDamageEnum);     _assert_(damage_input);
	Input* thickness_input     = element->GetInput(SeaiceThicknessEnum);     _assert_(thickness_input);
	Input* concentration_input = element->GetInput(SeaiceConcentrationEnum); _assert_(concentration_input);
	damage_input->GetInputValue(&damage,gauss);
	thickness_input->GetInputValue(&thickness,gauss);
	concentration_input->GetInputValue(&concentration,gauss);

	/*Build D tensor once for all*/
	D[0][0] = 1.;      D[0][1] = poisson; D[0][2] = 0.;
	D[1][0] = poisson; D[1][1] = 1.;      D[1][2] = 0.;
	D[2][0] = 0.;      D[2][1] = 0;       D[2][2] = (1.-poisson)/2.;
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			D[i][j] = D[i][j]/(1-poisson*poisson);
		}
	}

	/*Get f(A), E*/
	IssmDouble f = exp(ridging_exponent*(1.-concentration));
	IssmDouble E = young_modulus*(1.-damage)*f;

	/*Let's build C = hE*D */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			C[i*3+j] = thickness*E*D[i][j];
		}
	}
}/*}}}*/
void SeaiceAnalysis::GetB(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For node i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dN/dx           0    ] 
	 *          [   0           dN/dy  ]      
	 *          [ 1/2*dN/dy  1/2*dN/dx ]     
	 * where N is the basis function for node i.
	 *
	 * We assume B has been allocated already           */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* dbasis=xNew<IssmDouble>(2*numnodes);
	element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[2*numnodes*0+2*i+0] = dbasis[0*numnodes+i];
		B[2*numnodes*0+2*i+1] = 0.;
		B[2*numnodes*1+2*i+0] = 0.;
		B[2*numnodes*1+2*i+1] = dbasis[1*numnodes+i];
		B[2*numnodes*2+2*i+0] = .5*dbasis[1*numnodes+i];
		B[2*numnodes*2+2*i+1] = .5*dbasis[0*numnodes+i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(dbasis);
}/*}}}*/
void SeaiceAnalysis::GetM(IssmDouble* M,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute M  matrix. M=[M1 M2 M3] where Mi is of size 3*NDOF2. 
	 * For node i, Mi can be expressed in the actual coordinate system
	 * by: 
	 *       Mi=[ N  N ] 
	 * where N is the basis function for node i.
	 *
	 * We assume M has been allocated already           */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* basis=xNew<IssmDouble>(numnodes);
	element->NodalFunctions(basis,gauss);

	/*Build M: */
	for(int i=0;i<numnodes;i++){
		M[2*i+0] = basis[i];
		M[2*i+1] = basis[i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(basis);
}/*}}}*/
