function results=diagnostic_core(models,inputs);
%DIAGNOSTIC_CORE - compute the core velocity field 
%
%   Usage:
%      results=diagnostic_core(models,inputs);
%

%recover models
m_dh=models.dh;
m_dv=models.dv;
m_ds=models.ds;
m_dhu=models.dhu;
m_sl=models.sl;

%recover parameters common to all solutions
debug=m_dhu.parameters.debug;
dim=m_dhu.parameters.dim;
ishutter=m_dhu.parameters.ishutter;
ismacayealpattyn=m_dh.parameters.ismacayealpattyn;
isstokes=m_ds.parameters.isstokes;
numrifts=m_dhu.parameters.numrifts;

if ishutter,

	displaystring(debug,'\n%s',['computing surface slope (x and y derivatives)...']);
	slopex=diagnostic_core_linear(m_sl,inputs,SlopeComputeAnalysisEnum(),SurfaceXAnalysisEnum());
	slopey=diagnostic_core_linear(m_sl,inputs,SlopeComputeAnalysisEnum(),SurfaceYAnalysisEnum());

	if dim==3,
		displaystring(debug,'\n%s',['extruding slopes in 3d...']);
		slopex=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopex,'slopex',0);
		slopey=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopey,'slopey',0);
	end

	displaystring(debug,'\n%s',['computing slopes...']);
	inputs=add(inputs,'surfaceslopex',slopex,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
	inputs=add(inputs,'surfaceslopey',slopey,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);

	displaystring(debug,'\n%s',['computing hutter velocities...']);
	u_g=diagnostic_core_linear(m_dhu,inputs,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

	displaystring(debug,'\n%s',['computing pressure according to MacAyeal...']);
	p_g=ComputePressure(m_dhu.elements,m_dhu.nodes,m_dhu.loads,m_dhu.materials,m_dhu.parameters,inputs);

	displaystring(debug,'\n%s',['update boundary conditions for macyeal pattyn using hutter results...']);
	if ismacayealpattyn,
		m_dh.y_g=u_g;
		[m_dh.ys m_dh.ys0]=Reducevectorgtos(m_dh.y_g,m_dh.nodesets);
	end

end
		
if ismacayealpattyn,

	displaystring(debug,'\n%s',['computing horizontal velocities...']);
	[u_g m_dh.loads]=diagnostic_core_nonlinear(m_dh,inputs,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	displaystring(debug,'\n%s',['computing pressure according to MacAyeal...']);
	p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,m_dh.parameters,inputs);
end
	
if dim==3,

	displaystring(debug,'\n%s',['extruding horizontal velocities...']);
	u_g_horiz=FieldExtrude(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,u_g,'velocity',1);

	displaystring(debug,'\n%s',['computing vertical velocities...']);
	inputs=add(inputs,'velocity',u_g_horiz,'doublevec',m_dh.parameters.numberofdofspernode,m_dh.parameters.numberofnodes);
	u_g_vert=diagnostic_core_linear(m_dv,inputs,DiagnosticAnalysisEnum(),VertAnalysisEnum());

	displaystring(debug,'\n%s',['combining horizontal and vertical velocities...']);
	u_g=zeros(m_dh.parameters.numberofnodes*3,1);
	u_g(dofsetgen([1,2],3,m_dh.parameters.numberofnodes*3))=u_g_horiz;
	u_g(dofsetgen([3],3,m_dh.parameters.numberofnodes*3))=u_g_vert;

	displaystring(debug,'\n%s',['computing pressure according to Pattyn...']);
	p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,m_dh.parameters,inputs);
	
	if isstokes,

		%"recondition" pressure 
		p_g=p_g/m_ds.parameters.stokesreconditioning;

		displaystring(debug,'\n%s',['computing bed slope (x and y derivatives)...']);
		slopex=diagnostic_core_linear(m_sl,inputs,SlopeComputeAnalysisEnum(),BedXAnalysisEnum());
		slopey=diagnostic_core_linear(m_sl,inputs,SlopeComputeAnalysisEnum(),BedYAnalysisEnum());
		slopex=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopex,'slopex',0);
		slopey=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopey,'slopey',0);

		inputs=add(inputs,'bedslopex',slopex,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
		inputs=add(inputs,'bedslopey',slopey,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
		
		%recombine u_g and p_g: 
		u_g_stokes=zeros(m_ds.nodesets.gsize,1);
		u_g_stokes(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
		u_g_stokes(dofsetgen([4],4,m_ds.nodesets.gsize))=p_g;

		inputs=add(inputs,'velocity',u_g_stokes,'doublevec',4,m_ds.parameters.numberofnodes);

		displaystring(debug,'\n%s',['update boundary conditions for stokes using velocities previously computed...']);
		m_ds.y_g=zeros(m_ds.nodesets.gsize,1);
		m_ds.y_g(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
		[m_ds.ys m_ds.ys0]=Reducevectorgtos(m_ds.y_g,m_ds.nodesets);

		displaystring(debug,'\n%s',['computing stokes velocities and pressure ...']);
		u_g=diagnostic_core_nonlinear(m_ds,inputs,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
		%"decondition" pressure
		p_g=u_g(4:4:end)*m_dh.parameters.stokesreconditioning;
	end
end

%load onto results
results.step=1;
results.time=0;
results.u_g=u_g;
results.p_g=p_g;

if numrifts,
	results.riftproperties=OutputRifts(m_dh.loads,m_dh.parameters);
end
