/*!\file: solutionsequence_linear.cpp
 * \brief: numerical core of linear solutions
 */ 

#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

void solutionsequence_fct(FemModel* femmodel){

	/*intermediary: */
	Vector<IssmDouble>*  Mlf = NULL;
	Matrix<IssmDouble>*  Kff = NULL;
	Matrix<IssmDouble>*  Kfs = NULL;
	Vector<IssmDouble>*  ug  = NULL;
	Vector<IssmDouble>*  uf  = NULL;
	Vector<IssmDouble>*  ys  = NULL;

	IssmDouble theta,deltat;
	int        dof,ncols,ncols2,ncols3,rstart,rend;
	int        configuration_type,analysis_type;
	double     d,diagD,mi;
	double     dmax = 0.;
	int*       cols  = NULL;
	int*       cols2 = NULL;
	int*       cols3 = NULL;
	double*    vals  = NULL;
	double*    vals2 = NULL;
	double*    vals3 = NULL;

	/*Create analysis*/
	MasstransportAnalysis* analysis = new MasstransportAnalysis();

	/*Recover parameters: */
	femmodel->parameters->FindParam(&deltat,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	femmodel->UpdateConstraintsx();
	theta = 0.5;

	/*Create lumped mass matrix*/
	analysis->LumpedMassMatrix(&Mlf,femmodel);
	analysis->FctKMatrix(&Kff,&Kfs,femmodel);

	/*Create Dff Matrix*/
	#ifdef _HAVE_PETSC_
	Mat Kff_transp = NULL;
	Mat Dff_petsc  = NULL;
	Mat LHS        = NULL;
	Mat Kff_petsc  = Kff->pmatrix->matrix;
	Vec Mlf_petsc  = Mlf->pvector->vector;
	MatTranspose(Kff_petsc,MAT_INITIAL_MATRIX,&Kff_transp);
	MatDuplicate(Kff_petsc,MAT_SHARE_NONZERO_PATTERN,&Dff_petsc);
	MatGetOwnershipRange(Kff_transp,&rstart,&rend);
	for(int row=rstart; row<rend; row++){
		diagD = 0.;
		MatGetRow(Kff_petsc ,row,&ncols, (const int**)&cols, (const double**)&vals);
		MatGetRow(Kff_transp,row,&ncols2,(const int**)&cols2,(const double**)&vals2);
		_assert_(ncols==ncols2);
		for(int j=0; j<ncols; j++) {
			_assert_(cols[j]==cols2[j]);
			d = max(max(-vals[j],-vals2[j]),0.);
			MatSetValues(Dff_petsc,1,&row,1,&cols[j],(const double*)&d,INSERT_VALUES);
			if(cols[j]!=row) diagD -= d;
		}
		MatSetValues(Dff_petsc,1,&row,1,&row,(const double*)&diagD,INSERT_VALUES);
		MatRestoreRow(Kff_petsc, row,&ncols, (const int**)&cols, (const double**)&vals);
		MatRestoreRow(Kff_transp,row,&ncols2,(const int**)&cols2,(const double**)&vals2);
	}
	MatAssemblyBegin(Dff_petsc,MAT_FINAL_ASSEMBLY);
	MatAssemblyEnd(  Dff_petsc,MAT_FINAL_ASSEMBLY);
	MatFree(&Kff_transp);

	/*Create LHS: [ML − theta*detlat *(K+D)^n+1]*/
	MatDuplicate(Kff_petsc,MAT_SHARE_NONZERO_PATTERN,&LHS);
	for(int row=rstart; row<rend; row++){
		MatGetRow(Kff_petsc,row,&ncols, (const int**)&cols, (const double**)&vals);
		MatGetRow(Dff_petsc,row,&ncols2,(const int**)&cols2,(const double**)&vals2);
		_assert_(ncols==ncols2);
		for(int j=0; j<ncols; j++) {
			_assert_(cols[j]==cols2[j]);
			d = -theta*deltat*(vals[j] + vals2[j]);
			if(cols[j]==row){
				VecGetValues(Mlf_petsc,1,(const int*)&cols[j],&mi);
				d += mi;
			}
			if(fabs(d)>dmax) dmax = fabs(d);
			MatSetValues(LHS,1,&row,1,&cols[j],(const double*)&d,INSERT_VALUES);
		}
		MatRestoreRow(Kff_petsc,row,&ncols, (const int**)&cols, (const double**)&vals);
		MatRestoreRow(Dff_petsc,row,&ncols2,(const int**)&cols2,(const double**)&vals2);
	}

	/*Penalize Dirichlet boundary*/
	dmax = dmax * 1.e+3;
	for(int i=0;i<femmodel->constraints->Size();i++){
		Constraint* constraint=(Constraint*)femmodel->constraints->GetObjectByOffset(i);
		if(constraint->InAnalysis(analysis_type)){
			constraint->PenaltyDofAndValue(&dof,&d,femmodel->nodes,femmodel->parameters);
			if(dof!=-1){
				MatSetValues(LHS,1,&dof,1,&dof,(const double*)&dmax,INSERT_VALUES);
			}
		}
	}
	MatAssemblyBegin(LHS,MAT_FINAL_ASSEMBLY);
	MatAssemblyEnd(  LHS,MAT_FINAL_ASSEMBLY);

	/*Create RHS: [ML + (1 − theta) deltaT L^n] u^n */
	GetSolutionFromInputsx(&ug,femmodel);
	Reducevectorgtofx(&uf, ug, femmodel->nodes,femmodel->parameters);
	delete ug;
	Vec u  = uf->pvector->vector;
	Vec Ku = NULL;
	Vec Du = NULL;
	Vec RHS = NULL;
	VecDuplicate(u,&Ku);
	VecDuplicate(u,&Du);
	VecDuplicate(u,&RHS);
	MatMult(Kff_petsc,u,Ku);
	MatMult(Dff_petsc,u,Du);
	VecPointwiseMult(RHS,Mlf_petsc,u);
	VecAXPBYPCZ(RHS,(1-theta)*deltat,(1-theta)*deltat,1,Ku,Du);// RHS = M*u + (1-theta)*deltat*K*u + (1-theta)*deltat*D*u
	VecFree(&Ku);
	VecFree(&Du);
	MatFree(&Dff_petsc);
	delete uf;

	/*Penalize Dirichlet boundary*/
	for(int i=0;i<femmodel->constraints->Size();i++){
		Constraint* constraint=(Constraint*)femmodel->constraints->GetObjectByOffset(i);
		if(constraint->InAnalysis(analysis_type)){
			constraint->PenaltyDofAndValue(&dof,&d,femmodel->nodes,femmodel->parameters);
			d = d*dmax;
			if(dof!=-1){
				VecSetValues(RHS,1,&dof,(const double*)&d,INSERT_VALUES);
			}
		}
	}

	/*Go solve!*/
	SolverxPetsc(&u,LHS,RHS,NULL,NULL, femmodel->parameters); 
	MatFree(&LHS);
	VecFree(&RHS);
	uf =new Vector<IssmDouble>(u);
	VecFree(&u);

	Mergesolutionfromftogx(&ug, uf,ys,femmodel->nodes,femmodel->parameters);delete uf; delete ys;
	InputUpdateFromSolutionx(femmodel,ug); 
	delete ug;  

	#else
	_error_("PETSc needs to be installed");
	#endif

	delete Mlf;
	delete Kff;
	delete Kfs;
	delete analysis;

}
