/*\file KMLOverlay.c
 *\brief: KML file overlay mex module.
 */
#include "./KMLOverlay.h"

void KMLOverlayUsage(void){/*{{{*/
	_printf0_("KMLOverlay - KML file overlay module:\n");
	_printf0_("\n");
	_printf0_("   This module reads a list of image files and writes a KML or KMZ overlay file.\n");
	_printf0_("\n");
	_printf0_("   Usage:\n");
	_printf0_("      ierror=KMLOverlay(kmlfile,'param name',param,...);\n");
	_printf0_("\n");
	_printf0_("      kmlfile     KML or KMZ file name (string)\n");
	_printf0_("\n");
	_printf0_("      lataxis     latitude axis (double vector [south north], required)\n");
	_printf0_("      longaxis    longitude axis (double vector [west east], required)\n");
	_printf0_("      images      relative or http image file names (string or array of strings or cell array of strings, required)\n");
	_printf0_("      zip         flag to zip the doc.kml and image files into kmzfile (double, non-zero for kmz)\n");
	_printf0_("\n");
	_printf0_("      ierror     error flag (double, non-zero for error)\n");
	_printf0_("\n");
	_printf0_("   Example:\n");
	_printf0_("      KMLOverlay(kmlfile,'lataxis',[south north],'longaxis',[west east],'images',{'file1.png','http://issm/file2.png'},'zip',1);\n");
	_printf0_("\n");
}/*}}}*/
WRAPPER(KMLOverlay){

	int i,verbose=1;

	/*input: */
	char*   filkml=NULL;
	char*   filkmz=NULL;

	FILE*   fid=NULL;

	Options* options=NULL;
	int      nlat=0,nlong=0;
	double*  lataxis =NULL;
	double*  longaxis=NULL;
	int      nimages=0;
	char**   pimages=NULL;
	double   dzip=0;
	char*    czip=NULL;

	/* output: */
	int     ierror=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if(nlhs>NLHS){
		KMLOverlayUsage(); _error_("KMLOverlay usage error");
	}
	if(nrhs<NRHS){
		KMLOverlayUsage(); _error_("KMLOverlay usage error");
	}

	/*Input datasets: */
	FetchData(&filkml,FILENAME);
	FetchData(&options,NRHS,nrhs,prhs);

	options->Get(&lataxis ,&nlat ,(char*)"lataxis");
	if (verbose && lataxis) for (i=0; i<nlat; i++) _printf_("  lataxis [" << i << "]=" << lataxis[i] << "\n");
	options->Get(&longaxis,&nlong,(char*)"longaxis");
	if (verbose && longaxis) for (i=0; i<nlong; i++) _printf_("  longaxis[" << i << "]=" << longaxis[i] << "\n");
	options->Get(&pimages,&nimages,(char*)"images");
	if (verbose && pimages) for (i=0; i<nimages; i++) _printf_("  pimages[" << i << "]=\"" << pimages[i] << "\"\n");
	options->Get(&dzip,(char*)"zip",0.);
	if (verbose) _printf_("  dzip=" << dzip << "\n");

	/*some checks*/
	if (nlat !=2) _error_("Latitudinal axes \"lataxis\" require two double values, not " << nlat << ".");
	if (nlong!=2) _error_("Longitudinal axes \"longaxis\" require two double values, not " << nlong << ".");
	if (!nimages) _error_("No image files provided.");

	if ((int)dzip){
		filkmz=filkml;
		filkml=(char*)mxMalloc(8*sizeof(char));
		strcpy(filkml,"doc.kml");
	}

	if(!strlen(filkml)) strcpy(filkml,"stdout");

	if(verbose) _printf_("Opening kml overlay file \"" << filkml << "\".\n");
	fid=fopen(filkml,"w");

	/* Run core computations: */
	if (verbose) _printf_("Calling core:\n");
	KMLOverlayx(&ierror,lataxis,longaxis,nimages,pimages,fid);

	if (verbose) _printf_("Closing file \"" << filkml << "\".\n");
	fclose(fid);

	/* Create kmz file, if specified: */
	if ((int)dzip) {
		czip=(char*)mxMalloc((5+strlen(filkmz)+1+strlen(filkml)+1)*sizeof(char));
		czip[0]='\0';
		strcat(czip,"!zip ");
		strcat(czip,filkmz);
		strcat(czip," ");
		strcat(czip,filkml);
		for (i=0; i<nimages; i++)
			if (strlen(pimages[i]) && strncmp(pimages[i],"http",4)) {
				czip=(char*)mxRealloc(czip,(strlen(czip)+1+strlen(pimages[i])+1)*sizeof(char));
				strcat(czip," ");
				strcat(czip,pimages[i]);
			}
		if (verbose) _printf_("Zipping file \"" << filkmz << "\".\n");
		if (verbose) _printf_(czip << "\n");

		if (mexEvalString(czip)) _error_("Error zipping file \"" << filkmz << "\".");
		xDelete<char>(czip);
		xDelete<char>(filkmz);
	}

	/*Write data: */
	WriteData(ERRORFLAG,ierror);

	/*Clean-up*/
	if (pimages) {
		for (i=nimages; i>0; i--) xDelete<char>(pimages[i-1]);
		xDelete<char*>(pimages);
	}
	xDelete<double>(longaxis);
	xDelete<double>(lataxis);
	xDelete<char>(filkml);
	delete options;

	/*end module: */
	MODULEEND();
}
