/*!\file TriaInput.c
 * \brief: implementation of the TriaInput object
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../classes.h"
#include "../../shared/shared.h"

/*TriaInput constructors and destructor*/
/*FUNCTION TriaInput::TriaInput(){{{*/
TriaInput::TriaInput(){
	values = NULL;
}
/*}}}*/
/*FUNCTION TriaInput::TriaInput(int in_enum_type,IssmDouble* invalues,element_type_in){{{*/
TriaInput::TriaInput(int in_enum_type,IssmDouble* in_values,int element_type_in)
	:TriaRef(1)
{

	/*Set TriaRef*/
	this->SetElementType(element_type_in,0);
	this->element_type=element_type_in;

	/*Set Enum*/
	enum_type=in_enum_type;

	/*Set values*/
	this->values=xNew<IssmDouble>(this->NumberofNodes());
	for(int i=0;i<this->NumberofNodes();i++) values[i]=in_values[i];
}
/*}}}*/
/*FUNCTION TriaInput::~TriaInput(){{{*/
TriaInput::~TriaInput(){
	xDelete<IssmDouble>(this->values);
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION TriaInput::Echo {{{*/
void TriaInput::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION TriaInput::DeepEcho{{{*/
void TriaInput::DeepEcho(void){

	_printf_(setw(15)<<"   TriaInput "<<setw(25)<<left<<EnumToStringx(this->enum_type)<<" [");
	for(int i=0;i<this->NumberofNodes();i++) _printf_(" "<<this->values[i]);
	_printf_("] ("<<EnumToStringx(this->element_type)<<")\n");
}
/*}}}*/
/*FUNCTION TriaInput::Id{{{*/
int    TriaInput::Id(void){ return -1; }
/*}}}*/
/*FUNCTION TriaInput::ObjectEnum{{{*/
int TriaInput::ObjectEnum(void){

	return TriaInputEnum;

}
/*}}}*/
/*FUNCTION TriaInput::copy{{{*/
Object* TriaInput::copy() {

	return new TriaInput(this->enum_type,this->values,this->element_type);

}
/*}}}*/

/*TriaInput management*/
/*FUNCTION TriaInput::InstanceEnum{{{*/
int TriaInput::InstanceEnum(void){

	return this->enum_type;

}
/*}}}*/
/*FUNCTION TriaInput::SpawnTriaInput{{{*/
Input* TriaInput::SpawnTriaInput(int index1,int index2,int index3){

	/*output*/
	TriaInput* outinput=NULL;

	/*Create new Tria input (copy of current input)*/
	outinput=new TriaInput(this->enum_type,&this->values[0],this->element_type);

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaInput::SpawnSegInput{{{*/
Input* TriaInput::SpawnSegInput(int index1,int index2){

	/*output*/
	SegInput* outinput=NULL;

	if(this->element_type==P0Enum){ 
		outinput=new SegInput(this->enum_type,&this->values[0],P0Enum);
	}
	else{
		/*Assume P1 interpolation only for now*/
		IssmDouble newvalues[2];

		/*Create array of indices depending on location (0=base 1=surface)*/
		newvalues[0]=this->values[index1];
		newvalues[1]=this->values[index2];

		/*Create new Seg input*/
		outinput=new SegInput(this->enum_type,&newvalues[0],P1Enum);
	}

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaInput::GetResultInterpolation{{{*/
int  TriaInput::GetResultInterpolation(void){

	return P1Enum;

}
/*}}}*/
/*FUNCTION TriaInput::GetResultNumberOfNodes{{{*/
int  TriaInput::GetResultNumberOfNodes(void){

	return this->NumberofNodes();

}
/*}}}*/
/*FUNCTION TriaInput::ResultToPatch{{{*/
void TriaInput::ResultToPatch(IssmDouble* values,int nodesperelement,int sid){

	int numnodes = this->NumberofNodes();

	/*Some checks*/
	_assert_(values);
	_assert_(numnodes==nodesperelement);

	/*Fill in arrays*/
	for(int i=0;i<numnodes;i++) values[sid*numnodes + i] = this->values[i];
}
/*}}}*/

/*Object functions*/
/*FUNCTION TriaInput::GetInputValue(IssmDouble* pvalue,Gauss* gauss){{{*/
void TriaInput::GetInputValue(IssmDouble* pvalue,Gauss* gauss){

	/*Call TriaRef function*/
	_assert_(gauss->Enum()==GaussTriaEnum);
	TriaRef::GetInputValue(pvalue,&values[0],(GaussTria*)gauss);

}
/*}}}*/
/*FUNCTION TriaInput::GetInputDerivativeValue(IssmDouble* p, IssmDouble* xyz_list, Gauss* gauss){{{*/
void TriaInput::GetInputDerivativeValue(IssmDouble* p, IssmDouble* xyz_list, Gauss* gauss){

	/*Call TriaRef function*/
	_assert_(gauss->Enum()==GaussTriaEnum);
	TriaRef::GetInputDerivativeValue(p,&values[0],xyz_list,(GaussTria*)gauss);
}
/*}}}*/
/*FUNCTION TriaInput::ChangeEnum{{{*/
void TriaInput::ChangeEnum(int newenumtype){
	this->enum_type=newenumtype;
}
/*}}}*/
/*FUNCTION TriaInput::GetInputAverage{{{*/
void TriaInput::GetInputAverage(IssmDouble* pvalue){

	int        numnodes  = this->NumberofNodes();
	IssmDouble numnodesd = reCast<int,IssmDouble>(numnodes);
	IssmDouble value     = 0.;

	for(int i=0;i<numnodes;i++) value+=values[i];
	value = value/numnodesd;

	*pvalue=value;
}
/*}}}*/
/*FUNCTION TriaInput::GetInputAllTimeAverages{{{*/
void TriaInput::GetInputAllTimeAverages(IssmDouble** pvalues,IssmDouble** ptimes, int* pnumtimes){

	IssmDouble* outvalues=NULL;
	IssmDouble* times=NULL;
	int         numtimes;

	/*this is not a transient forcing, so we only have 1 value, steady state: */
	numtimes=1;
	outvalues=xNew<IssmDouble>(1);
	times=xNew<IssmDouble>(1);

	this->GetInputAverage(&outvalues[0]);
	times[0]=0.; /*we don't have a time*/

	*pvalues=outvalues;
	*ptimes=times;
	*pnumtimes=numtimes;
}
/*}}}*/
/*FUNCTION TriaInput::GetInputUpToCurrentTimeAverages{{{*/
void TriaInput::GetInputUpToCurrentTimeAverages(IssmDouble** pvalues, IssmDouble** ptimes, int* pnumtimes, IssmDouble currenttime){

	IssmDouble* outvalues=NULL;
	IssmDouble* times=NULL;
	int         numtimes;

	/*this is not a transient forcing, so we only have 1 value, steady state: */
	numtimes=1;
	outvalues=xNew<IssmDouble>(1);
	times=xNew<IssmDouble>(1);

	this->GetInputAverage(&outvalues[0]);
	times[0]=currenttime; /*we don't have a time*/

	*pvalues=outvalues;
	*ptimes=times;
	*pnumtimes=numtimes;
}
/*}}}*/

/*Intermediary*/
/*FUNCTION TriaInput::SquareMin{{{*/
void TriaInput::SquareMin(IssmDouble* psquaremin,Parameters* parameters){

	int        numnodes=this->NumberofNodes();
	IssmDouble squaremin;

	/*Now, figure out minimum of valuescopy: */
	squaremin=pow(this->values[0],2);
	for(int i=1;i<numnodes;i++){
		if(pow(this->values[i],2)<squaremin)squaremin=pow(this->values[i],2);
	}
	/*Assign output pointers:*/
	*psquaremin=squaremin;
}
/*}}}*/
/*FUNCTION TriaInput::ContrainMin{{{*/
void TriaInput::ConstrainMin(IssmDouble minimum){

	int numnodes = this->NumberofNodes();
	for(int i=0;i<numnodes;i++) if (values[i]<minimum) values[i]=minimum;
}
/*}}}*/
/*FUNCTION TriaInput::InfinityNorm{{{*/
IssmDouble TriaInput::InfinityNorm(void){

	/*Output*/
	IssmDouble norm=0.;
	int numnodes=this->NumberofNodes();

	for(int i=0;i<numnodes;i++) if(fabs(values[i])>norm) norm=fabs(values[i]);
	return norm;
}
/*}}}*/
/*FUNCTION TriaInput::Max{{{*/
IssmDouble TriaInput::Max(void){

	int  numnodes=this->NumberofNodes();
	IssmDouble max=values[0];

	for(int i=1;i<numnodes;i++){
		if(values[i]>max) max=values[i];
	}
	return max;
}
/*}}}*/
/*FUNCTION TriaInput::MaxAbs{{{*/
IssmDouble TriaInput::MaxAbs(void){

	int  numnodes=this->NumberofNodes();
	IssmDouble max=fabs(values[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(values[i])>max) max=fabs(values[i]);
	}
	return max;
}
/*}}}*/
/*FUNCTION TriaInput::Min{{{*/
IssmDouble TriaInput::Min(void){

	const int  numnodes=this->NumberofNodes();
	IssmDouble min=values[0];

	for(int i=1;i<numnodes;i++){
		if(values[i]<min) min=values[i];
	}
	return min;
}
/*}}}*/
/*FUNCTION TriaInput::MinAbs{{{*/
IssmDouble TriaInput::MinAbs(void){

	const int  numnodes=this->NumberofNodes();
	IssmDouble min=fabs(values[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(values[i])<min) min=fabs(values[i]);
	}
	return min;
}
/*}}}*/
/*FUNCTION TriaInput::Scale{{{*/
void TriaInput::Scale(IssmDouble scale_factor){

	const int numnodes=this->NumberofNodes();
	for(int i=0;i<numnodes;i++)values[i]=values[i]*scale_factor;
}
/*}}}*/
/*FUNCTION TriaInput::Set{{{*/
void TriaInput::Set(IssmDouble setvalue){

	const int numnodes=this->NumberofNodes();
	for(int i=0;i<numnodes;i++)values[i]=setvalue;
}
/*}}}*/
/*FUNCTION TriaInput::AXPY{{{*/
void TriaInput::AXPY(Input* xinput,IssmDouble scalar){

	const int numnodes=this->NumberofNodes();
	TriaInput*  xtriainput=NULL;

	/*xinput is of the same type, so cast it: */
	if(xinput->ObjectEnum()!=TriaInputEnum) _error_("Operation not permitted because xinput is of type " << EnumToStringx(xinput->ObjectEnum()));
	xtriainput=(TriaInput*)xinput;
	if(xtriainput->element_type!=this->element_type) _error_("Operation not permitted because xinput is of type " << EnumToStringx(xinput->ObjectEnum()));

	/*Carry out the AXPY operation depending on type:*/
	for(int i=0;i<numnodes;i++)this->values[i]=this->values[i]+scalar*xtriainput->values[i];

}
/*}}}*/
/*FUNCTION TriaInput::Constrain{{{*/
void TriaInput::Constrain(IssmDouble cm_min, IssmDouble cm_max){

	int i;
	const int numnodes=this->NumberofNodes();

	if(!xIsNan<IssmDouble>(cm_min)) for(i=0;i<numnodes;i++)if (this->values[i]<cm_min)this->values[i]=cm_min;
	if(!xIsNan<IssmDouble>(cm_max)) for(i=0;i<numnodes;i++)if (this->values[i]>cm_max)this->values[i]=cm_max;

}
/*}}}*/
/*FUNCTION TriaInput::GetVectorFromInputs{{{*/
void TriaInput::GetVectorFromInputs(Vector<IssmDouble>* vector,int* doflist){

	const int numnodes=this->NumberofNodes();
	vector->SetValues(numnodes,doflist,this->values,INS_VAL);

} /*}}}*/
/*FUNCTION TriaInput::PointwiseMin{{{*/
Input* TriaInput::PointwiseMin(Input* inputB){

	/*Ouput*/
	TriaInput* outinput=NULL;

	/*Intermediaries*/
	int         i;
	TriaInput  *xinputB   = NULL;
	const int   numnodes  = this->NumberofNodes();
	IssmDouble *minvalues = xNew<IssmDouble>(numnodes);

	/*Check that inputB is of the same type*/
	if(inputB->ObjectEnum()!=TriaInputEnum)       _error_("Operation not permitted because inputB is of type " << EnumToStringx(inputB->ObjectEnum()));
	xinputB=(TriaInput*)inputB;
	if(xinputB->element_type!=this->element_type) _error_("Operation not permitted because inputB is of type " << EnumToStringx(xinputB->element_type));

	/*Create point wise min*/
	for(i=0;i<numnodes;i++){
		if(this->values[i] > xinputB->values[i]) minvalues[i]=xinputB->values[i];
		else minvalues[i]=this->values[i];
	}

	/*Create new Tria vertex input (copy of current input)*/
	outinput=new TriaInput(this->enum_type,&minvalues[0],this->element_type);

	/*Return output pointer*/
	xDelete<IssmDouble>(minvalues);
	return outinput;

}
/*}}}*/
/*FUNCTION TriaInput::PointwiseMax{{{*/
Input* TriaInput::PointwiseMax(Input* inputB){

	/*Ouput*/
	TriaInput* outinput=NULL;

	/*Intermediaries*/
	int         i;
	TriaInput  *xinputB   = NULL;
	const int   numnodes  = this->NumberofNodes();
	IssmDouble *maxvalues = xNew<IssmDouble>(numnodes);

	/*Check that inputB is of the same type*/
	if(inputB->ObjectEnum()!=TriaInputEnum) _error_("Operation not permitted because inputB is of type " << EnumToStringx(inputB->ObjectEnum()));
	xinputB=(TriaInput*)inputB;
	if(xinputB->element_type!=this->element_type) _error_("Operation not permitted because inputB is of type " << EnumToStringx(xinputB->element_type));

	/*Create point wise max*/
	for(i=0;i<numnodes;i++){
		if(this->values[i] < xinputB->values[i]) maxvalues[i]=xinputB->values[i];
		else maxvalues[i]=this->values[i];
	}

	/*Create new Tria vertex input (copy of current input)*/
	outinput=new TriaInput(this->enum_type,&maxvalues[0],this->element_type);

	/*Return output pointer*/
	xDelete<IssmDouble>(maxvalues);
	return outinput;

}
/*}}}*/
/*FUNCTION TriaInput::PointwiseDivide{{{*/
Input* TriaInput::PointwiseDivide(Input* inputB){

	/*Ouput*/
	TriaInput* outinput=NULL;

	/*Intermediaries*/
	TriaInput *xinputB  = NULL;
	const int   numnodes = this->NumberofNodes();

	/*Check that inputB is of the same type*/
	if(inputB->ObjectEnum()!=TriaInputEnum)     _error_("Operation not permitted because inputB is of type " << EnumToStringx(inputB->ObjectEnum()));
	xinputB=(TriaInput*)inputB;
	if(xinputB->element_type!=this->element_type) _error_("Operation not permitted because inputB is of type " << EnumToStringx(xinputB->element_type));

	/*Allocate intermediary*/
	IssmDouble* AdotBvalues=xNew<IssmDouble>(numnodes);

	/*Create point wise division*/
	for(int i=0;i<numnodes;i++){
		_assert_(xinputB->values[i]!=0);
		AdotBvalues[i]=this->values[i]/xinputB->values[i];
	}

	/*Create new Tria vertex input (copy of current input)*/
	outinput=new TriaInput(this->enum_type,AdotBvalues,this->element_type);

	/*Return output pointer*/
	xDelete<IssmDouble>(AdotBvalues);
	return outinput;

}
/*}}}*/
/*FUNCTION TriaInput::Configure{{{*/
void TriaInput::Configure(Parameters* parameters){
	/*do nothing: */
}
/*}}}*/
