import numpy
from fielddisplay import fielddisplay
from EnumDefinitions import *
from checkfield import checkfield
from WriteData import WriteData
from project3d import project3d

class SMBpdd(object):
	"""
	SMBpdd Class definition

	   Usage:
	      SMBpdd=SMBpdd();
	"""

	def __init__(self): # {{{
		self.precipitation             = float('NaN')
		self.monthlytemperatures       = float('NaN')
		self.desfac                    = 0.
		self.s0p                       = 0.
		self.isdelta18o                = 0
		self.delta18o                  = float('NaN')
		self.delta18o_surface          = float('NaN')
		self.temperatures_presentday   = float('NaN')
		self.temperatures_lgm          = float('NaN')
		self.precipitations_presentday = float('NaN')

		#set defaults
		self.setdefaultparameters()
		#}}}
	def __repr__(self): # {{{
		string="   surface forcings parameters:"

		string="%s\n%s"%(string,fielddisplay(self,'precipitation','surface precipitation [m/yr water eq]'))
		string="%s\n%s"%(string,fielddisplay(self,'desfac','desertification elevation factor (between 0 and 1, default is 0.5) [m]'))
		string="%s\n%s"%(string,fielddisplay(self,'isdelta18o','is temperature and precipitation delta18o parametrisation activated (0 or 1, default is 0)'))
		string="%s\n%s"%(string,fielddisplay(self,'s0p','should be set to elevation from precip source (between 0 and a few 1000s m, default is 0) [m]'))
		string="%s\n%s"%(string,fielddisplay(self,'monthlytemperatures','monthly surface temperatures [K], required if pdd is activated and delta18o not activated'))
		string="%s\n%s"%(string,fielddisplay(self,'temperatures_presentday','monthly present day surface temperatures [K], required if pdd is activated and delta18o activated'))
		string="%s\n%s"%(string,fielddisplay(self,'temperatures_lgm','monthly LGM surface temperatures [K], required if pdd is activated and delta18o activated'))
		string="%s\n%s"%(string,fielddisplay(self,'precipitations_presentday','monthly surface precipitation [m/yr water eq], required if pdd is activated and delta18o activated'))
		string="%s\n%s"%(string,fielddisplay(self,'delta18o','delta18o, required if pdd is activated and delta18o activated'))
		string="%s\n%s"%(string,fielddisplay(self,'delta18o_surface','surface elevation of the delta18o site, required if pdd is activated and delta18o activated'))

		return string
		#}}}
	def extrude(self,md): # {{{

		self.precipitation=project3d(md,'vector',self.precipitation,'type','node');
		self.monthlytemperatures=project3d(md,'vector',self.monthlytemperatures,'type','node');
		if self.isdelta18o: self.temperatures_lgm=project3d(md,'vector',self.temperatures_lgm,'type','node')
		if self.isdelta18o: self.temperatures_presentday=project3d(md,'vector',self.temperatures_presentday,'type','node')
		if self.isdelta18o: self.precipitations_presentday=project3d(md,'vector',self.precipitations_presentday,'type','node')
		return self
	#}}}
	def initialize(self,md): # {{{

		if numpy.all(numpy.isnan(self.precipitation)):
			self.precipitation=numpy.zeros((md.mesh.numberofvertices,1))
			print "      no SMBpdd.precipitation specified: values set as zero"

		return self
	#}}}
	def setdefaultparameters(self): # {{{
		  
		#pdd method not used in default mode
		self.isdelta18o = 0
		self.desfac     = 0.5
		self.s0p        = 0.

		return self
	#}}}
	def checkconsistency(self,md,solution,analyses):    # {{{

		if MasstransportAnalysisEnum() in analyses:
			md = checkfield(md,'fieldname','surfaceforcings.desfac','<=',1,'numel',[1]);
			md = checkfield(md,'fieldname','surfaceforcings.s0p','>=',0,'numel',[1]);
			if not self.isdelta18o:
				md = checkfield(md,'fieldname','surfaceforcings.monthlytemperatures','forcing',1,'NaN',1)
				md = checkfield(md,'fieldname','surfaceforcings.precipitation','forcing',1,'NaN',1)
			else:
				md = checkfield(md,'fieldname','surfaceforcings.delta18o','NaN',1)
				md = checkfield(md,'fieldname','surfaceforcings.delta18o_surface','NaN',1)
				md = checkfield(md,'fieldname','surfaceforcings.temperatures_presentday','size',[md.mesh.numberofvertices+1,12],'NaN',1)
				md = checkfield(md,'fieldname','surfaceforcings.temperatures_lgm','size',[md.mesh.numberofvertices+1,12],'NaN',1)
				md = checkfield(md,'fieldname','surfaceforcings.precipitations_presentday','size',[md.mesh.numberofvertices+1,12],'NaN',1)

		return md
	# }}}
	def marshall(self,md,fid):    # {{{

		yts=365.0*24.0*3600.0

		WriteData(fid,'enum',SurfaceforcingsEnum(),'data',SMBpddEnum(),'format','Integer');

		WriteData(fid,'object',self,'class','surfaceforcings','fieldname','precipitation','format','DoubleMat','mattype',1,'scale',1./yts,'forcinglength',md.mesh.numberofvertices+1)
		WriteData(fid,'object',self,'class','surfaceforcings','fieldname','isdelta18o','format','Boolean')
		WriteData(fid,'object',self,'class','surfaceforcings','fieldname','desfac','format','Double');
		WriteData(fid,'object',self,'class','surfaceforcings','fieldname','s0p','format','Double');
		if self.isdelta18o:
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','temperatures_presentday','format','DoubleMat','mattype',1)
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','temperatures_lgm','format','DoubleMat','mattype',1)
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','precipitations_presentday','format','DoubleMat','mattype',1)
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','delta18o_surface','format','DoubleMat','mattype',1)
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','delta18o','format','DoubleMat','mattype',1)
		else:
			WriteData(fid,'object',self,'class','surfaceforcings','fieldname','monthlytemperatures','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1)
	# }}}
