/*!\file:  thermalstatic.cpp
 * \brief: thermalstatic solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "thermalstatic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;

	/*Model: */
	Model* model=NULL;

	/*Results: */
	DataSet* results=NULL;
	Result* result=NULL;
	
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	
	double* u_g_initial=NULL;
	double* p_g_initial=NULL;
	double  dt;
	Param*  param=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading diagnostic horiz model data:\n");
	CreateFemModel(model->DiagnosticHorizontal(),fid,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	_printf_("\n   reading diagnostic vert model data:\n");
	CreateFemModel(model->DiagnosticVertical(),fid,DiagnosticAnalysisEnum(),VertAnalysisEnum());
	_printf_("\n   reading diagnostic stokes model data:\n");
	CreateFemModel(model->DiagnosticStokes(),fid,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	_printf_("\n   reading diagnostic hutter model data:\n");
	CreateFemModel(model->DiagnosticHutter(),fid,DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	_printf_("\n   reading surface and bed slope computation model data:\n");
	CreateFemModel(model->Slope(),fid,SlopeComputeAnalysisEnum(),NoneAnalysisEnum());
	_printf_("\n   read and create thermal finite element model:\n");
	CreateFemModel(model->Thermal(),fid,ThermalAnalysisEnum(),SteadyAnalysisEnum());
	_printf_("\n   read and create melting finite element model:\n");
	CreateFemModel(model->Melting(),fid,MeltingAnalysisEnum(),SteadyAnalysisEnum());

	_printf_("initialize inputs:\n");
	model->DiagnosticHorizontal()->parameters->FindParam((void*)&u_g_initial,"u_g");
	model->DiagnosticHorizontal()->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	model->Thermal()->parameters->FindParam((void*)&dt,"dt");
	model->Thermal()->parameters->FindParam((void*)&p_g_initial,"p_g");

	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	inputs->Add("pressure",p_g_initial,1,numberofnodes);
	inputs->Add("dt",dt);
	
	/*erase velocities: */
	param=(Param*)model->DiagnosticHorizontal()->parameters->FindParamObject("u_g");
	model->DiagnosticHorizontal()->parameters->DeleteObject((Object*)param);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	model->DiagnosticHorizontal()->parameters->FindParam((void*)&qmu_analysis,"qmu_analysis");
	if(!qmu_analysis){

		/*run diagnostic analysis: */
		_printf_("call computational core:\n");
		thermalstatic_core(results,model,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on thermalstatic core:\n");

		#ifdef _HAVE_DAKOTA_ 
		Qmux(model,inputs,ThermalstaticAnalysisEnum(),NoneAnalysisEnum());
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","thermalstatic");
	results->AddObject(result);

	_printf_("process results:\n");
	ProcessResults(&results,model,ThermalstaticAnalysisEnum());

	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	model->DiagnosticHorizontal()->parameters->FindParam((void*)&waitonlock,"waitonlock");
	if (waitonlock){
		WriteLockFile(lockname);
	}
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 
	

	/*end module: */
	MODULEEND();
	
	/*Free ressources */
	xfree((void**)&u_g_initial);
	xfree((void**)&p_g_initial);

	return 0; //unix success return;
}
