/*!\file:  objectivefunctionC
 * \brief  objective function that returns a misfit, for a certain parameter.
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "diagnostic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

double objectivefunctionC(double search_scalar,OptArgs* optargs){

	int i;  
	
	/*output: */
	double J;
	
	/*parameters: */
	FemModel* femmodel=NULL;
	double* param_g=NULL;
	double* grad_g=NULL;
	ParameterInputs* inputs=NULL;
	int numberofdofspernode;
	int n;

	/*intermediary:*/
	int gsize;
	double* optscal=NULL;
	double* fit=NULL;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	char*   control_type=NULL;
	double* param_g_copy=NULL;
	int     analysis_type;
	int     sub_analysis_type;
	Vec     u_g=NULL;
	double* u_g_double=NULL;
	int     numberofnodes;


	/*Recover parameters: */
	femmodel=optargs->femmodel;
	param_g=optargs->param_g;
	grad_g=optargs->grad_g;
	inputs=optargs->inputs;
	n=optargs->n;

	gsize=femmodel->nodesets->GetGSize();
	femmodel->parameters->FindParam((void*)&optscal,"optscal");
	femmodel->parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	femmodel->parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	femmodel->parameters->FindParam((void*)&control_type,"control_type");
	femmodel->parameters->FindParam((void*)&fit,"fit");
	femmodel->parameters->FindParam((void*)&analysis_type,"analysis_type");
	femmodel->parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	femmodel->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	femmodel->parameters->FindParam((void*)&numberofdofspernode,"numberofdofspernode");

	/*First copy param_g so we don't modify it: */
	param_g_copy=(double*)xmalloc(numberofnodes*sizeof(double));
	memcpy(param_g_copy,param_g,numberofnodes*sizeof(double));

	/*First, update param_g using search_scalar: */
	for(i=0;i<numberofnodes;i++)param_g_copy[i]=param_g_copy[i]+search_scalar*optscal[n]*grad_g[i];

	/*Constrain:*/
	ControlConstrainx(param_g_copy,numberofnodes,mincontrolconstraint,maxcontrolconstraint,control_type);

	/*Add new parameter to inputs: */
	inputs->Add(control_type,param_g_copy,1,numberofnodes);

	//Run diagnostic with updated parameters.
	diagnostic_core_nonlinear(&u_g,NULL,NULL,NULL,femmodel,inputs,DiagnosticAnalysisEnum(),sub_analysis_type);
	VecToMPISerial(&u_g_double,u_g); VecFree(&u_g);
	inputs->Add("velocity",u_g_double,numberofdofspernode,numberofnodes);

	//Compute misfit for this velocity field. 
	inputs->Add("fit",fit[n]);
	Misfitx( &J, femmodel->elements,femmodel->nodes, femmodel->loads, femmodel->materials, 
				inputs,analysis_type,sub_analysis_type);

	/*Free ressources:*/
	xfree((void**)&optscal);
	xfree((void**)&control_type);
	xfree((void**)&param_g_copy);
	xfree((void**)&u_g_double);

	return J;
}
