/*!\file: controlvalidation_core.cpp
 * \brief: core of the control solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

void controlvalidation_core(FemModel* femmodel){

	int         solution_type,n;
	IssmDouble  j0,j;
	IssmDouble  Ialpha,alpha;
	IssmDouble *G = NULL;
	IssmDouble *X = NULL;
	IssmDouble *X0= NULL;

	/*Solution and Adjoint core pointer*/
	void (*solutioncore)(FemModel*) = NULL;
	void (*adjointcore)(FemModel*)  = NULL;

	/*Recover parameters used throughout the solution*/
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->SetParam(false,SaveResultsEnum);

	/*Get initial guess*/
	Vector<IssmDouble> *Xpetsc = NULL;
	GetVectorFromControlInputsx(&Xpetsc,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,"value");
	Xpetsc->GetSize(&n);
	X0 = Xpetsc->ToMPISerial();
	delete Xpetsc;

	/*Allocate Gradient and current vector*/
	G = xNew<double>(n);
	X = xNew<double>(n);

	/*out of solution_type, figure out solution core and adjoint function pointer*/
	CorePointerFromSolutionEnum(&solutioncore,femmodel->parameters,solution_type);
	AdjointCorePointerFromSolutionEnum(&adjointcore,solution_type);

	if(VerboseControl()) _printf0_("   Compute Initial solution\n");
	solutioncore(femmodel);
	if(VerboseControl()) _printf0_("   Compute Adjoint\n");
	adjointcore(femmodel);
	if(VerboseControl()) _printf0_("   Compute Initial cost function\n");
	femmodel->CostFunctionx(&j0,NULL,NULL);
	if(VerboseControl()) _printf0_("   Compute Gradient\n");
	Gradjx(&G,NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
	for(int i=0;i<n;i++) G[i] = -G[i];

	_printf0_("j0 = "<<j0<<"\n");
	for(IssmDouble exponent = 4.; exponent > -4. ; exponent-=0.2){
	//for(IssmDouble exponent = -1.; exponent > -9. ; exponent-=0.2){

		alpha = pow(10.,exponent);

		/*Create new vector*/
		for(int i=0;i<n;i++) X[i] = X0[i] + alpha;

		/*Calculate j(k+alpha delta k) */
		SetControlInputsFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X);
		solutioncore(femmodel);
		femmodel->CostFunctionx(&j,NULL,NULL);

		IssmDouble Den = 0.;
		for(int i=0;i<n;i++) Den += alpha * G[i] * 1.;
		Ialpha = fabs((j - j0)/Den - 1.);

		//_printf0_("alpha = "<<alpha<<" Ialpha = "<<Ialpha<<"\n");
		_printf0_(alpha<<" "<<Ialpha<<"\n");
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(G);
}
