/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

/*Local prototypes*/
bool controlconvergence(IssmDouble J, IssmDouble tol_cm);
IssmDouble objectivefunction(IssmDouble search_scalar,void* optargs);

void control_core(FemModel* femmodel){

	int     i;

	/*parameters: */
	int        num_controls;
	int        nsteps;
	IssmDouble tol_cm;
	int        solution_type;
	bool       isFS;
	bool       dakota_analysis = false;

	int        *control_type   = NULL;
	IssmDouble *maxiter        = NULL;
	IssmDouble *cm_jump        = NULL;

	/*intermediary: */
	IssmDouble search_scalar = 1.;
	OptPars    optpars;

	/*Solution and Adjoint core pointer*/
	void (*solutioncore)(FemModel*) = NULL;
	void (*adjointcore)(FemModel*)  = NULL;

	/*output: */
	IssmDouble* J=NULL;

	/*Recover parameters used throughout the solution*/
	femmodel->parameters->FindParam(&num_controls,InversionNumControlParametersEnum);
	femmodel->parameters->FindParam(&control_type,NULL,InversionControlParametersEnum);
	femmodel->parameters->FindParam(&nsteps,InversionNstepsEnum);
	femmodel->parameters->FindParam(&maxiter,NULL,InversionMaxiterPerStepEnum);
	femmodel->parameters->FindParam(&cm_jump,NULL,InversionStepThresholdEnum);
	femmodel->parameters->FindParam(&tol_cm,InversionCostFunctionThresholdEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&isFS,FlowequationIsFSEnum);
	femmodel->parameters->FindParam(&dakota_analysis,QmuIsdakotaEnum);
	femmodel->parameters->SetParam(false,SaveResultsEnum);

	/*out of solution_type, figure out solution core and adjoint function pointer*/
	CorePointerFromSolutionEnum(&solutioncore,femmodel->parameters,solution_type);
	AdjointCorePointerFromSolutionEnum(&adjointcore,solution_type);

	/*Launch once a complete solution to set up all inputs*/
	if(VerboseControl()) _printf0_("   preparing initial solution\n");
	if(isFS) solutioncore(femmodel);

	/*Initialize cost function: */
	J=xNew<IssmDouble>(nsteps);

	/*Initialize some of the BrentSearch arguments: */
	optpars.xmin=0; optpars.xmax=1;

	/*Start looping: */
	for(int n=0;n<nsteps;n++){

		/*Display info*/
		if(VerboseControl()) _printf0_("\n" << "   control method step " << n+1 << "/" << nsteps << "\n");


		/*In steady state inversion, compute new temperature field now*/
		if(solution_type==SteadystateSolutionEnum) solutioncore(femmodel);

		if(VerboseControl()) _printf0_("   compute adjoint state:\n");
		adjointcore(femmodel);
		gradient_core(femmodel,n,search_scalar==0.);

		if(VerboseControl()) _printf0_("   optimizing along gradient direction\n");
		optpars.maxiter=reCast<int,IssmDouble>(maxiter[n]); optpars.cm_jump=cm_jump[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunction,(void*)femmodel);

		if(VerboseControl()) _printf0_("   updating parameter using optimized search scalar\n"); //true means update save controls
		InputControlUpdatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,search_scalar,true);

		if(controlconvergence(J[n],tol_cm)) break;
	}

	if(VerboseControl()) _printf0_("   preparing final solution\n");
	femmodel->parameters->SetParam(true,SaveResultsEnum);
	solutioncore(femmodel);

	/*some results not computed by steadystate_core or stressbalance_core: */
	if(!dakota_analysis){ //do not save this if we are running the control core from a qmu run!
		femmodel->OutputControlsx(&femmodel->results);

		#ifdef _HAVE_ADOLC_
		IssmPDouble* J_passive=xNew<IssmPDouble>(nsteps);
		for(i=0;i<nsteps;i++) J_passive[i]=reCast<IssmPDouble>(J[i]);
		femmodel->results->AddObject(new GenericExternalResult<IssmPDouble*>(femmodel->results->Size()+1,JEnum,J_passive,nsteps,1,1,0));
		xDelete<IssmPDouble>(J_passive);
		#else
		femmodel->results->AddObject(new GenericExternalResult<IssmPDouble*>(femmodel->results->Size()+1,JEnum,J,nsteps,1,1,0));
		#endif
	}

	/*Free ressources: */
	xDelete<int>(control_type);
	xDelete<IssmDouble>(maxiter);
	xDelete<IssmDouble>(cm_jump);
	xDelete<IssmDouble>(J);
}
bool controlconvergence(IssmDouble J, IssmDouble tol_cm){

	bool converged=false;

	/*Has convergence been reached?*/
	if (!xIsNan<IssmDouble>(tol_cm) && J<tol_cm){
		converged=true;
		if(VerboseConvergence()) _printf0_("      Convergence criterion reached: J = " << J << " < " << tol_cm);
	}

	return converged;
}

IssmDouble objectivefunction(IssmDouble search_scalar,void* optargs){

	/*output: */
	IssmDouble J;

	/*parameters: */
	int        solution_type,analysis_type;
	bool       isFS       = false;
	bool       conserve_loads = true;
	FemModel  *femmodel       = (FemModel*)optargs;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&isFS,FlowequationIsFSEnum);
	femmodel->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	/*set analysis type to compute velocity: */
	switch(solution_type){
		case SteadystateSolutionEnum:
		case StressbalanceSolutionEnum:
			femmodel->SetCurrentConfiguration(StressbalanceAnalysisEnum);
			break;
		case BalancethicknessSolutionEnum:
			femmodel->SetCurrentConfiguration(BalancethicknessAnalysisEnum);
			break;
		case BalancethicknessSoftSolutionEnum:
			femmodel->SetCurrentConfiguration(BalancethicknessAnalysisEnum);
			break;
		case Balancethickness2SolutionEnum:
			femmodel->SetCurrentConfiguration(Balancethickness2AnalysisEnum);
			break;
		default:
			_error_("Solution " << EnumToStringx(solution_type) << " not implemented yet");
	}

	/*update parameter according to scalar: */ //false means: do not save control
	InputControlUpdatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,search_scalar,false);

	/*Run stressbalance with updated inputs: */
	if (solution_type==SteadystateSolutionEnum){
		stressbalance_core(femmodel);	//We need a 3D velocity!! (vz is required for the next thermal run)
	}
	else if (solution_type==StressbalanceSolutionEnum){
		solutionsequence_nonlinear(femmodel,conserve_loads); 
	}
	else if (solution_type==BalancethicknessSolutionEnum){
		solutionsequence_linear(femmodel); 
	}
	else if (solution_type==Balancethickness2SolutionEnum){
		solutionsequence_linear(femmodel); 
	}
	else if (solution_type==BalancethicknessSoftSolutionEnum){
		/*Don't do anything*/
	}
	else{
		_error_("Solution " << EnumToStringx(solution_type) << " not implemented yet");
	}

	/*Compute misfit for this velocity field.*/
	femmodel->CostFunctionx(&J,NULL,NULL);

	/*Free ressources:*/
	return J;
}
