#include "./HydrologyDCInefficientAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../classes/Node.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  HydrologyDCInefficientAnalysis::DofsPerNode(int** doflist,int domaintype,int approximation){/*{{{*/
	return 1;
}/*}}}*/
void HydrologyDCInefficientAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	int         hydrology_model;
	int         sedimentlimit_flag;
	int         transfer_flag;
	int         penalty_lock;
	int         hydro_maxiter;
	bool        isefficientlayer;
	IssmDouble  sedimentlimit;
	IssmDouble  penalty_factor;
	IssmDouble  leakagefactor;
	IssmDouble  rel_tol;

	/*retrieve some parameters: */
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	iomodel->FetchData(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	iomodel->FetchData(&sedimentlimit_flag,HydrologydcSedimentlimitFlagEnum);
	iomodel->FetchData(&transfer_flag,HydrologydcTransferFlagEnum);
	iomodel->FetchData(&penalty_factor,HydrologydcPenaltyFactorEnum);
	iomodel->FetchData(&rel_tol,HydrologydcRelTolEnum);
	iomodel->FetchData(&penalty_lock,HydrologydcPenaltyLockEnum);
	iomodel->FetchData(&hydro_maxiter,HydrologydcMaxIterEnum);

	if(sedimentlimit_flag==1){
		iomodel->FetchData(&sedimentlimit,HydrologydcSedimentlimitEnum);
		parameters->AddObject(new DoubleParam(HydrologydcSedimentlimitEnum,sedimentlimit));
	}

	if(transfer_flag==1){
		iomodel->FetchData(&leakagefactor,HydrologydcLeakageFactorEnum);
		parameters->AddObject(new DoubleParam(HydrologydcLeakageFactorEnum,leakagefactor));
	}

	parameters->AddObject(new DoubleParam(HydrologydcPenaltyFactorEnum,penalty_factor));
	parameters->AddObject(new IntParam(HydrologyModelEnum,hydrology_model));
	parameters->AddObject(new BoolParam(HydrologydcIsefficientlayerEnum,isefficientlayer));
	parameters->AddObject(new IntParam(HydrologydcSedimentlimitFlagEnum,sedimentlimit_flag));
	parameters->AddObject(new IntParam(HydrologydcTransferFlagEnum,transfer_flag));
	parameters->AddObject(new DoubleParam(HydrologydcRelTolEnum,rel_tol));
	parameters->AddObject(new IntParam(HydrologydcPenaltyLockEnum,penalty_lock));
	parameters->AddObject(new IntParam(HydrologydcMaxIterEnum,hydro_maxiter));

}/*}}}*/
void HydrologyDCInefficientAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	bool   isefficientlayer;
	int    hydrology_model;

	/*Fetch data needed: */
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	/*Fetch data needed: */
	iomodel->Constant(&isefficientlayer,HydrologydcIsefficientlayerEnum);

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	iomodel->FetchDataToInput(elements,ThicknessEnum);
	iomodel->FetchDataToInput(elements,BaseEnum);
	iomodel->FetchDataToInput(elements,MaskIceLevelsetEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsMeltingRateEnum);
	iomodel->FetchDataToInput(elements,HydrologydcBasalMoulinInputEnum);
	iomodel->FetchDataToInput(elements,SedimentHeadEnum);
	iomodel->FetchDataToInput(elements,HydrologydcSedimentTransmitivityEnum);
	if(iomodel->domaintype!=Domain2DhorizontalEnum){
		iomodel->FetchDataToInput(elements,MeshVertexonbaseEnum);
		iomodel->FetchDataToInput(elements,MeshVertexonsurfaceEnum);
	}

	if(isefficientlayer)iomodel->FetchDataToInput(elements,HydrologydcMaskEplactiveNodeEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	/*Fetch parameters: */
	int  hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	if(iomodel->domaintype!=Domain2DhorizontalEnum) iomodel->FetchData(2,MeshVertexonbaseEnum,MeshVertexonsurfaceEnum);
	::CreateNodes(nodes,iomodel,HydrologyDCInefficientAnalysisEnum,P1Enum);
	iomodel->DeleteData(2,MeshVertexonbaseEnum,MeshVertexonsurfaceEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	/*retrieve some parameters: */
	int hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);
	if(hydrology_model!=HydrologydcEnum) return;

	IoModelToConstraintsx(constraints,iomodel,HydrologydcSpcsedimentHeadEnum,HydrologyDCInefficientAnalysisEnum,P1Enum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	/*Fetch parameters: */
	int hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);
	if(hydrology_model!=HydrologydcEnum) return;

	if(iomodel->domaintype==Domain3DEnum) iomodel->FetchData(1,MeshVertexonbaseEnum);

	//create penalties for nodes: no node can have water above the max
	CreateSingleNodeToElementConnectivity(iomodel);
	for(int i=0;i<iomodel->numberofvertices;i++){
		if (iomodel->domaintype!=Domain3DEnum){
			/*keep only this partition's nodes:*/
			if(iomodel->my_vertices[i]){
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,HydrologyDCInefficientAnalysisEnum));
			}
		}
		else if(reCast<int>(iomodel->Data(MeshVertexonbaseEnum)[i])){
			if(iomodel->my_vertices[i]){
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,HydrologyDCInefficientAnalysisEnum));
			}	
		}
	}
	iomodel->DeleteData(1,MeshVertexonbaseEnum);
}/*}}}*/

/*Finite Element Analysis*/
void           HydrologyDCInefficientAnalysis::Core(FemModel* femmodel){/*{{{*/
	_error_("not implemented");
}/*}}}*/
ElementVector* HydrologyDCInefficientAnalysis::CreateDVector(Element* element){/*{{{*/
	/*Default, return NULL*/
	return NULL;
}/*}}}*/
ElementMatrix* HydrologyDCInefficientAnalysis::CreateJacobianMatrix(Element* element){/*{{{*/
_error_("Not implemented");
}/*}}}*/
ElementMatrix* HydrologyDCInefficientAnalysis::CreateKMatrix(Element* element){/*{{{*/

	/*Intermediaries*/
	int      domaintype;
	Element* basalelement;

	/*Get basal element*/
	element->FindParam(&domaintype,DomainTypeEnum);
	switch(domaintype){
		case Domain2DhorizontalEnum:
			basalelement = element;
			break;
		case Domain3DEnum:
			if(!element->IsOnBase()) return NULL;
			basalelement = element->SpawnBasalElement();
			break;
		default: _error_("mesh "<<EnumToStringx(domaintype)<<" not supported yet");
	}

	/*Intermediaries */
	bool        active_element,isefficientlayer;
	IssmDouble  D_scalar,Jdet,dt;
	IssmDouble  sediment_transmitivity;
	IssmDouble  transfer;
	IssmDouble *xyz_list  = NULL;

	/*Define transfer related variables*/
	Input* active_element_input =NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = basalelement->GetNumberOfNodes();

	/*Initialize Element vector*/
	ElementMatrix* Ke     = basalelement->NewElementMatrix();
	IssmDouble*    B      = xNew<IssmDouble>(2*numnodes);
	IssmDouble*    basis  = xNew<IssmDouble>(numnodes);
	IssmDouble     D[2][2]= {0.};

	/*Retrieve all inputs and parameters*/
	basalelement ->GetVerticesCoordinates(&xyz_list);
	basalelement ->FindParam(&dt,TimesteppingTimeStepEnum);
	basalelement ->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);

	Input* epl_thick_input   = basalelement->GetInput(HydrologydcEplThicknessEnum);
	Input* sed_head_input    = basalelement->GetInput(SedimentHeadEnum);
	Input* epl_head_input    = basalelement->GetInput(EplHeadEnum);
	Input* thickness_input   = basalelement->GetInput(ThicknessEnum);
	Input* base_input         = basalelement->GetInput(BaseEnum);
	Input* SedTrans_input    = basalelement->GetInput(HydrologydcSedimentTransmitivityEnum); _assert_(SedTrans_input);
	IssmDouble sediment_storing = SedimentStoring(basalelement);
	/*Transfer related Inputs*/
	if(isefficientlayer){
		active_element_input = basalelement->GetInput(HydrologydcMaskEplactiveEltEnum); _assert_(active_element_input);
	}
	
	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=basalelement->NewGauss(2);
	for(int ig=gauss -> begin();ig<gauss->end();ig++){
		gauss          -> GaussPoint(ig);
		basalelement   -> JacobianDeterminant(&Jdet,xyz_list,gauss);
		SedTrans_input -> GetInputValue(&sediment_transmitivity,gauss);
		/*Diffusivity*/
		D_scalar=sediment_transmitivity*gauss->weight*Jdet;
		if(dt!=0.) D_scalar=D_scalar*dt;
		D[0][0]=D_scalar;
		D[1][1]=D_scalar;
		GetB(B,basalelement,xyz_list,gauss); 
		TripleMultiply(B,2,numnodes,1,
									 &D[0][0],2,2,0,
									 B,2,numnodes,0,
									 &Ke->values[0],1);

		/*Transient*/
		if(dt!=0.){
			basalelement->NodalFunctions(&basis[0],gauss);
			D_scalar=sediment_storing*gauss->weight*Jdet;
			TripleMultiply(basis,numnodes,1,0,
										 &D_scalar,1,1,0,
										 basis,1,numnodes,0,
										 &Ke->values[0],1);
			
			/*Transfer EPL part*/
			if(isefficientlayer){
				active_element_input->GetInputValue(&active_element);
				if(active_element){
					transfer=GetHydrologyKMatrixTransfer(basalelement,gauss,epl_thick_input,sed_head_input,epl_head_input,SedTrans_input,thickness_input,base_input);
					basalelement->NodalFunctions(&basis[0],gauss);
					D_scalar=transfer*gauss->weight*Jdet*dt;
					TripleMultiply(basis,numnodes,1,0,
												 &D_scalar,1,1,0,
												 basis,1,numnodes,0,
												 &Ke->values[0],1);
				}
			}
		}
	}
	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(basis);
	delete gauss;
	if(domaintype!=Domain2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
	return Ke;
}/*}}}*/
ElementVector* HydrologyDCInefficientAnalysis::CreatePVector(Element* element){/*{{{*/

	/*Intermediaries*/
	int      domaintype;
	Element* basalelement;

	/*Get basal element*/
	element->FindParam(&domaintype,DomainTypeEnum);
	switch(domaintype){
		case Domain2DhorizontalEnum:
			basalelement = element;
			break;
		case Domain3DEnum:
			if(!element->IsOnBase()) return NULL;
			basalelement = element->SpawnBasalElement();
			break;
		default: _error_("mesh "<<EnumToStringx(domaintype)<<" not supported yet");
	}

	/*Intermediaries */
	bool       active_element,isefficientlayer;
	IssmDouble dt,scalar;
	IssmDouble water_head;
	IssmDouble water_load,transfer;
	IssmDouble Jdet;

	IssmDouble *xyz_list             = NULL;
	Input*      old_wh_input         = NULL;
	Input*      active_element_input = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = basalelement->GetNumberOfNodes();

	/*Initialize Element vector*/
	ElementVector* pe    = basalelement->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	basalelement->GetVerticesCoordinates(&xyz_list);
	basalelement->FindParam(&dt,TimesteppingTimeStepEnum);
	basalelement->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);

	Input* epl_thick_input   = basalelement->GetInput(HydrologydcEplThicknessEnum);
	Input* sed_head_input    = basalelement->GetInput(SedimentHeadEnum);
	Input* epl_head_input    = basalelement->GetInput(EplHeadEnum);
	Input* sed_trans_input   = basalelement->GetInput(HydrologydcSedimentTransmitivityEnum);
	Input* thickness_input   = basalelement->GetInput(ThicknessEnum);
	Input* base_input        = basalelement->GetInput(BaseEnum);
	Input* water_input       = basalelement->GetInput(BasalforcingsMeltingRateEnum);    _assert_(water_input);
	if(dt!= 0.){old_wh_input = basalelement->GetInput(SedimentHeadOldEnum);             _assert_(old_wh_input);}

	IssmDouble sediment_storing    = SedimentStoring(basalelement);

	/*Transfer related Inputs*/
	if(isefficientlayer){
		active_element_input = basalelement->GetInput(HydrologydcMaskEplactiveEltEnum); _assert_(active_element_input);
	}

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=basalelement->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);
	
		basalelement->JacobianDeterminant(&Jdet,xyz_list,gauss);
		basalelement->NodalFunctions(basis,gauss);

		/*Loading term*/
		water_input->GetInputValue(&water_load,gauss);

		scalar = Jdet*gauss->weight*(water_load);

		if(dt!=0.) scalar = scalar*dt;
		for(int i=0;i<numnodes;i++){
			pe->values[i]+=scalar*basis[i];
		}
			
		/*Transient and transfer terms*/
		if(dt!=0.){
			old_wh_input    ->GetInputValue(&water_head,gauss);
			if(isefficientlayer){
				/*Dealing with the sediment part of the transfer term*/
				active_element_input->GetInputValue(&active_element);
				if(active_element){
					transfer=GetHydrologyPVectorTransfer(basalelement,gauss,epl_thick_input,sed_head_input,epl_head_input,sed_trans_input,thickness_input,base_input);
				}
				else{
					transfer=0.0;
				}
				scalar = Jdet*gauss->weight*((water_head*sediment_storing)+(dt*transfer));
				for(int i=0;i<numnodes;i++)pe->values[i]+=scalar*basis[i];
			}
			else{
				scalar = Jdet*gauss->weight*(water_head*sediment_storing);
				for(int i=0;i<numnodes;i++)pe->values[i]+=scalar*basis[i];
			}
		}
	}
	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	delete gauss;
	if(domaintype!=Domain2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
	return pe;
}/*}}}*/
void HydrologyDCInefficientAnalysis::GetB(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For node i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dN/dx ]
	 *          [ dN/dy ]
	 * where N is the finiteelement function for node i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numnodes)
	 */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* dbasis=xNew<IssmDouble>(2*numnodes);
	element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[numnodes*0+i] = dbasis[0*numnodes+i];
		B[numnodes*1+i] = dbasis[1*numnodes+i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(dbasis);
}/*}}}*/
void HydrologyDCInefficientAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	element->GetSolutionFromInputsOneDof(solution,SedimentHeadEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	int        domaintype;
	bool       converged;
	int*       doflist=NULL;
	Element*   basalelement=NULL;

	element->FindParam(&domaintype,DomainTypeEnum);
	if(domaintype!=Domain2DhorizontalEnum){
		if(!element->IsOnBase()) return;
		basalelement=element->SpawnBasalElement();
	}
	else{
		basalelement = element;
	}

	/*Fetch number of nodes for this finite element*/
	int numnodes = basalelement->GetNumberOfNodes();

	/*Fetch dof list and allocate solution vector*/
	basalelement->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values   = xNew<IssmDouble>(numnodes);
	IssmDouble* pressure = xNew<IssmDouble>(numnodes);
	IssmDouble* residual = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numnodes;i++){
		values[i] =solution[doflist[i]];
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
	}

	/*If converged keep the residual in mind, also compute effective pressure*/
	basalelement->GetInputValue(&converged,ConvergedEnum);

	/*Get inputs*/
	if(converged){
		IssmDouble  penalty_factor,kmax,kappa,h_max;
		IssmDouble* thickness = xNew<IssmDouble>(numnodes);
		IssmDouble* base      = xNew<IssmDouble>(numnodes);

		basalelement->FindParam(&kmax,HydrologySedimentKmaxEnum);
		basalelement->FindParam(&penalty_factor,HydrologydcPenaltyFactorEnum);
		IssmDouble rho_freshwater = basalelement->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
		IssmDouble rho_ice        = basalelement->GetMaterialParameter(MaterialsRhoIceEnum);
		IssmDouble g              = basalelement->GetMaterialParameter(ConstantsGEnum);
		
		basalelement->GetInputListOnVertices(thickness,ThicknessEnum);
		basalelement->GetInputListOnVertices(base,BaseEnum);

		kappa=kmax*pow(10.,penalty_factor);

		for(int i=0;i<numnodes;i++){

			GetHydrologyDCInefficientHmax(&h_max,basalelement,basalelement->GetNode(i));
			if(values[i]>h_max) {
				residual[i] = kappa*(values[i]-h_max);
			}
			else{
				residual[i] = 0.;
			}
			pressure[i]=(rho_ice*g*thickness[i])-(rho_freshwater*g*(min(h_max,values[i])-base[i]));
		}
		xDelete<IssmDouble>(thickness);
		xDelete<IssmDouble>(base);
	}

	/*Add input to the element: */
	element->AddBasalInput(SedimentHeadEnum,values,P1Enum);
	element->AddBasalInput(SedimentHeadResidualEnum,residual,P1Enum);
	element->AddBasalInput(EffectivePressureEnum,pressure,P1Enum);

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(residual);
	xDelete<IssmDouble>(pressure);
	xDelete<int>(doflist);
	if(domaintype!=Domain2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
}/*}}}*/
void HydrologyDCInefficientAnalysis::UpdateConstraints(FemModel* femmodel){/*{{{*/
	/*Default, do nothing*/
	return;
}/*}}}*/

/*Intermediaries*/
IssmDouble HydrologyDCInefficientAnalysis::SedimentStoring(Element* element){/*{{{*/
	IssmDouble rho_freshwater           = element->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
	IssmDouble g                        = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble sediment_porosity        = element->GetMaterialParameter(HydrologydcSedimentPorosityEnum);
	IssmDouble sediment_thickness       = element->GetMaterialParameter(HydrologydcSedimentThicknessEnum);
	IssmDouble sediment_compressibility = element->GetMaterialParameter(HydrologydcSedimentCompressibilityEnum);
	IssmDouble water_compressibility    = element->GetMaterialParameter(HydrologydcWaterCompressibilityEnum);
	return rho_freshwater*g*sediment_porosity*sediment_thickness*(water_compressibility+(sediment_compressibility/sediment_porosity));		 
}/*}}}*/

IssmDouble HydrologyDCInefficientAnalysis::EplSpecificStoring(Element* element){/*{{{*/
	IssmDouble rho_freshwater        = element->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
	IssmDouble g                     = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble epl_porosity          = element->GetMaterialParameter(HydrologydcEplPorosityEnum);
	IssmDouble epl_compressibility   = element->GetMaterialParameter(HydrologydcEplCompressibilityEnum);
	IssmDouble water_compressibility = element->GetMaterialParameter(HydrologydcWaterCompressibilityEnum);
	return rho_freshwater*g*epl_porosity*(water_compressibility+(epl_compressibility/epl_porosity));		 
}/*}}}*/

IssmDouble HydrologyDCInefficientAnalysis::GetHydrologyDCInefficientHmax(Element* element, Gauss* gauss, Input* thick_input, Input* base_input){/*{{{*/
	int        hmax_flag;
	IssmDouble h_max;
	IssmDouble rho_ice,rho_water;
	IssmDouble thickness,bed;
	/*Get the flag to the limitation method*/
	element->FindParam(&hmax_flag,HydrologydcSedimentlimitFlagEnum);
	
	/*Switch between the different cases*/
	switch(hmax_flag){
	case 0:
		h_max=1.0e+10;
		break;
	case 1:
		element->FindParam(&h_max,HydrologydcSedimentlimitEnum);
		break;
	case 2:
	
		rho_water = element->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
		rho_ice   = element->GetMaterialParameter(MaterialsRhoIceEnum);

		_assert_(thick_input);
		_assert_(base_input);

		/*Compute max*/
		thick_input->GetInputValue(&thickness,gauss);
		base_input->GetInputValue(&bed,gauss);
		h_max=((rho_ice*thickness)/rho_water)+bed;
		break;
	case 3:
		_error_("Using normal stress  not supported yet");
		break;
	default:
		_error_("no case higher than 3 for SedimentlimitFlag");
	}
	return h_max;
}/*}}}*/
void  HydrologyDCInefficientAnalysis::GetHydrologyDCInefficientHmax(IssmDouble* ph_max,Element* element, Node* innode){/*{{{*/
	
	int        hmax_flag;
	IssmDouble h_max;
	IssmDouble rho_ice,rho_water;
	IssmDouble thickness,bed;
	/*Get the flag to the limitation method*/
	element->FindParam(&hmax_flag,HydrologydcSedimentlimitFlagEnum);
	
	/*Switch between the different cases*/
	switch(hmax_flag){
	case 0:
		h_max=1.0e+10;
		break;
	case 1:
		element->FindParam(&h_max,HydrologydcSedimentlimitEnum);
		break;
	case 2:
		/*Compute max*/
		rho_water = element->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
		rho_ice   = element->GetMaterialParameter(MaterialsRhoIceEnum);
		element->GetInputValue(&thickness,innode,ThicknessEnum);
		element->GetInputValue(&bed,innode,BaseEnum);
		h_max=((rho_ice*thickness)/rho_water)+bed;
		break;
	case 3:
		_error_("Using normal stress  not supported yet");
		break;
	default:
		_error_("no case higher than 3 for SedimentlimitFlag");
	}
	/*Assign output pointer*/
	*ph_max=h_max;
}
/*}}}*/
IssmDouble HydrologyDCInefficientAnalysis::GetHydrologyKMatrixTransfer(Element* element, Gauss* gauss, Input* epl_thick_input, Input* sed_head_input, Input* epl_head_input, Input* sed_trans_input, Input* thickness_input, Input* base_input){/*{{{*/

	int transfermethod;
	IssmDouble epl_thickness;
	IssmDouble epl_head,sed_head;
	IssmDouble sediment_transmitivity;
	IssmDouble leakage,h_max,transfer;

	IssmDouble sediment_thickness  = element->GetMaterialParameter(HydrologydcSedimentThicknessEnum);
	IssmDouble sediment_storing    = SedimentStoring(element);
 	IssmDouble epl_specificstoring = EplSpecificStoring(element);		

	element->FindParam(&transfermethod,HydrologydcTransferFlagEnum);
	/*Switch between the different transfer methods cases*/
	switch(transfermethod){
	case 0:
		/*Just keepping the transfer to zero*/
		transfer=0.0;
		break;
	case 1:
		
		_assert_(epl_thick_input);
		_assert_(sed_head_input);
		_assert_(epl_head_input);
		_assert_(sed_trans_input);
		_assert_(thickness_input);
		_assert_(base_input);

		epl_thick_input->GetInputValue(&epl_thickness,gauss);
		sed_head_input->GetInputValue(&sed_head,gauss);
		epl_head_input->GetInputValue(&epl_head,gauss);
		sed_trans_input->GetInputValue(&sediment_transmitivity,gauss);
		element->FindParam(&leakage,HydrologydcLeakageFactorEnum);
		transfer=(sediment_transmitivity)/(sediment_thickness*leakage);
		/* if(epl_head>sed_head){ */
		/* 	h_max=GetHydrologyDCInefficientHmax(element,gauss,thickness_input,base_input); */
		/* 	if(sed_head>=h_max){ */
		/* 		transfer=0.0; */
		/* 	} */
		/* 	else{ */
		/* 		transfer=(sediment_transmitivity)/(sediment_thickness*leakage); */
		/* 		//transfer=(epl_specificstoring*epl_thickness*sediment_transmitivity)/(sediment_thickness*leakage); */
		/* 	} */
		/* } */
		/* else{ */
		/* 	transfer=(sediment_transmitivity)/(sediment_thickness*leakage); */
		/* 	//transfer=(sediment_storing*sediment_transmitivity)/(sediment_thickness*leakage); */
		/* } */
		break;
	default:
		_error_("no case higher than 1 for the Transfer method");
	}
	
	return transfer;
}/*}}}*/

IssmDouble HydrologyDCInefficientAnalysis::GetHydrologyPVectorTransfer(Element* element, Gauss* gauss, Input* epl_thick_input, Input* sed_head_input, Input* epl_head_input, Input* sed_trans_input, Input* thickness_input, Input* base_input){/*{{{*/

	int transfermethod;
	IssmDouble epl_thickness;
	IssmDouble epl_head,sediment_head;
	IssmDouble sediment_transmitivity;
	IssmDouble leakage,h_max,transfer;

	IssmDouble sediment_thickness = element->GetMaterialParameter(HydrologydcSedimentThicknessEnum);
	IssmDouble sediment_storing   = SedimentStoring(element);
 	IssmDouble epl_specificstoring = EplSpecificStoring(element);		

	element->FindParam(&transfermethod,HydrologydcTransferFlagEnum);
	/*Switch between the different transfer methods cases*/
	switch(transfermethod){
	case 0:
		/*Just keepping the transfer to zero*/
		transfer=0.0;
		break;
	case 1:
		
		_assert_(epl_thick_input);
		_assert_(sed_head_input);
		_assert_(epl_head_input);
		_assert_(sed_trans_input);
		_assert_(thickness_input);
		_assert_(base_input);

		epl_thick_input->GetInputValue(&epl_thickness,gauss);
		sed_head_input->GetInputValue(&sediment_head,gauss);
		epl_head_input->GetInputValue(&epl_head,gauss);
		sed_trans_input->GetInputValue(&sediment_transmitivity,gauss);

		element->FindParam(&leakage,HydrologydcLeakageFactorEnum);
 		transfer=(sediment_transmitivity*epl_head)/(sediment_thickness*leakage);
		/* if(epl_head>sediment_head){ */
		/* 	h_max=GetHydrologyDCInefficientHmax(element,gauss,thickness_input,base_input); */
 		/* 	if(sediment_head>=h_max){ */
		/* 		transfer=0.0; */
		/* 	} */
		/* 	else{ */
		/* 		transfer=(sediment_transmitivity*epl_head)/(sediment_thickness*leakage); */
		/* 		//transfer=(epl_specificstoring*epl_thickness*sediment_transmitivity*epl_head)/(sediment_thickness*leakage); */
		/* 	} */
		/* } */
		/* else{ */
		/* 	transfer=(sediment_transmitivity*epl_head)/(sediment_thickness*leakage); */
		/* 	//transfer=(sediment_storing*sediment_transmitivity*epl_head)/(sediment_thickness*leakage); */
		/* } */
		break;
	default:
		_error_("no case higher than 1 for the Transfer method");
	}
	return transfer;
}/*}}}*/

void HydrologyDCInefficientAnalysis::ElementizeEplMask(FemModel* femmodel){/*{{{*/

	bool     element_active;
	Element* element=NULL;

	for(int i=0;i<femmodel->elements->Size();i++){
		element=dynamic_cast<Element*>(femmodel->elements->GetObjectByOffset(i));

		int numvertices = element->GetNumberOfVertices();
		Input* node_mask_input = element->GetInput(HydrologydcMaskEplactiveNodeEnum);
		if(node_mask_input->Max()>0.) {
			element_active = true;
		}
		else{
			element_active = false;
		}
		element->AddInput(new BoolInput(HydrologydcMaskEplactiveEltEnum,element_active));
	}
}/*}}}*/

