%TRANSIENT class definition
%
%   Usage:
%      transient=transient();

classdef transient
	properties (SetAccess=public) 
		ismasstransport   = 0;
		isstressbalance   = 0;
		isthermal         = 0;
		isgroundingline   = 0;
		isgia             = 0;
		isdamageevolution = 0;
		islevelset        = 0;
		ishydrology       = 0;
		requested_outputs = {};
	end
	methods
        function createxml(obj,fid) % {{{
            fprintf(fid, '<!-- transient -->\n');            
                    
            % transient solution parameters
            fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="transient solution parameters">','<section name="transient" />');                    
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="ismasstransport" type="',class(obj.ismasstransport),'" default="',convert2str(obj.ismasstransport),'">','     <section name="transient" />','     <help> indicates whether a masstransport solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isstressbalance" type="',class(obj.isstressbalance),'" default="',convert2str(obj.isstressbalance),'">','     <section name="transient" />','     <help> indicates whether a stressbalance solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isthermal" type="',class(obj.isthermal),'" default="',convert2str(obj.isthermal),'">','     <section name="transient" />','     <help> indicates whether a thermal solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isgroundingline" type="',class(obj.isgroundingline),'" default="',convert2str(obj.isgroundingline),'">','     <section name="transient" />','     <help> indicates whether a groundingline migration is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isgia" type="',class(obj.isgia),'" default="',convert2str(obj.isgia),'">','     <section name="transient" />','     <help> indicates whether a postglacial rebound model is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isdamageevolution" type="',class(obj.isdamageevolution),'" default="',convert2str(obj.isdamageevolution),'">','     <section name="transient" />','     <help> indicates whether damage evolution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="islevelset" type="',class(obj.islevelset),'" default="',convert2str(obj.islevelset),'">','     <section name="transient" />','     <help> LEVEL SET DESCRIPTION...  </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="requested_outputs" type="',class(obj.requested_outputs),'" default="',convert2str(obj.requested_outputs),'">','     <section name="transient" />','     <help> list of additional outputs requested </help>','  </parameter>');
            fprintf(fid,'%s\n%s\n','</frame>');    
        
        end % }}}
		function obj = transient(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = deactivateall(obj) % {{{

			%full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
			obj.ismasstransport = 0;
			obj.isstressbalance = 0;
			obj.isthermal       = 0;
			obj.isgroundingline = 0;
			obj.isgia           = 0;
			obj.isdamageevolution = 0;
			obj.islevelset      = 0;
			obj.ishydrology     = 0;

			%default output
			obj.requested_outputs={};
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
			obj.ismasstransport = 1;
			obj.isstressbalance = 1;
			obj.isthermal       = 1;
			obj.isgroundingline = 0;
			obj.isgia           = 0;
			obj.isdamageevolution = 0;
			obj.islevelset      = 0;
			obj.ishydrology     = 0;

			%default output
			obj.requested_outputs={'default'};
		end % }}}
		function list = defaultoutputs(self,md) % {{{
			if(self.ismasstransport)
				list = {'SurfaceforcingsMassBalance'};
			else
				list = {};
			end
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if solution~=TransientSolutionEnum(), return; end

			md = checkfield(md,'fieldname','transient.ismasstransport','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isstressbalance','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isthermal','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgroundingline','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgia','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isdamageevolution','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.islevelset','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.ishydrology','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.requested_outputs','stringrow',1);

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   transient solution parameters:'));

			fielddisplay(obj,'ismasstransport','indicates whether a masstransport solution is used in the transient');
			fielddisplay(obj,'isstressbalance','indicates whether a stressbalance solution is used in the transient');
			fielddisplay(obj,'isthermal','indicates whether a thermal solution is used in the transient');
			fielddisplay(obj,'isgroundingline','indicates whether a groundingline migration is used in the transient');
			fielddisplay(obj,'isgia','indicates whether a postglacial rebound model is used in the transient');
			fielddisplay(obj,'isdamageevolution','indicates whether damage evolution is used in the transient');
			fielddisplay(obj,'islevelset','LEVEL SET DESCRIPTION...');
			fielddisplay(obj,'ishydrology','indicates whether an hydrology model is used');
			fielddisplay(obj,'requested_outputs','list of additional outputs requested');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','ismasstransport','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isstressbalance','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isthermal','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isgroundingline','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isgia','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isdamageevolution','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','ishydrology','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','islevelset','format','Boolean');

			%process requested outputs
			outputs = obj.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(obj,md)]; %add defaults
			end
			WriteData(fid,'data',outputs,'enum',TransientRequestedOutputsEnum,'format','StringArray');
		end % }}}
	end
end
