%SETTINGS class definition
%
%   Usage:
%      settings=settings();

classdef settings
	properties (SetAccess=public) 
		results_on_nodes    = 0;
		io_gather           = 0;
		lowmem              = 0;
		output_frequency    = 0;
		waitonlock          = 0;
		upload_server       = '';
		upload_path         = '';
		upload_login        = '';
		upload_port         = 0;
		upload_filename     = '';

	end
	methods
        function createxml(obj,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- settings -->');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="results_on_nodes" type="',      class(obj.results_on_nodes),'" default="',        convert2str(obj.results_on_nodes),'">',	'     <section name="settings" />','     <help> results are output for all the nodes of each element </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="io_gather" type="',          class(obj.io_gather),'" default="',           convert2str(obj.io_gather),'">',     '     <section name="settings" />','     <help> I/O gathering strategy for result outputs (default 1) </help>','</parameter>');
            
            % lowmen drop-down (0 or 1)
            fprintf(fid,'%s\n%s\n%s\n%s\n',	'<parameter key ="lowmem" type="alternative" optional="true">',	'     <section name="settings" />','     <help> is the memory limited ? (0 or 1) </help>');
            fprintf(fid,'%s\n','       <option value="0" type="string" default="true"> </option>');
            fprintf(fid,'%s\n%s\n','       <option value="1" type="string" default="false"> </option>','</parameter>');

            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="output_frequency" type="',          class(obj.output_frequency),'" default="',           convert2str(obj.output_frequency),'">',	'     <section name="settings" />','     <help> frequency at which results are saved in all solutions with multiple time_steps </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="waitonlock" type="',  	class(obj.waitonlock),'" default="',      convert2str(obj.waitonlock),'">',     '     <section name="settings" />','     <help> maximum number of minutes to wait for batch results (NaN to deactivate) </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_server" type="',         class(obj.upload_server),'" default="',          convert2str(obj.upload_server),'">',	'     <section name="settings" />','     <help> server hostname where model should be uploaded </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_path" type="',             class(obj.upload_path),'" default="',              convert2str(obj.upload_path),'">',     '     <section name="settings" />','     <help> path on server where model should be uploaded </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_login" type="',        class(obj.upload_login),'" default="',         convert2str(obj.upload_login),'">',	'     <section name="settings" />','     <help> server login </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_port" type="',    	class(obj.upload_port),'" default="',      convert2str(obj.upload_port),'">',     '     <section name="settings" />','     <help> port login (default is 0) </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n',    	'<parameter key ="upload_filename" type="',         class(obj.upload_filename),'" default="',          convert2str(obj.upload_filename),'">',	'     <section name="settings" />','     <help> unique id generated when uploading the file to server </help>','</parameter>');
             
        end % }}}
		function obj = settings(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%are we short in memory ? (0 faster but requires more memory)
			obj.lowmem=0;

			%i/o:
			obj.io_gather=1;

			%results frequency by default every step
			obj.output_frequency=1;

			%this option can be activated to load automatically the results
			%onto the model after a parallel run by waiting for the lock file
			%N minutes that is generated once the solution has converged
			%0 to deactivate
			obj.waitonlock=Inf;

			%upload options: 
			upload_port         = 0;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','settings.results_on_nodes','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.io_gather','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.lowmem','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.output_frequency','numel',[1],'>=',1);
			md = checkfield(md,'fieldname','settings.waitonlock','numel',[1]);

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   general settings parameters:'));

			fielddisplay(obj,'results_on_nodes','results are output for all the nodes of each element');
			fielddisplay(obj,'io_gather','I/O gathering strategy for result outputs (default 1)');
			fielddisplay(obj,'lowmem','is the memory limited ? (0 or 1)');
			fielddisplay(obj,'output_frequency','frequency at which results are saved in all solutions with multiple time_steps');
			fielddisplay(obj,'waitonlock','maximum number of minutes to wait for batch results (NaN to deactivate)');
			fielddisplay(obj,'upload_server','server hostname where model should be uploaded');
			fielddisplay(obj,'upload_path','path on server where model should be uploaded');
			fielddisplay(obj,'upload_login','server login');
			fielddisplay(obj,'upload_port','port login (default is 0)');
			fielddisplay(obj,'upload_filename','unique id generated when uploading the file to server');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','results_on_nodes','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','io_gather','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','lowmem','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','output_frequency','format','Integer');
			if obj.waitonlock>0,
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',true,'format','Boolean');
			else
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',false,'format','Boolean');
			end
		end % }}}
	end
end
