%HYDROLOGYSHREVE class definition
%
%   Usage:
%      hydrologyshreve=hydrologyshreve();

classdef hydrologyshreve
	properties (SetAccess=public) 
		spcwatercolumn = NaN;
		n              = 0;
		CR             = 0;
		p              = 0;
		q              = 0;
		kn             = 0;
		stabilization  = 0;
	end
	methods
        function createxml(obj,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- Hydrology -->');
            
            % Convergence criteria            
            fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="Hydrologyshreve solution parameters">','<section name="hydrologyshreve" />');                    
            
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="spcwatercolumn" type="',      class(obj.spcwatercolumn),'" default="',        convert2str(obj.spcwatercolumn),'">',	'     <section name="hydrologyshreve" />','     <help> water thickness constraints (NaN means no constraint) [m] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="n" type="',          class(obj.n),'" default="',            convert2str(obj.n),'">',     '     <section name="hydrologyshreve" />','     <help> Manning roughness coefficient </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',	'<parameter key ="CR" type="',        class(obj.CR),'" default="',          convert2str(obj.CR),'">',	'     <section name="hydrologyshreve" />','     <help> tortuosity parameter </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="p" type="',          class(obj.p),'" default="',            convert2str(obj.p),'">',	'     <section name="hydrologyshreve" />','     <help> dimensionless exponent in Manning velocity formula </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="q" type="',  	class(obj.q),'" default="',       convert2str(obj.q),'">',     '     <section name="hydrologyshreve" />','     <help> dimensionless exponent in Manning velocity formula </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="kn" type="',         class(obj.kn),'" default="',           convert2str(obj.kn),'">',	'     <section name="hydrologyshreve" />','     <help> parameter in effective pressure formula </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="stabilization" type="',         class(obj.stabilization),'" default="',           convert2str(obj.stabilization),'">',	'     <section name="hydrologyshreve" />','     <help> artificial diffusivity (default is 1). can be more than 1 to increase diffusivity. </help>','</parameter>');
            fprintf(fid,'%s\n%s\n','</frame>');
        end % }}}
		function obj = hydrologyshreve(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					obj=structtoobj(obj,varargin{1});
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%Parameters from Johnson's 2002 thesis, section 3.5.4			 
			obj.n  = .02;
			obj.CR = .01;
			obj.p  = 2;
			obj.q  = 1;
			obj.kn = 0;

			%Type of stabilization to use 0:nothing 1:artificial_diffusivity
			obj.stabilization=1;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(HydrologyShreveAnalysisEnum(),analyses)
				return;
			end

			md = checkfield(md,'fieldname','hydrology.spcwatercolumn','forcing',1);
			md = checkfield(md,'fieldname','hydrology.stabilization','>=',0);
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   hydrologyshreve solution parameters:'));

			fielddisplay(obj,'spcwatercolumn','water thickness constraints (NaN means no constraint) [m]');
			fielddisplay(obj,'n','Manning roughness coefficient');
			fielddisplay(obj,'CR','tortuosity parameter');
			fielddisplay(obj,'p','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'q','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'kn','parameter in effective pressure formula');
			fielddisplay(obj,'stabilization','artificial diffusivity (default is 1). can be more than 1 to increase diffusivity.');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'enum',HydrologyModelEnum(),'data',HydrologyshreveEnum(),'format','Integer');
			WriteData(fid,'object',obj,'fieldname','spcwatercolumn','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'fieldname','n','format','Double');
			WriteData(fid,'object',obj,'fieldname','CR','format','Double');
			WriteData(fid,'object',obj,'fieldname','p','format','Double');
			WriteData(fid,'object',obj,'fieldname','q','format','Double');
			WriteData(fid,'object',obj,'fieldname','kn','format','Double');
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Double');
		end % }}}
	end
end
