%GENERIC cluster class definition
%
%   Usage:
%      cluster=generic('name','astrid','np',3);
%      cluster=generic('name',oshostname(),'np',3,'login','username');

classdef generic
	properties (SetAccess=public) 
		% {{{
		name='';
		login='';
		np=1;
		port=0;
		interactive=1;
		codepath=[issmdir() '/bin'];
		etcpath=[issmdir() '/etc'];
		executionpath=[issmdir() '/execution'];
		valgrind=[issmdir() '/externalpackages/valgrind/install/bin/valgrind'];
		valgrindlib=[issmdir() '/externalpackages/valgrind/install/lib/libmpidebug.so'];
		valgrindsup=[issmdir() '/externalpackages/valgrind/issm.supp'];
		verbose=1;
		%}}}
	end
	methods
        function createxml(obj,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- generic -->');
            
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="name" type="',class(obj.name),'" default="',convert2str(obj.name),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="login" type="',class(obj.login),'" default="',convert2str(obj.login),'">',     '     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',	'<parameter key ="np" type="',class(obj.np),'" default="',convert2str(obj.np),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="port" type="',class(obj.port),'" default="',convert2str(obj.port),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="codepath" type="',class(obj.codepath),'" default="',convert2str(obj.codepath),'">',     '     <section name="cluster" />','     <help> N/A </help>','</parameter>');
             
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="executionpath" type="',class(obj.executionpath),'" default="',convert2str(obj.executionpath),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="etcpath" type="',class(obj.etcpath),'" default="',convert2str(obj.etcpath),'">',     '     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="valgrind" type="',class(obj.valgrind),'" default="',convert2str(obj.valgrind),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="valgrindlib" type="',class(obj.valgrindlib),'" default="',convert2str(obj.valgrindlib),'">',     '     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="valgrindsup" type="',class(obj.valgrindsup),'" default="',convert2str(obj.valgrindsup),'">',     '     <section name="cluster" />','     <help> N/A </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n',    	'<parameter key ="verbose" type="',class(obj.verbose),'" default="',convert2str(obj.verbose),'">',	'     <section name="cluster" />','     <help> N/A </help>','</parameter>');
             
        end % }}}
		function cluster=generic(varargin) % {{{

			%use provided options to change fields
			options=pairoptions(varargin{:});

			%get name
			cluster.name=getfieldvalue(options,'name',oshostname());

			%initialize cluster using user settings if provided
			if (exist([cluster.name '_settings'])==2), eval([cluster.name '_settings']); end

			%OK get other fields
			cluster=AssignObjectFields(pairoptions(varargin{:}),cluster);
		end
		%}}}
		function disp(cluster) % {{{
			%  display the object
			disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			disp(sprintf('    name: %s',cluster.name));
			disp(sprintf('    login: %s',cluster.login));
			disp(sprintf('    np: %i',cluster.np));
			disp(sprintf('    port: %i',cluster.port));
			disp(sprintf('    codepath: %s',cluster.codepath));
			disp(sprintf('    executionpath: %s',cluster.executionpath));
			disp(sprintf('    etcpath: %s',cluster.executionpath));
			disp(sprintf('    valgrind: %s',cluster.valgrind));
			disp(sprintf('    valgrindlib: %s',cluster.valgrindlib));
			disp(sprintf('    valgrindsup: %s',cluster.valgrindsup));
			disp(sprintf('    verbose: %s',cluster.verbose));
		end
		%}}}
		function md = checkconsistency(cluster,md,solution,analyses) % {{{
			if cluster.np<1
				md = checkmessage(md,['number of processors should be at least 1']);
			end
			if isnan(cluster.np),
				md = checkmessage(md,'number of processors should not be NaN!');
			end
		end
		%}}}
		function BuildQueueScript(cluster,dirname,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			%write queuing script 
			if ~ispc(),

				fid=fopen([modelname '.queue'],'w');
				fprintf(fid,'#!/bin/sh\n');
				if ~isvalgrind,
					if cluster.interactive
						if IssmConfig('_HAVE_MPI_'),
							fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s ',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
						else
							fprintf(fid,'%s/issm.exe %s %s %s ',cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
						end
					else
						if IssmConfig('_HAVE_MPI_'),
							fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname,modelname,modelname);
						else
							fprintf(fid,'%s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname,modelname,modelname);
						end
					end
				elseif isgprof,
					fprintf(fid,'\n gprof %s/issm.exe gmon.out > %s.performance',cluster.codepath,modelname);
				else
					%Add --gen-suppressions=all to get suppression lines
					fprintf(fid,'LD_PRELOAD=%s \\\n',cluster.valgrindlib);
					if ismac, 
						if IssmConfig('_HAVE_MPI_'),
							fprintf(fid,'mpiexec -np %i %s --leak-check=full --dsymutil=yes --suppressions=%s %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',...
							cluster.np,cluster.valgrind,cluster.valgrindsup,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname], modelname,modelname,modelname);
						else
							fprintf(fid,'%s --leak-check=full --dsymutil=yes --suppressions=%s %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',...
							cluster.valgrind,cluster.valgrindsup,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname], modelname,modelname,modelname);
						end
					else
						if IssmConfig('_HAVE_MPI_'),
							fprintf(fid,'mpiexec -np %i %s --leak-check=full --suppressions=%s %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',...
							cluster.np,cluster.valgrind,cluster.valgrindsup,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname,modelname,modelname);
						else
							fprintf(fid,'%s --leak-check=full --suppressions=%s %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',...
							cluster.valgrind,cluster.valgrindsup,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname,modelname,modelname);
						end
					end
				end
				if ~io_gather, %concatenate the output files:
					fprintf(fid,'\ncat %s.outbin.* > %s.outbin',modelname,modelname);
				end
				fclose(fid);

			else % Windows

				fid=fopen([modelname '.bat'],'w');
				fprintf(fid,'@echo off\n');

				warning('parallel runs not allowed yet in Windows. Defaulting to 1 cpus');
				cluster.np=1;

				if cluster.np>1,
					fprintf(fid,'"C:\\Program Files\\MPICH2\\bin\\mpiexec.exe" -n %i "%s/issm.exe" %s ./ %s ',cluster.np,cluster.codepath,EnumToString(solution),modelname);
				else
					fprintf(fid,'"%s/issm.exe" %s ./ %s ',cluster.codepath,EnumToString(solution),modelname);
				end
				fclose(fid);
			end

			%in interactive mode, create a run file, and errlog and outlog file
			if cluster.interactive,
				fid=fopen([modelname '.errlog'],'w'); fclose(fid);
				fid=fopen([modelname '.outlog'],'w'); fclose(fid);
			end
		end
		%}}}
		function BuildKrigingQueueScript(cluster,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			%write queuing script 
			if ~ispc(),

				fid=fopen([modelname '.queue'],'w');
				fprintf(fid,'#!/bin/sh\n');
				if ~isvalgrind,
					if cluster.interactive
						fprintf(fid,'mpiexec -np %i %s/kriging.exe %s %s ',cluster.np,cluster.codepath,[cluster.executionpath '/' modelname],modelname);
					else
						fprintf(fid,'mpiexec -np %i %s/kriging.exe %s %s 2> %s.errlog >%s.outlog ',cluster.np,cluster.codepath,[cluster.executionpath '/' modelname],modelname,modelname,modelname);
					end
				elseif isgprof,
					fprintf(fid,'\n gprof %s/kriging.exe gmon.out > %s.performance',cluster.codepath,modelname);
				else
					%Add --gen-suppressions=all to get suppression lines
					fprintf(fid,'LD_PRELOAD=%s \\\n',cluster.valgrindlib);
					fprintf(fid,'mpiexec -np %i %s --leak-check=full --suppressions=%s %s/kriging.exe %s %s 2> %s.errlog >%s.outlog ',...
						cluster.np,cluster.valgrind,cluster.valgrindsup,cluster.codepath,[cluster.executionpath '/' modelname],modelname,modelname,modelname);
				end
				if ~io_gather, %concatenate the output files:
					fprintf(fid,'\ncat %s.outbin.* > %s.outbin',modelname,modelname);
				end
				fclose(fid);

			else % Windows

				fid=fopen([modelname '.bat'],'w');
				fprintf(fid,'@echo off\n');
				if cluster.interactive
					fprintf(fid,'"%s/issm.exe" %s "%s" %s ',cluster.codepath,EnumToString(solution),[cluster.executionpath '/' modelname],modelname);
				else
					fprintf(fid,'"%s/issm.exe" %s "%s" %s 2> %s.errlog >%s.outlog',...
						cluster.codepath,EnumToString(solution),[cluster.executionpath '/' modelname],modelname,modelname,modelname);
				end
				fclose(fid);
			end

			%in interactive mode, create a run file, and errlog and outlog file
			if cluster.interactive,
				fid=fopen([modelname '.errlog'],'w'); fclose(fid);
				fid=fopen([modelname '.outlog'],'w'); fclose(fid);
			end
		end
		%}}}
		function LaunchQueueJob(cluster,modelname,dirname,filelist)% {{{

			if ~ispc,
				%compress the files into one zip.
				compressstring=['tar -zcf ' dirname '.tar.gz '];
				for i=1:numel(filelist),
					compressstring = [compressstring ' ' filelist{i}];
				end
				if cluster.interactive,
					compressstring = [compressstring ' ' modelname '.errlog ' modelname '.outlog '];
				end
				system(compressstring);

				if cluster.verbose, disp('uploading input file and queueing script'); end
				issmscpout(cluster.name,cluster.executionpath,cluster.login,cluster.port,{[dirname '.tar.gz']});

				if cluster.verbose, disp('launching solution sequence on remote cluster'); end
				launchcommand=['source ' cluster.etcpath '/environment.sh && cd ' cluster.executionpath ' && rm -rf ./' dirname ' && mkdir ' dirname ...
					' && cd ' dirname ' && mv ../' dirname '.tar.gz ./ && tar -zxf ' dirname '.tar.gz  && source  ' modelname '.queue '];
				issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
			else
				system([modelname '.bat']);
			end
		end %}}}
		function Download(cluster,dirname,filelist)% {{{

			if ispc(),
				%do nothing
				return;
			end

			%copy files from cluster to current directory
			directory=[cluster.executionpath '/' dirname '/'];
			issmscpin(cluster.name,cluster.login,cluster.port,directory,filelist);
		end %}}}
	end
end
