%TIMESTEPPING Class definition
%
%   Usage:
%      timestepping=timestepping();

classdef timestepping
	properties (SetAccess=public) 
		start_time      = 0.;
		final_time      = 0.;
		time_step       = 0.;
		time_adapt      = 0;
		cfl_coefficient = 0.;
	end
	methods
       function createxml(obj,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- Timestepping -->');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="start_time" type="',              class(obj.start_time),'" default="',                  convert2str(obj.start_time),'">',              '     <section name="timestepping" />','     <help> simulation starting time [yr] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="final_time" type="',            class(obj.final_time),'" default="',                convert2str(obj.final_time),'">',   '     <section name="timestepping" />','     <help> final time to stop the simulation [yr] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',        '<parameter key ="time_step" type="',      class(obj.time_step),'" default="',          convert2str(obj.time_step),'">',            '     <section name="timestepping" />','     <help> length of time steps [yr] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',        '<parameter key ="time_adapt" type="',     class(obj.time_adapt),'" default="',         convert2str(obj.time_adapt),'">',            '     <section name="timestepping" />','     <help> use cfl condition to define time step ? (0 or 1)  </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="cfl_coefficient" type="',       class(obj.cfl_coefficient),'" default="',              convert2str(obj.cfl_coefficient),'">',              '     <section name="timestepping" />','     <help> coefficient applied to cfl condition </help>','</parameter>');
            
        end % }}}
        
		function obj = timestepping(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%time between 2 time steps
			obj.time_step=1./2.;

			%final time
			obj.final_time=10.*obj.time_step;

			%time adaptation? 
			obj.time_adapt=0;
			obj.cfl_coefficient=0.5;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','timestepping.start_time','numel',[1],'NaN',1);
			md = checkfield(md,'fieldname','timestepping.final_time','numel',[1],'NaN',1);
			md = checkfield(md,'fieldname','timestepping.time_step','numel',[1],'>=',0,'NaN',1);
			md = checkfield(md,'fieldname','timestepping.time_adapt','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','timestepping.cfl_coefficient','numel',[1],'>',0,'<=',1);
			if obj.final_time-obj.start_time<0,
				md = checkmessage(md,'timestepping.final_time should be larger than timestepping.start_time');
			end 
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   timestepping parameters:'));

			fielddisplay(obj,'start_time','simulation starting time [yr]');
			fielddisplay(obj,'final_time','final time to stop the simulation [yr]');
			fielddisplay(obj,'time_step','length of time steps [yr]');
			fielddisplay(obj,'time_adapt','use cfl condition to define time step ? (0 or 1) ');
			fielddisplay(obj,'cfl_coefficient','coefficient applied to cfl condition');

		end % }}}
		function marshall(obj,md,fid) % {{{

			yts=365.0*24.0*3600.0;

			WriteData(fid,'object',obj,'fieldname','start_time','format','Double','scale',yts);
			WriteData(fid,'object',obj,'fieldname','final_time','format','Double','scale',yts);
			WriteData(fid,'object',obj,'fieldname','time_step','format','Double','scale',yts);
			WriteData(fid,'object',obj,'fieldname','time_adapt','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','cfl_coefficient','format','Double');
		end % }}}
	end
end
