from fielddisplay import fielddisplay
from EnumDefinitions import *
from StringToEnum import StringToEnum
from checkfield import checkfield
from WriteData import WriteData
import MatlabFuncs as m

class damage(object):
	"""
	DAMAGE class definition

	   Usage:
	      damage=damage()
	"""

	def __init__(self,*args):    # {{{
			
		#damage: 
		self.D						= float('NaN')
		self.law						= ''
		self.spcdamage				= float('NaN')
		self.max_damage			= float('NaN')
		
		#numerical
		stabilization				= float('NaN')
		penalty_threshold			= float('NaN')
		maxiter						= float('NaN')
		penalty_lock				= float('NaN')
		penalty_factor				= float('NaN')

		#general parameters for evolution law: 
		self.stress_threshold   = float('NaN')
		self.c1                 = float('NaN')
		self.c2                 = float('NaN')
		self.c3                 = float('NaN')
		self.c4                 = float('NaN')
		self.healing				= float('NaN')
		self.equiv_stress       = float('NaN')
		self.requested_outputs  = []

		if not len(args):
			self.setdefaultparameters()
		else:
			raise RuntimeError("constructor not supported")

	# }}}
	def __repr__(self):    # {{{
		s ='   Damage:\n'

		s+="%s\n" % fielddisplay(self,"D","damage tensor (scalar for now)")
		s+="%s\n" % fielddisplay(self,"law","damage law (string) from ['undamaged','pralong']")
		s+="%s\n" % fielddisplay(self,"spcdamage","damage constraints (NaN means no constraint)")
		s+="%s\n" % fielddisplay(self,"max_damage","maximum possible damage (0<=max_damage<1)")

		s+="%s\n" % fielddisplay(self,"stabilization","0: no, 1: artificial_diffusivity, 2: SUPG")
		s+="%s\n" % fielddisplay(self,"maxiter","maximum number of non linear iterations")
		s+="%s\n" % fielddisplay(self,"penalty_lock","stabilize unstable damage constraints that keep zigzagging after n iteration (default is 0, no stabilization)")
		s+="%s\n" % fielddisplay(self,"penalty_threshold","threshold to declare convergence of damage evolution solution (default is 0)")
		s+="%s\n" % fielddisplay(self,"penalty_factor","scaling exponent (default is 3)")

		if (self.law=='pralong'):
			s+="%s\n" % fielddisplay(self,"c1","damage parameter 1 ")
			s+="%s\n" % fielddisplay(self,"c2","damage parameter 2 ")
			s+="%s\n" % fielddisplay(self,"c3","damage parameter 3 ")
			s+="%s\n" % fielddisplay(self,"c4","damage parameter 4 ")
			s+="%s\n" % fielddisplay(self,"stress_threshold","damage stress threshold [Pa]")
			s+="%s\n" % fielddisplay(self,"healing","damage healing parameter")
			s+="%s\n" % fielddisplay(self,"equiv_stress","0: von Mises, 1: max principal")
			s+="%s\n" % fielddisplay(self,'requested_outputs','additional outputs requested')

		return s
	# }}}
	def setdefaultparameters(self):    # {{{

		#damage parameters: 
		self.D=0
		self.law='undamaged'

		self.max_damage=1-1e-5 #if damage reaches 1, solve becomes singular, as viscosity becomes nil
		
		#Type of stabilization used
		self.stabilization=2
			
		#Maximum number of iterations
		self.maxiter=100

		#factor used to compute the values of the penalties: kappa=max(stiffness matrix)*10^penalty_factor
		self.penalty_factor=3
			
		#stabilize unstable damage constraints that keep zigzagging after n iteration (default is 0, no stabilization)
		self.penalty_lock=0
			
		#threshold to declare convergence of thermal solution (default is 0)
		self.penalty_threshold=0
		
		#damage evolution parameters 
		self.stress_threshold=0
		self.c1=0
		self.c2=0
		self.c3=0
		self.c4=0
		self.healing=0
		self.equiv_stress=0

		#output default:
		self.requested_outputs=['default']

		return self
	# }}}
	def defaultoutputs(self,md): # {{{

		if m.strcmp(md.mesh.meshtype(),'3D'):
			list = ['DamageD']
		elif m.strcmp(md.mesh.meshtype(),'2Dhorizontal'):
			list = ['DamageD']
		elif m.strcmp(md.mesh.meshtype(),'2Dvertical'):
			list = ['DamageD']
		else:
			raise TypeError('mesh type not supported yet')
		return list

	#}}}
	def checkconsistency(self,md,solution,analyses):    # {{{

		md = checkfield(md,'fieldname','damage.D','>=',0,'<=',self.max_damage,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'fieldname','damage.max_damage','<',1,'>=',0)
		md = checkfield(md,'fieldname','damage.law','values',['undamaged','pralong'])
		md = checkfield(md,'fieldname','damage.spcdamage','forcing',1)
			
		md = checkfield(md,'fieldname','damage.stabilization','numel',[1],'values',[0,1,2])
		md = checkfield(md,'fieldname','damage.maxiter','>=0',0)
		md = checkfield(md,'fieldname','damage.penalty_factor','>=0',0)
		md = checkfield(md,'fieldname','damage.penalty_lock','>=0',0)
		md = checkfield(md,'fieldname','damage.penalty_threshold','>=0',0)

		if self.law == 'pralong':
			md = checkfield(md,'fieldname','damage.healing','>=',0)
			md = checkfield(md,'fieldname','damage.c1','>=',0)
			md = checkfield(md,'fieldname','damage.c2','>=',0)
			md = checkfield(md,'fieldname','damage.c3','>=',0)
			md = checkfield(md,'fieldname','damage.c4','>=',0)
			md = checkfield(md,'fieldname','damage.stress_threshold','>=',0)
			md = checkfield(md,'fieldname','damage.healing','>=',0)
			md = checkfield(md,'fieldname','damage.equiv_stress','numel',[1],'values',[0,1])
			md = checkfield(md,'fieldname','damage.requested_outputs','stringrow',1)
		elif m.strcmpi(self.law,'undamaged'):
			if (solution==DamageEvolutionSolutionEnum):
				raise RuntimeError('Invalid evolution law (md.damage.law) for a damage solution')

		return md
	# }}}
	def marshall(self,md,fid):    # {{{

		WriteData(fid,'object',self,'fieldname','D','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'fieldname','law','format','String')
		WriteData(fid,'object',self,'fieldname','spcdamage','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1)
		WriteData(fid,'object',self,'fieldname','max_damage','format','Double')

		WriteData(fid,'object',self,'fieldname','stabilization','format','Integer')
		WriteData(fid,'object',self,'fieldname','penalty_threshold','format','Integer')
		WriteData(fid,'object',self,'fieldname','maxiter','format','Integer')
		WriteData(fid,'object',self,'fieldname','penalty_lock','format','Integer')
		WriteData(fid,'object',self,'fieldname','penalty_factor','format','Double')

		if self.law=='pralong':
			WriteData(fid,'object',self,'fieldname','c1','format','Double')
			WriteData(fid,'object',self,'fieldname','c2','format','Double')
			WriteData(fid,'object',self,'fieldname','c3','format','Double')
			WriteData(fid,'object',self,'fieldname','c4','format','Double')
			WriteData(fid,'object',self,'fieldname','stress_threshold','format','Double')
			WriteData(fid,'object',self,'fieldname','healing','format','Double')
			WriteData(fid,'object',self,'fieldname','equiv_stress','format','Integer')
			
		#process requested outputs
		outputs = self.requested_outputs
		indices = [i for i, x in enumerate(outputs) if x == 'default']
		if len(indices) > 0:
			outputscopy=outputs[0:max(0,indices[0]-1)]+self.defaultoutputs(md)+outputs[indices[0]+1:]
			outputs    =outputscopy
		WriteData(fid,'data',outputs,'enum',DamageEvolutionRequestedOutputsEnum(),'format','StringArray')
	# }}}
