/*!\file: solutionsequence_la_theta.cpp
 * \brief: numerical core of la_theta solutions
 */ 

#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

void solutionsequence_la_theta(FemModel* femmodel){

	/*intermediary: */
	Matrix<IssmDouble>*  Kff    = NULL;
	Matrix<IssmDouble>*  Kfs    = NULL;
	Vector<IssmDouble>*  ug_old = NULL;
	Vector<IssmDouble>*  ug     = NULL;
	Vector<IssmDouble>*  uf     = NULL;
	Vector<IssmDouble>*  pf     = NULL;
	Vector<IssmDouble>*  df     = NULL;
	Vector<IssmDouble>*  ys     = NULL;
	int  configuration_type,max_nonlinear_iterations;

	/*Create analysis*/
	StressbalanceAnalysis* analysis = new StressbalanceAnalysis();

	/*Recover parameters: */
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,StressbalanceMaxiterEnum);

	/*Update constraints and initialize d and tau if necessary*/
	femmodel->UpdateConstraintsx();
	analysis->InitializeXTH(femmodel->elements,femmodel->parameters);

	/*Convergence criterion*/
	int        count   = 0;
	IssmDouble eps_rel = .001;
	femmodel->parameters->SetParam(.6,AugmentedLagrangianREnum);
	GetSolutionFromInputsx(&ug,femmodel);

	while(true){

		/*save pointer to old velocity*/
		delete ug_old;ug_old=ug;

		/*Solve KU=F*/
		SystemMatricesx(&Kff,&Kfs,&pf,&df,NULL,femmodel);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf, Kfs, ys); delete Kfs;

	//	pf->Echo();
	//	IssmDouble* temp=Kff->ToSerial();
	//	int m,n;
	//	Kff->GetSize(&m,&n);
	//	printarray(temp,m,n);
	//	xDelete<IssmDouble>(temp);

		Solverx(&uf, Kff, pf, NULL, df, femmodel->parameters); 
		//uf->Echo();
		delete Kff; delete pf; delete df;

		Mergesolutionfromftogx(&ug, uf,ys,femmodel->nodes,femmodel->parameters);delete uf; delete ys;

		/*Update solution*/
		_printf_("ug norm = " << ug->Norm(NORM_TWO) << "\n");
		InputUpdateFromSolutionx(femmodel,ug); 

		/*Update d and tau accordingly*/
		analysis->InputUpdateFromSolutionFSXTH(femmodel->elements,femmodel->parameters);

		/*Check for convergence*/
		Vector<IssmDouble>* dug=ug_old->Duplicate(); ug_old->Copy(dug); dug->AYPX(ug,-1.0);
		IssmDouble ndu=dug->Norm(NORM_TWO);   delete dug;
		IssmDouble nu =ug_old->Norm(NORM_TWO);
		if (xIsNan<IssmDouble>(ndu) || xIsNan<IssmDouble>(nu)) _error_("convergence criterion is NaN!");
		if((ndu/nu)<eps_rel){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: norm(du)/norm(u)" << ndu/nu*100 << " < " << eps_rel*100 << " %\n");
			break;
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "   Convergence criterion: norm(du)/norm(u)" << ndu/nu*100 << " > " << eps_rel*100 << " %\n");
		}

		count++;
		if(count>=max_nonlinear_iterations) break;
	}

	delete ug;  
	delete ug_old;  
	delete analysis;
}
