%GEOMETRY class definition
%
%   Usage:
%      geometry=geometry();

classdef geometry
	properties (SetAccess=public) 
		surface           = NaN;
		thickness         = NaN;
		base              = NaN;
		bed               = NaN;
		hydrostatic_ratio = NaN;
	end
	methods (Static)
		function obj = loadobj(obj) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. Update old properties here

			%2014 March 26th
			if isstruct(obj),
				disp('WARNING: updating geometry');
				disp('         md.geometry.bed        is now md.geometry.base');
				disp('         md.geometry.bathymetry is now md.geometry.bed');
				obj2=obj;
				obj=geometry();
				obj.surface    = obj2.surface;
				obj.thickness  = obj2.thickness;
				obj.base       = obj2.bed;
				obj.bed        = obj2.bathymetry;
			end

		end% }}}
	end
	methods
		function obj = geometry(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			if (solution==TransientSolutionEnum() & md.transient.isgia) | (solution==GiaSolutionEnum()),
				md = checkfield(md,'fieldname','geometry.thickness','forcing',1,'NaN',1,'>=',0);
			else
				md = checkfield(md,'fieldname','geometry.surface'  ,'NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','geometry.base'      ,'NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','geometry.thickness','NaN',1,'size',[md.mesh.numberofvertices 1],'>',0);
				if any((obj.thickness-obj.surface+obj.base)>10^-9),
					md = checkmessage(md,['equality thickness=surface-base violated']);
				end 
				if solution==TransientSolutionEnum() & md.transient.isgroundingline,
					md = checkfield(md,'fieldname','geometry.bed','NaN',1,'size',[md.mesh.numberofvertices 1]);
				end
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   geometry parameters:'));

			fielddisplay(obj,'surface','ice upper surface elevation [m]');
			fielddisplay(obj,'thickness','ice thickness [m]');
			fielddisplay(obj,'base','ice base elevation [m]');
			fielddisplay(obj,'bed','bed elevation [m]');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'data',obj.surface,'format','DoubleMat','mattype',1,'enum',SurfaceEnum());
			WriteData(fid,'data',obj.thickness,'format','DoubleMat','mattype',1,'enum',ThicknessEnum(),'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'data',obj.base,'format','DoubleMat','mattype',1,'enum',BaseEnum());
			WriteData(fid,'data',obj.bed,'format','DoubleMat','mattype',1,'enum',BedEnum());
			WriteData(fid,'object',obj,'fieldname','hydrostatic_ratio','format','DoubleMat','mattype',1);
		end % }}}
	end
end
