function md=SetMarineIceSheetBC(md,varargin)
%SETICEMARINESHEETBC - Create the boundary conditions for diagnostic and thermal models for a  Marine Ice Sheet with Ice Front
%
%   Neumann BC are used on the ice front (an ARGUS contour around the ice front
%   can be given in input, or it will be deduced as oniceshelf & onboundary)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetMarineIceSheetBC(md,icefrontfile)
%      md=SetMarineIceSheetBC(md)
%
%   Example:
%      md=SetMarineIceSheetBC(md,'Front.exp')
%      md=SetMarineIceSheetBC(md)
%
%   See also: SETICESHELFBC, SETMARINEICESHEETBC

%grid on Dirichlet (boundary and ~icefront)
if nargin==2,
	%User provided Front.exp, use it
	icefrontfile=varargin{1};
	if ~exist(icefrontfile)
		error(['SetMarineIceSheetBC error message: ice front file ' icefrontfile ' not found']);
	end
	gridinsideicefront=ContourToMesh(md.elements,md.x,md.y,expread(icefrontfile,1),'node',2);
	gridonicefront=double(md.gridonboundary & gridinsideicefront);
else
	%Guess where the ice front is
	gridonicefront=double(md.gridonboundary & md.gridoniceshelf);
end
pos=find(md.gridonboundary & ~gridonicefront);
if isempty(pos),
	error('SetMarineIceSheetBC error message: ice front all around the glacier, no dirichlet found. Set BC manually')
end
md.spcvelocity=zeros(md.numberofgrids,6);
md.spcvelocity(pos,1:3)=1;

%Dirichlet Values
if (length(md.vx_obs)==md.numberofgrids & length(md.vy_obs)==md.numberofgrids)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.spcvelocity(pos,4:5)=[md.vx_obs(pos) md.vy_obs(pos)]; %zeros for vz
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

%segment on Neumann (Ice Front)
pos=find(gridonicefront(md.segments(:,1)) | gridonicefront(md.segments(:,2)));
md.pressureload=md.segments(pos,:);

%Create zeros melting and accumulation if not specified
if isnan(md.accumulation),
	md.accumulation=zeros(md.numberofgrids,1);
	disp('      no accumulation specified: values set as zero');
end
if isnan(md.melting),
	md.melting=zeros(md.numberofgrids,1);
	disp('      no melting specified: values set as zero');
end

displaystring(md.debug,'%s',['      boundary conditions for prognostic model initialization']);
md.spcthickness=zeros(md.numberofgrids,2);

if (length(md.observed_temperature)==md.numberofgrids),
	displaystring(md.debug,'%s',['      boundary conditions for thermal model']);
	md.spctemperature=ones(md.numberofgrids,2); %surface temperature
	md.spctemperature(:,2)=md.observed_temperature; %values of the spcs
	if (length(md.geothermalflux)~=md.numberofgrids),
		md.geothermalflux=zeros(md.numberofgrids,1);
		md.geothermalflux(find(md.gridonicesheet))=50*10^-3; %50mW/m2
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
