/*!\file Penta.cpp
 * \brief: implementation of the Penta object
 */

/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../classes.h"
#include "../../shared/shared.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES   6
#define NUMVERTICES2D 3

/*Constructors/destructor/copy*/
/*FUNCTION Penta::~Penta(){{{*/
Penta::~Penta(){
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Penta::Penta(int id, int index, IoModel* iomodel,int nummodels) {{{*/
Penta::Penta(int penta_id, int penta_sid, int index, IoModel* iomodel,int nummodels)
	:PentaRef(nummodels)
	,ElementHook(nummodels,index+1,6,iomodel){

	int penta_elements_ids[2];

	/*Checks in debugging mode*/
	_assert_(iomodel->Data(MeshUpperelementsEnum));
	_assert_(iomodel->Data(MeshLowerelementsEnum));

	/*id: */
	this->id=penta_id;
	this->sid=penta_sid;

	/*Build neighbors list*/
	if (xIsNan<IssmDouble>(iomodel->Data(MeshUpperelementsEnum)[index]) || iomodel->Data(MeshUpperelementsEnum)[index]==-1.) penta_elements_ids[1]=this->id; //upper penta is the same penta
	else                                    penta_elements_ids[1]=reCast<int,IssmDouble>((iomodel->Data(MeshUpperelementsEnum)[index]));
	if (xIsNan<IssmDouble>(iomodel->Data(MeshLowerelementsEnum)[index]) || iomodel->Data(MeshLowerelementsEnum)[index]==-1.) penta_elements_ids[0]=this->id; //lower penta is the same penta
	else                                    penta_elements_ids[0]=reCast<int,IssmDouble>((iomodel->Data(MeshLowerelementsEnum)[index]));
	this->InitHookNeighbors(penta_elements_ids);

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs: */
	this->inputs=new Inputs();

	/*initialize pointers:*/
	this->nodes             = NULL;
	this->vertices          = NULL;
	this->material          = NULL;
	this->matpar            = NULL;
	this->verticalneighbors = NULL;
}
/*}}}*/
/*FUNCTION Penta::copy {{{*/
Object* Penta::copy() {

	int i;

	Penta* penta=NULL;

	penta=new Penta();

	//deal with PentaRef mother class
	penta->element_type_list=xNew<int>(this->numanalyses);
	for(i=0;i<this->numanalyses;i++) penta->element_type_list[i]=this->element_type_list[i];

	//deal with ElementHook
	penta->numanalyses=this->numanalyses;
	penta->hnodes=new Hook*[penta->numanalyses];
	for(i=0;i<penta->numanalyses;i++)penta->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	penta->hvertices=(Hook*)this->hvertices->copy();
	penta->hmaterial=(Hook*)this->hmaterial->copy();
	penta->hmatpar=(Hook*)this->hmatpar->copy();
	penta->hneighbors=(Hook*)this->hneighbors->copy();

	/*deal with Penta  copy fields: */
	penta->id=this->id;
	penta->sid=this->sid;
	if(this->inputs){
		penta->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		penta->inputs=new Inputs();
	}
	/*point parameters: */
	penta->parameters=this->parameters;

	/*recover objects: */
	penta->nodes=xNew<Node*>(6); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<6;i++)penta->nodes[i]=this->nodes[i];
	penta->vertices=(Vertex**)penta->hvertices->deliverp();
	penta->material=(Material*)penta->hmaterial->delivers();
	penta->matpar=(Matpar*)penta->hmatpar->delivers();
	penta->verticalneighbors=(Penta**)penta->hneighbors->deliverp();

	return penta;
}
/*}}}*/

/*Other*/
/*FUNCTION Penta::AddInput{{{*/
void  Penta::AddInput(int input_enum,IssmDouble* values, int interpolation_enum){

	_assert_(this->inputs);
	this->inputs->AddInput(new PentaInput(input_enum,values,interpolation_enum));
}
/*}}}*/
/*FUNCTION Penta::AddBasalInput{{{*/
void  Penta::AddBasalInput(int input_enum,IssmDouble* values, int interpolation_enum){

	_assert_(this->inputs);
	if(!IsOnBed()) return;
	else{
		if(interpolation_enum==P1Enum){
			int        i;
			IssmDouble extrudedvalues[NUMVERTICES];
			Penta*     penta=NULL;

			for(i=0;i<NUMVERTICES2D;i++){
				extrudedvalues[i]=values[i];
				extrudedvalues[i+NUMVERTICES2D]=values[i];
			}
			penta=this;
			for(;;){
				penta->inputs->AddInput(new PentaInput(input_enum,&extrudedvalues[0],P1Enum));
				if (penta->IsOnSurface()) break;
				penta=penta->GetUpperPenta(); _assert_(penta->Id()!=this->id);
			}
		}
		else _error_("not implemented yet");
	}
}
/*}}}*/
/*FUNCTION Penta::AddMaterialInput{{{*/
void  Penta::AddMaterialInput(int input_enum,IssmDouble* values, int interpolation_enum){

	_assert_(this->material);
	this->material->inputs->AddInput(new PentaInput(input_enum,values,interpolation_enum));
}
/*}}}*/
/*FUNCTION Penta::BasalFrictionCreateInput {{{*/
void Penta::BasalFrictionCreateInput(void){

	/*Intermediaries */
	int         count;
	IssmDouble  basalfriction[NUMVERTICES];
	IssmDouble  alpha2                       ,vx,vy;
	Friction   *friction                   = NULL;
	GaussPenta *gauss                      = NULL;

	/* Basal friction can only be found at the base of an ice sheet: */
	if (!IsOnBed() || IsFloating()){
		//empty friction: 
		this->inputs->AddInput(new PentaInput(BasalFrictionEnum,&basalfriction[0],P1Enum));
		return;
	}

	/*Retrieve all inputs and parameters*/
	Input* vx_input=inputs->GetInput(VxEnum);                         _assert_(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);                         _assert_(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);                         _assert_(vz_input);

	/*Build friction element, needed later: */
	friction=new Friction(this,3);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	gauss=new GaussPenta(0,1,2,2);
	count=0;
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		friction->GetAlpha2(&alpha2,gauss,vx_input,vy_input,vz_input);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		basalfriction[count]=alpha2*(pow(vx,2.0)+pow(vy,2.0));
		count++;
	}

	/*Create PentaVertex input, which will hold the basal friction:*/
	this->inputs->AddInput(new PentaInput(BasalFrictionEnum,&basalfriction[0],P1Enum));

	/*Clean up and return*/
	delete gauss;
	delete friction;
}
/*}}}*/
/*FUNCTION Penta::ComputeBasalStress {{{*/
void  Penta::ComputeBasalStress(Vector<IssmDouble>* sigma_b){

	int         i,j;
	int         dofv[3]={0,1,2};
	int         dofp[1]={3};
	int         analysis_type,approximation;
	IssmDouble  xyz_list[NUMVERTICES][3];
	IssmDouble  xyz_list_tria[3][3];
	IssmDouble  rho_ice,gravity,FSreconditioning;
	IssmDouble  pressure,viscosity,Jdet2d;
	IssmDouble  bed_normal[3];
	IssmDouble  basalforce[3];
	IssmDouble  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	IssmDouble  stresstensor[6]={0.0};
	IssmDouble  sigma_xx,sigma_yy,sigma_zz;
	IssmDouble  sigma_xy,sigma_xz,sigma_yz;
	IssmDouble  surface=0,value=0;
	GaussPenta* gauss;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*Check analysis_types*/
	if (analysis_type!=StressbalanceAnalysisEnum) _error_("Not supported yet!");
	if (approximation!=FSApproximationEnum) _error_("Not supported yet!");

	/*retrieve some parameters: */
	this->parameters->FindParam(&FSreconditioning,StressbalanceFSreconditioningEnum);

	if(!IsOnBed()){
		//put zero
		sigma_b->SetValue(id-1,0.0,INS_VAL);
		return;
	}

	/*recovre material parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<3;i++) for(j=0;j<3;j++) xyz_list_tria[i][j]=xyz_list[i][j];

	/*Retrieve all inputs we will be needing: */
	Input* pressure_input=inputs->GetInput(PressureEnum); _assert_(pressure_input);
	Input* vx_input=inputs->GetInput(VxEnum);             _assert_(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);             _assert_(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);             _assert_(vz_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(0,1,2,2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Compute strain rate viscosity and pressure: */
		this->StrainRateFS(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		material->GetViscosity3dFS(&viscosity,&epsilon[0]);
		pressure_input->GetInputValue(&pressure,gauss);

		/*Compute Stress*/
		sigma_xx=2*viscosity*epsilon[0]-pressure*FSreconditioning; // sigma = nu eps - pressure
		sigma_yy=2*viscosity*epsilon[1]-pressure*FSreconditioning;
		sigma_zz=2*viscosity*epsilon[2]-pressure*FSreconditioning;
		sigma_xy=2*viscosity*epsilon[3];
		sigma_xz=2*viscosity*epsilon[4];
		sigma_yz=2*viscosity*epsilon[5];

		/*Get normal vector to the bed */
		NormalBase(&bed_normal[0],&xyz_list_tria[0][0]);

		/*basalforce*/
		basalforce[0] += sigma_xx*bed_normal[0] + sigma_xy*bed_normal[1] + sigma_xz*bed_normal[2];
		basalforce[1] += sigma_xy*bed_normal[0] + sigma_yy*bed_normal[1] + sigma_yz*bed_normal[2];
		basalforce[2] += sigma_xz*bed_normal[0] + sigma_yz*bed_normal[1] + sigma_zz*bed_normal[2];

		GetTriaJacobianDeterminant(&Jdet2d, &xyz_list_tria[0][0],gauss);
		value+=sigma_zz*Jdet2d*gauss->weight;
		surface+=Jdet2d*gauss->weight;
	}
	value=value/surface;

	/*Add value to output*/
	sigma_b->SetValue(id-1,value,INS_VAL);
}
/*}}}*/
/*FUNCTION Penta::ComputeStrainRate {{{*/
void  Penta::ComputeStrainRate(Vector<IssmDouble>* eps){

	_error_("Not implemented yet");

}
/*}}}*/
/*FUNCTION Penta::ComputeStressTensor {{{*/
void  Penta::ComputeStressTensor(){

	IssmDouble      xyz_list[NUMVERTICES][3];
	IssmDouble      pressure,viscosity;
	IssmDouble      epsilon[6]; /* epsilon=[exx,eyy,exy];*/
	IssmDouble      sigma_xx[NUMVERTICES];
	IssmDouble		sigma_yy[NUMVERTICES];
	IssmDouble		sigma_zz[NUMVERTICES];
	IssmDouble      sigma_xy[NUMVERTICES];
	IssmDouble		sigma_xz[NUMVERTICES];
	IssmDouble		sigma_yz[NUMVERTICES];
	GaussPenta* gauss=NULL;

	/* Get node coordinates and dof list: */
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	/*Retrieve all inputs we will be needing: */
	Input* pressure_input=inputs->GetInput(PressureEnum); _assert_(pressure_input);
	Input* vx_input=inputs->GetInput(VxEnum);             _assert_(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);             _assert_(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);             _assert_(vz_input);

	/* Start looping on the number of vertices: */
	gauss=new GaussPenta();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);

		/*Compute strain rate viscosity and pressure: */
		this->StrainRateFS(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input,vz_input);
		material->GetViscosity3d(&viscosity,&epsilon[0]);
		pressure_input->GetInputValue(&pressure,gauss);

		/*Compute Stress*/
		sigma_xx[iv]=2*viscosity*epsilon[0]-pressure; // sigma = nu eps - pressure
		sigma_yy[iv]=2*viscosity*epsilon[1]-pressure;
		sigma_zz[iv]=2*viscosity*epsilon[2]-pressure;
		sigma_xy[iv]=2*viscosity*epsilon[3];
		sigma_xz[iv]=2*viscosity*epsilon[4];
		sigma_yz[iv]=2*viscosity*epsilon[5];
	}

	/*Add Stress tensor components into inputs*/
	this->inputs->AddInput(new PentaInput(StressTensorxxEnum,&sigma_xx[0],P1Enum));
	this->inputs->AddInput(new PentaInput(StressTensorxyEnum,&sigma_xy[0],P1Enum));
	this->inputs->AddInput(new PentaInput(StressTensorxzEnum,&sigma_xz[0],P1Enum));
	this->inputs->AddInput(new PentaInput(StressTensoryyEnum,&sigma_yy[0],P1Enum));
	this->inputs->AddInput(new PentaInput(StressTensoryzEnum,&sigma_yz[0],P1Enum));
	this->inputs->AddInput(new PentaInput(StressTensorzzEnum,&sigma_zz[0],P1Enum));

	/*Clean up and return*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::Configure {{{*/
void  Penta::Configure(Elements* elementsin, Loads* loadsin, Nodes* nodesin,Vertices* verticesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;

	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if (this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hvertices->configure(verticesin);
	this->hmaterial->configure(materialsin);
	this->hmatpar->configure(materialsin);
	this->hneighbors->configure(elementsin);

	/*Now, go pick up the objects inside the hooks: */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->vertices          = (Vertex**)this->hvertices->deliverp();
	this->material          = (Material*)this->hmaterial->delivers();
	this->matpar            = (Matpar*)this->hmatpar->delivers();
	this->verticalneighbors = (Penta**)this->hneighbors->deliverp();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

	/*get inputs configured too: */
	this->inputs->Configure(parameters);
}
/*}}}*/
/*FUNCTION Penta::DeepEcho{{{*/
void Penta::DeepEcho(void){

	_printf_("Penta:\n");
	_printf_("   id: " << id << "\n");
	nodes[0]->DeepEcho();
	nodes[1]->DeepEcho();
	nodes[2]->DeepEcho();
	nodes[3]->DeepEcho();
	nodes[4]->DeepEcho();
	nodes[5]->DeepEcho();
	material->DeepEcho();
	matpar->DeepEcho();
	_printf_("   neighbor ids: " << verticalneighbors[0]->Id() << "-" << verticalneighbors[1]->Id() << "\n");
	_printf_("   parameters\n");
	parameters->DeepEcho();
	_printf_("   inputs\n");
	inputs->DeepEcho();
}
/*}}}*/
/*FUNCTION Penta::Delta18oParameterization{{{*/
void  Penta::Delta18oParameterization(void){
        /*Are we on the base? If not, return*/
        if(!IsOnBed()) return;

	int        i;
	IssmDouble monthlytemperatures[NUMVERTICES][12],monthlyprec[NUMVERTICES][12];
	IssmDouble TemperaturesPresentday[NUMVERTICES][12],TemperaturesLgm[NUMVERTICES][12];
	IssmDouble PrecipitationsPresentday[NUMVERTICES][12];
	IssmDouble tmp[NUMVERTICES];
	IssmDouble Delta18oPresent,Delta18oLgm,Delta18oTime;
	IssmDouble Delta18oSurfacePresent,Delta18oSurfaceLgm,Delta18oSurfaceTime;
	IssmDouble time,yts,finaltime;
	this->parameters->FindParam(&time,TimeEnum);
	this->parameters->FindParam(&yts,ConstantsYtsEnum);
	this->parameters->FindParam(&finaltime,TimesteppingFinalTimeEnum);

	/*Recover present day temperature and precipitation*/
	Input*     input=inputs->GetInput(SurfaceforcingsTemperaturesPresentdayEnum);    _assert_(input);
	Input*     input2=inputs->GetInput(SurfaceforcingsTemperaturesLgmEnum);          _assert_(input2);
	Input*     input3=inputs->GetInput(SurfaceforcingsPrecipitationsPresentdayEnum); _assert_(input3);
	GaussPenta* gauss=new GaussPenta();
	for(int month=0;month<12;month++) {
		for(int iv=0;iv<NUMVERTICES;iv++) {
			gauss->GaussVertex(iv);
			input->GetInputValue(&TemperaturesPresentday[iv][month],gauss,month/12.*yts);
			input2->GetInputValue(&TemperaturesLgm[iv][month],gauss,month/12.*yts);
			input3->GetInputValue(&PrecipitationsPresentday[iv][month],gauss,month/12.*yts);
			PrecipitationsPresentday[iv][month]=PrecipitationsPresentday[iv][month]/yts; // converion in m/sec
		}
	}

	/*Recover delta18o and Delta18oSurface at present day, lgm and at time t*/
	this->parameters->FindParam(&Delta18oPresent,SurfaceforcingsDelta18oEnum,finaltime);
	this->parameters->FindParam(&Delta18oLgm,SurfaceforcingsDelta18oEnum,finaltime-(21000*yts));
	this->parameters->FindParam(&Delta18oTime,SurfaceforcingsDelta18oEnum,time);
	this->parameters->FindParam(&Delta18oSurfacePresent,SurfaceforcingsDelta18oSurfaceEnum,finaltime);
	this->parameters->FindParam(&Delta18oSurfaceLgm,SurfaceforcingsDelta18oSurfaceEnum,finaltime-(21000*yts));
	this->parameters->FindParam(&Delta18oSurfaceTime,SurfaceforcingsDelta18oSurfaceEnum,time);

	/*Compute the temperature and precipitation*/
	for(int iv=0;iv<NUMVERTICES;iv++){
		ComputeDelta18oTemperaturePrecipitation(Delta18oSurfacePresent, Delta18oSurfaceLgm, Delta18oSurfaceTime, 
					Delta18oPresent, Delta18oLgm, Delta18oTime,
					&PrecipitationsPresentday[iv][0], 
					&TemperaturesLgm[iv][0], &TemperaturesPresentday[iv][0], 
					&monthlytemperatures[iv][0], &monthlyprec[iv][0]);
	}

	/*Update inputs*/ 
	TransientInput* NewTemperatureInput = new TransientInput(SurfaceforcingsMonthlytemperaturesEnum);
	TransientInput* NewPrecipitationInput = new TransientInput(SurfaceforcingsPrecipitationEnum);
	for (int imonth=0;imonth<12;imonth++) {
		for(i=0;i<NUMVERTICES;i++) tmp[i]=monthlytemperatures[i][imonth];
		PentaInput* newmonthinput1 = new PentaInput(SurfaceforcingsMonthlytemperaturesEnum,&tmp[0],P1Enum);
		NewTemperatureInput->AddTimeInput(newmonthinput1,time+imonth/12.*yts);

		for(i=0;i<NUMVERTICES;i++) tmp[i]=monthlyprec[i][imonth];
		PentaInput* newmonthinput2 = new PentaInput(SurfaceforcingsPrecipitationEnum,&tmp[0],P1Enum);
		NewPrecipitationInput->AddTimeInput(newmonthinput2,time+imonth/12.*yts);
	}
	NewTemperatureInput->Configure(this->parameters);
	NewPrecipitationInput->Configure(this->parameters);

	this->inputs->AddInput(NewTemperatureInput);
	this->inputs->AddInput(NewPrecipitationInput);

	this->InputExtrude(SurfaceforcingsMonthlytemperaturesEnum,ElementEnum);
	this->InputExtrude(SurfaceforcingsPrecipitationEnum,ElementEnum);

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::Echo{{{*/

void Penta::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION Penta::ThermalToEnthalpy{{{*/
void Penta::ThermalToEnthalpy(IssmDouble* penthalpy,IssmDouble temperature,IssmDouble waterfraction,IssmDouble pressure){
	matpar->ThermalToEnthalpy(penthalpy,temperature,waterfraction,pressure);
}
/*}}}*/
/*FUNCTION Penta::EnthalpyToThermal{{{*/
void Penta::EnthalpyToThermal(IssmDouble* ptemperature,IssmDouble* pwaterfraction,IssmDouble enthalpy,IssmDouble pressure){
	matpar->EnthalpyToThermal(ptemperature,pwaterfraction,enthalpy,pressure);
}
/*}}}*/
/*FUNCTION Penta::EnthalpyDiffusionParameter{{{*/
IssmDouble Penta::EnthalpyDiffusionParameter(IssmDouble enthalpy,IssmDouble pressure){
	return matpar->GetEnthalpyDiffusionParameter(enthalpy,pressure);
}
/*}}}*/
/*FUNCTION Penta::EnthalpyDiffusionParameterVolume{{{*/
IssmDouble Penta::EnthalpyDiffusionParameterVolume(int numvertices,IssmDouble* enthalpy,IssmDouble* pressure){
	return matpar->GetEnthalpyDiffusionParameterVolume(numvertices,enthalpy,pressure);
}
/*}}}*/
/*FUNCTION Penta::FiniteElement{{{*/
int Penta::FiniteElement(void){
	return this->element_type;
}
/*}}}*/
/*FUNCTION Penta::ObjectEnum{{{*/
int Penta::ObjectEnum(void){

	return PentaEnum;

}
/*}}}*/
/*FUNCTION Penta::GetAreaCoordinates{{{*/
void Penta::GetAreaCoordinates(IssmDouble* area_coordinates,IssmDouble* xyz_zero,IssmDouble* xyz_list,int numpoints){
	/*Computeportion of the element that is grounded*/ 

	int         i,j,k;
	IssmDouble  area_init,area_portion;
	IssmDouble  xyz_bis[3][3];

	area_init=fabs(xyz_list[1*3+0]*xyz_list[2*3+1] - xyz_list[1*3+1]*xyz_list[2*3+0] + xyz_list[0*3+0]*xyz_list[1*3+1] - xyz_list[0*3+1]*xyz_list[1*3+0] + xyz_list[2*3+0]*xyz_list[0*3+1] - xyz_list[2*3+1]*xyz_list[0*3+0])/2.;

	/*Initialize xyz_list with original xyz_list of triangle coordinates*/
	for(j=0;j<3;j++){ 
		for(k=0;k<3;k++){
			xyz_bis[j][k]=xyz_list[j*3+k];
		}
	}
	for(i=0;i<numpoints;i++){
		for(j=0;j<3;j++){ 
			for(k=0;k<3;k++){
				/*Change appropriate line*/
				xyz_bis[j][k]=xyz_zero[i*3+k];
			}

			/*Compute area fraction*/
			area_portion=fabs(xyz_bis[1][0]*xyz_bis[2][1] - xyz_bis[1][1]*xyz_bis[2][0] + xyz_bis[0][0]*xyz_bis[1][1] - xyz_bis[0][1]*xyz_bis[1][0] + xyz_bis[2][0]*xyz_bis[0][1] - xyz_bis[2][1]*xyz_bis[0][0])/2.;
			*(area_coordinates+3*i+j)=area_portion/area_init;

			/*Reinitialize xyz_list*/
			for(k=0;k<3;k++){
				/*Reinitialize xyz_list with original coordinates*/
				xyz_bis[j][k]=xyz_list[j*3+k];
			}
		}
	}
}
/*}}}*/
/*FUNCTION Penta::GetUpperPenta{{{*/
Penta* Penta::GetUpperPenta(void){

	Penta* upper_penta=NULL;

	upper_penta=(Penta*)verticalneighbors[1]; //first one (0) under, second one (1) above

	return upper_penta;
}
/*}}}*/
/*FUNCTION Penta::GetLowerPenta{{{*/
Penta* Penta::GetLowerPenta(void){

	Penta* lower_penta=NULL;

	lower_penta=(Penta*)verticalneighbors[0]; //first one (0) under, second one (1) above

	return lower_penta;
}
/*}}}*/
/*FUNCTION Penta::GetSurfacePenta{{{*/
Penta* Penta::GetSurfacePenta(void){

	/*Output*/
	Penta* penta=NULL;

	/*Go through all pentas till the surface is reached*/
	penta=this;
	for(;;){
		/*Stop if we have reached the surface, else, take upper penta*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperPenta();
		_assert_(penta->Id()!=this->id);
	}

	/*return output*/
	return penta;
}
/*}}}*/
/*FUNCTION Penta::GetBasalPenta{{{*/
Penta* Penta::GetBasalPenta(void){

	/*Output*/
	Penta* penta=NULL;

	/*Go through all pentas till the bed is reached*/
	penta=this;
	for(;;){
		/*Stop if we have reached the surface, else, take lower penta*/
		if (penta->IsOnBed()) break;

		/* get lower Penta*/
		penta=penta->GetLowerPenta();
		_assert_(penta->Id()!=this->id);
	}

	/*return output*/
	return penta;
}
/*}}}*/
/*FUNCTION Penta::GetUpperElement{{{*/
Element* Penta::GetUpperElement(void){

	/*Output*/
	Element* upper_element=this->GetUpperPenta();
	return upper_element;
}
/*}}}*/
/*FUNCTION Penta::GetLowerElement{{{*/
Element* Penta::GetLowerElement(void){

	/*Output*/
	Element* lower_element=this->GetLowerPenta();
	return lower_element;
}
/*}}}*/
/*FUNCTION Penta::GetSurfaceElement{{{*/
Element* Penta::GetSurfaceElement(void){

	/*Output*/
	Element* element=this->GetSurfacePenta();
	return element;
}
/*}}}*/
/*FUNCTION Penta::GetBasalElement{{{*/
Element* Penta::GetBasalElement(void){

	/*Output*/
	Element* element=this->GetBasalPenta();
	return element;
}
/*}}}*/
/*FUNCTION Penta::GetDofList {{{*/
void  Penta::GetDofList(int** pdoflist,int approximation_enum,int setenum){

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodes();

	/*First, figure out size of doflist and create it: */
	int numberofdofs=0;
	for(int i=0;i<numnodes;i++) numberofdofs+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);

	/*Allocate output*/
	int* doflist=xNew<int>(numberofdofs);

	/*Populate: */
	int count=0;
	for(int i=0;i<numnodes;i++){
		nodes[i]->GetDofList(doflist+count,approximation_enum,setenum);
		count+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;
}
/*}}}*/
/*FUNCTION Penta::GetDofListVelocity{{{*/
void  Penta::GetDofListVelocity(int** pdoflist,int setenum){

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodesVelocity();

	/*First, figure out size of doflist and create it: */
	int numberofdofs=0;
	for(int i=0;i<numnodes;i++) numberofdofs+=nodes[i]->GetNumberOfDofs(FSvelocityEnum,setenum);

	/*Allocate output*/
	int* doflist=xNew<int>(numberofdofs);

	/*Populate: */
	int count=0;
	for(int i=0;i<numnodes;i++){
		nodes[i]->GetDofList(doflist+count,FSvelocityEnum,setenum);
		count+=nodes[i]->GetNumberOfDofs(FSvelocityEnum,setenum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;
}
/*}}}*/
/*FUNCTION Penta::GetDofListPressure{{{*/
void  Penta::GetDofListPressure(int** pdoflist,int setenum){

	/*Fetch number of nodes and dof for this finite element*/
	int vnumnodes = this->NumberofNodesVelocity();
	int pnumnodes = this->NumberofNodesPressure();

	/*First, figure out size of doflist and create it: */
	int numberofdofs=0;
	for(int i=vnumnodes;i<vnumnodes+pnumnodes;i++) numberofdofs+=nodes[i]->GetNumberOfDofs(FSApproximationEnum,setenum);

	/*Allocate output*/
	int* doflist=xNew<int>(numberofdofs);

	/*Populate: */
	int count=0;
	for(int i=vnumnodes;i<vnumnodes+pnumnodes;i++){
		nodes[i]->GetDofList(doflist+count,FSApproximationEnum,setenum);
		count+=nodes[i]->GetNumberOfDofs(FSApproximationEnum,setenum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;
}
/*}}}*/
/*FUNCTION Penta::GetGroundedPart{{{*/
void Penta::GetGroundedPart(int* point1,IssmDouble* fraction1,IssmDouble* fraction2, bool* mainlyfloating){
	/*Computeportion of the element that is grounded*/ 

	bool               floating=true;
	int                point;
	const IssmPDouble  epsilon= 1.e-15;
	IssmDouble         gl[NUMVERTICES];
	IssmDouble         f1,f2;

	/*Recover parameters and values*/
	GetInputListOnVertices(&gl[0],MaskGroundediceLevelsetEnum);

	/*Be sure that values are not zero*/
	if(gl[0]==0.) gl[0]=gl[0]+epsilon;
	if(gl[1]==0.) gl[1]=gl[1]+epsilon;
	if(gl[2]==0.) gl[2]=gl[2]+epsilon;

	/*Check that not all nodes are grounded or floating*/
	if(gl[0]>0 && gl[1]>0 && gl[2]>0){ // All grounded
		point=0;
		f1=1.;
		f2=1.;
	}
	else if(gl[0]<0 && gl[1]<0 && gl[2]<0){ //All floating
		point=0;
		f1=0.;
		f2=0.;
	}
	else{
		if(gl[0]*gl[1]*gl[2]<0) floating=false;

		if(gl[0]*gl[1]>0){ //Nodes 0 and 1 are similar, so points must be found on segment 0-2 and 1-2
			point=2;
			f1=gl[2]/(gl[2]-gl[0]);
			f2=gl[2]/(gl[2]-gl[1]);
		}
		else if(gl[1]*gl[2]>0){ //Nodes 1 and 2 are similar, so points must be found on segment 0-1 and 0-2
			point=0;
			f1=gl[0]/(gl[0]-gl[1]);
			f2=gl[0]/(gl[0]-gl[2]);
		}
		else if(gl[0]*gl[2]>0){ //Nodes 0 and 2 are similar, so points must be found on segment 1-0 and 1-2
			point=1;
			f1=gl[1]/(gl[1]-gl[2]);
			f2=gl[1]/(gl[1]-gl[0]);
		}
	}
	*point1=point;
	*fraction1=f1;
	*fraction2=f2;
	*mainlyfloating=floating;
}
/*}}}*/
/*FUNCTION Penta::GetGroundedPortion{{{*/
IssmDouble Penta::GetGroundedPortion(IssmDouble* xyz_list){
	/*Computeportion of the element that is grounded*/ 

	bool               mainlyfloating = true;
	const IssmPDouble  epsilon= 1.e-15;
	IssmDouble         phi,s1,s2,area_init,area_grounded;
	IssmDouble         gl[NUMVERTICES];
	IssmDouble         xyz_bis[NUMVERTICES2D][3];

	/*Recover parameters and values*/
	GetInputListOnVertices(&gl[0],MaskGroundediceLevelsetEnum);

	/*Be sure that values are not zero*/
	if(gl[0]==0.) gl[0]=gl[0]+epsilon;
	if(gl[1]==0.) gl[1]=gl[1]+epsilon;
	if(gl[2]==0.) gl[2]=gl[2]+epsilon;

	/*Check that not all nodes are grounded or floating*/
	if(gl[0]>0 && gl[1]>0 && gl[2]>0){ // All grounded
		phi=1;
	}
	else if(gl[0]<0 && gl[1]<0 && gl[2]<0){ //All floating
		phi=0;
	}
	else{
		/*Figure out if two nodes are floating or grounded*/
		if(gl[0]*gl[1]*gl[2]>0) mainlyfloating=false;

		if(gl[0]*gl[1]>0){ //Nodes 0 and 1 are similar, so points must be found on segment 0-2 and 1-2
			/*Coordinates of point 2: same as initial point 2*/
			xyz_bis[2][0]=xyz_list[3*2+0];
			xyz_bis[2][1]=xyz_list[3*2+1];
			xyz_bis[2][2]=xyz_list[3*2+2];

			/*Portion of the segments*/
			s1=gl[2]/(gl[2]-gl[1]);
			s2=gl[2]/(gl[2]-gl[0]);

			/*New point 1*/
			xyz_bis[1][0]=xyz_list[3*2+0]+s1*(xyz_list[3*1+0]-xyz_list[3*2+0]);
			xyz_bis[1][1]=xyz_list[3*2+1]+s1*(xyz_list[3*1+1]-xyz_list[3*2+1]);
			xyz_bis[1][2]=xyz_list[3*2+2]+s1*(xyz_list[3*1+2]-xyz_list[3*2+2]);

			/*New point 0*/
			xyz_bis[0][0]=xyz_list[3*2+0]+s2*(xyz_list[3*0+0]-xyz_list[3*2+0]);
			xyz_bis[0][1]=xyz_list[3*2+1]+s2*(xyz_list[3*0+1]-xyz_list[3*2+1]);
			xyz_bis[0][2]=xyz_list[3*2+2]+s2*(xyz_list[3*0+2]-xyz_list[3*2+2]);
		}
		else if(gl[1]*gl[2]>0){ //Nodes 1 and 2 are similar, so points must be found on segment 0-1 and 0-2
			/*Coordinates of point 0: same as initial point 2*/
			xyz_bis[0][0]=*(xyz_list+3*0+0);
			xyz_bis[0][1]=*(xyz_list+3*0+1);
			xyz_bis[0][2]=*(xyz_list+3*0+2);

			/*Portion of the segments*/
			s1=gl[0]/(gl[0]-gl[1]);
			s2=gl[0]/(gl[0]-gl[2]);

			/*New point 1*/
			xyz_bis[1][0]=*(xyz_list+3*0+0)+s1*(*(xyz_list+3*1+0)-*(xyz_list+3*0+0));
			xyz_bis[1][1]=*(xyz_list+3*0+1)+s1*(*(xyz_list+3*1+1)-*(xyz_list+3*0+1));
			xyz_bis[1][2]=*(xyz_list+3*0+2)+s1*(*(xyz_list+3*1+2)-*(xyz_list+3*0+2));

			/*New point 2*/
			xyz_bis[2][0]=*(xyz_list+3*0+0)+s2*(*(xyz_list+3*2+0)-*(xyz_list+3*0+0));
			xyz_bis[2][1]=*(xyz_list+3*0+1)+s2*(*(xyz_list+3*2+1)-*(xyz_list+3*0+1));
			xyz_bis[2][2]=*(xyz_list+3*0+2)+s2*(*(xyz_list+3*2+2)-*(xyz_list+3*0+2));
		}
		else if(gl[0]*gl[2]>0){ //Nodes 0 and 2 are similar, so points must be found on segment 1-0 and 1-2
			/*Coordinates of point 1: same as initial point 2*/
			xyz_bis[1][0]=*(xyz_list+3*1+0);
			xyz_bis[1][1]=*(xyz_list+3*1+1);
			xyz_bis[1][2]=*(xyz_list+3*1+2);

			/*Portion of the segments*/
			s1=gl[1]/(gl[1]-gl[0]);
			s2=gl[1]/(gl[1]-gl[2]);

			/*New point 0*/
			xyz_bis[0][0]=*(xyz_list+3*1+0)+s1*(*(xyz_list+3*0+0)-*(xyz_list+3*1+0));
			xyz_bis[0][1]=*(xyz_list+3*1+1)+s1*(*(xyz_list+3*0+1)-*(xyz_list+3*1+1));
			xyz_bis[0][2]=*(xyz_list+3*1+2)+s1*(*(xyz_list+3*0+2)-*(xyz_list+3*1+2));

			/*New point 2*/
			xyz_bis[2][0]=*(xyz_list+3*1+0)+s2*(*(xyz_list+3*2+0)-*(xyz_list+3*1+0));
			xyz_bis[2][1]=*(xyz_list+3*1+1)+s2*(*(xyz_list+3*2+1)-*(xyz_list+3*1+1));
			xyz_bis[2][2]=*(xyz_list+3*1+2)+s2*(*(xyz_list+3*2+2)-*(xyz_list+3*1+2));
		}

		/*Compute fraction of grounded element*/
		GetTriaJacobianDeterminant(&area_init, xyz_list,NULL);
		GetTriaJacobianDeterminant(&area_grounded, &xyz_bis[0][0],NULL);
		if(mainlyfloating==true) area_grounded=area_init-area_grounded;
		phi=area_grounded/area_init;
	}

	if(phi>1. || phi<0.) _error_("Error. Problem with portion of grounded element: value should be between 0 and 1");

	return phi;
}
/*}}}*/
/*FUNCTION Penta::GetVertexPidList {{{*/
void  Penta::GetVertexPidList(int* doflist){

	for(int i=0;i<6;i++) doflist[i]=vertices[i]->Pid();

}
/*}}}*/
/*FUNCTION Penta::GetElementType {{{*/
int Penta::GetElementType(){

	/*return PentaRef field*/
	return this->element_type;
}
/*}}}*/
/*FUNCTION Penta::ElementSizes{{{*/
void Penta::ElementSizes(IssmDouble* hx,IssmDouble* hy,IssmDouble* hz){

	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble xmin,ymin,zmin;
	IssmDouble xmax,ymax,zmax;

	/*Get xyz list: */
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	xmin=xyz_list[0][0]; xmax=xyz_list[0][0];
	ymin=xyz_list[0][1]; ymax=xyz_list[0][1];
	zmin=xyz_list[0][2]; zmax=xyz_list[0][2];

	for(int i=1;i<NUMVERTICES;i++){
		if(xyz_list[i][0]<xmin) xmin=xyz_list[i][0];
		if(xyz_list[i][0]>xmax) xmax=xyz_list[i][0];
		if(xyz_list[i][1]<ymin) ymin=xyz_list[i][1];
		if(xyz_list[i][1]>ymax) ymax=xyz_list[i][1];
		if(xyz_list[i][2]<zmin) zmin=xyz_list[i][2];
		if(xyz_list[i][2]>zmax) zmax=xyz_list[i][2];
	}

	*hx=xmax-xmin;
	*hy=ymax-ymin;
	*hz=zmax-zmin;
}
/*}}}*/
/*FUNCTION Penta::GetNodeIndex {{{*/
int Penta::GetNodeIndex(Node* node){

	_assert_(nodes);
	int numnodes = this->NumberofNodes();

	for(int i=0;i<numnodes;i++){
		if(node==nodes[i]) return i;
	}
	_error_("Node provided not found among element nodes");

}
/*}}}*/
/*FUNCTION Penta::GetNodesSidList{{{*/
void Penta::GetNodesSidList(int* sidlist){

	_assert_(sidlist);
	_assert_(nodes);
	int numnodes = this->NumberofNodes();

	for(int i=0;i<numnodes;i++){
		sidlist[i]=nodes[i]->Sid();
	}
}
/*}}}*/
/*FUNCTION Penta::GetNodesLidList{{{*/
void Penta::GetNodesLidList(int* lidlist){

	_assert_(lidlist);
	_assert_(nodes);
	int numnodes = this->NumberofNodes();

	for(int i=0;i<numnodes;i++){
		lidlist[i]=nodes[i]->Lid();
	}
}
/*}}}*/
/*FUNCTION Penta::GetNumberOfNodes;{{{*/
int Penta::GetNumberOfNodes(void){
	return this->NumberofNodes();
}
/*}}}*/
/*FUNCTION Penta::GetNumberOfNodesPressure;{{{*/
int Penta::GetNumberOfNodesPressure(void){
	return this->NumberofNodesPressure();
}
/*}}}*/
/*FUNCTION Penta::GetNumberOfNodesVelocity;{{{*/
int Penta::GetNumberOfNodesVelocity(void){
	return this->NumberofNodesVelocity();
}
/*}}}*/
/*FUNCTION Penta::GetNumberOfVertices;{{{*/
int Penta::GetNumberOfVertices(void){
	return NUMVERTICES; 
}
/*}}}*/
/*FUNCTION Penta::GetNode(int node_number) {{{*/
Node* Penta::GetNode(int node_number){
	_assert_(node_number>=0); 
	_assert_(node_number<this->NumberofNodes()); 
	return this->nodes[node_number];
}
/*}}}*/
/*FUNCTION Penta::GetMaterialInput(int inputenum) {{{*/
Input* Penta::GetMaterialInput(int inputenum){
	return this->material->inputs->GetInput(inputenum);
}
/*}}}*/
/*FUNCTION Penta::GetInputListOnVertices(IssmDouble* pvalue,int enumtype) {{{*/
void Penta::GetInputListOnVertices(IssmDouble* pvalue,int enumtype){

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);
	if (!input) _error_("Input " << EnumToStringx(enumtype) << " not found in element");

	/*Checks in debugging mode*/
	_assert_(pvalue);

	/* Start looping on the number of vertices: */
	GaussPenta *gauss=new GaussPenta();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);
		input->GetInputValue(&pvalue[iv],gauss);
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetInputListOnVertices(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue) {{{*/
void Penta::GetInputListOnVertices(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue){

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);

	/*Checks in debugging mode*/
	_assert_(pvalue);

	/* Start looping on the number of vertices: */
	if (input){
		GaussPenta *gauss=new GaussPenta();
		for (int iv=0;iv<NUMVERTICES;iv++){
			gauss->GaussVertex(iv);
			input->GetInputValue(&pvalue[iv],gauss);
		}
		delete gauss;
	}
	else{
		for (int iv=0;iv<NUMVERTICES;iv++) pvalue[iv]=defaultvalue;
	}
}
/*}}}*/
/*FUNCTION Penta::GetInputListOnNodes(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue) {{{*/
void Penta::GetInputListOnNodes(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue){

	_assert_(pvalue);

	Input *input    = inputs->GetInput(enumtype);
	int    numnodes = this->NumberofNodes();

	/* Start looping on the number of vertices: */
	if(input){
		GaussPenta* gauss=new GaussPenta();
		for(int iv=0;iv<this->NumberofNodes();iv++){
			gauss->GaussNode(this->element_type,iv);
			input->GetInputValue(&pvalue[iv],gauss);
		}
		delete gauss;
	}
	else{
		for(int iv=0;iv<numnodes;iv++) pvalue[iv]=defaultvalue;
	}
}
/*}}}*/
/*FUNCTION Penta::GetInputListOnNodes(IssmDouble* pvalue,int enumtype) {{{*/
void Penta::GetInputListOnNodes(IssmDouble* pvalue,int enumtype){

	_assert_(pvalue);

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);
	if(!input) _error_("Input " << EnumToStringx(enumtype) << " not found in element");

	/* Start looping on the number of vertices: */
	GaussPenta* gauss=new GaussPenta();
	for(int iv=0;iv<this->NumberofNodes();iv++){
		gauss->GaussNode(this->element_type,iv);
		input->GetInputValue(&pvalue[iv],gauss);
	}
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetInputValue(IssmDouble* pvalue,Node* node,int enumtype) {{{*/
void Penta::GetInputValue(IssmDouble* pvalue,Node* node,int enumtype){

	Input* input=inputs->GetInput(enumtype);
	if(!input) _error_("No input of type " << EnumToStringx(enumtype) << " found in tria");

	GaussPenta* gauss=new GaussPenta();
	gauss->GaussVertex(this->GetNodeIndex(node));

	input->GetInputValue(pvalue,gauss);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GetInputValue(bool* pvalue,int inputenum) {{{*/
void Penta::GetInputValue(bool* pvalue,int inputenum){

	Input* input=inputs->GetInput(inputenum);
	if(!input) _error_("Input " << EnumToStringx(inputenum) << " not found in element");
	input->GetInputValue(pvalue);

}/*}}}*/
/*FUNCTION Penta::GetInputValue(int* pvalue,int inputenum) {{{*/
void Penta::GetInputValue(int* pvalue,int inputenum){

	Input* input=inputs->GetInput(inputenum);
	if(!input) _error_("Input " << EnumToStringx(inputenum) << " not found in element");
	input->GetInputValue(pvalue);

}/*}}}*/
/*FUNCTION Penta::GetInputValue(IssmDouble* pvalue,int inputenum) {{{*/
void Penta::GetInputValue(IssmDouble* pvalue,int inputenum){

	Input* input=inputs->GetInput(inputenum);
	if(!input) _error_("Input " << EnumToStringx(inputenum) << " not found in element");
	input->GetInputValue(pvalue);

}/*}}}*/
/*FUNCTION Penta::GetInputValue(IssmDouble* pvalue,Gauss* gauss,int inputenum) {{{*/
void Penta::GetInputValue(IssmDouble* pvalue,Gauss* gauss,int inputenum){

	Input* input=inputs->GetInput(inputenum);
	if(!input) _error_("Input " << EnumToStringx(inputenum) << " not found in element");
	input->GetInputValue(pvalue,gauss);

}/*}}}*/
/*FUNCTION Penta::GetVerticesCoordinates(IssmDouble** pxyz_list){{{*/
void Penta::GetVerticesCoordinates(IssmDouble** pxyz_list){

	IssmDouble* xyz_list = xNew<IssmDouble>(NUMVERTICES*3);
	::GetVerticesCoordinates(xyz_list,this->vertices,NUMVERTICES);

	/*Assign output pointer*/
	*pxyz_list = xyz_list;

}/*}}}*/
/*FUNCTION Penta::GetVerticesCoordinatesBase(IssmDouble** pxyz_list){{{*/
void Penta::GetVerticesCoordinatesBase(IssmDouble** pxyz_list){

	IssmDouble* xyz_list = xNew<IssmDouble>(NUMVERTICES2D*3);
	::GetVerticesCoordinates(xyz_list,this->vertices,NUMVERTICES2D);

	/*Assign output pointer*/
	*pxyz_list = xyz_list;

}/*}}}*/
/*FUNCTION Penta::GetVerticesCoordinatesTop(IssmDouble** pxyz_list){{{*/
void Penta::GetVerticesCoordinatesTop(IssmDouble** pxyz_list){

	IssmDouble* xyz_list = xNew<IssmDouble>(NUMVERTICES2D*3);
	::GetVerticesCoordinates(xyz_list,&this->vertices[3],NUMVERTICES2D);

	/*Assign output pointer*/
	*pxyz_list = xyz_list;

}/*}}}*/
/*FUNCTION Penta::GetMaterialInputValue(IssmDouble* pvalue,Node* node,int enumtype) {{{*/
void Penta::GetMaterialInputValue(IssmDouble* pvalue,Node* node,int enumtype){

	Input* input=this->material->inputs->GetInput(enumtype);
	if(!input) _error_("No input of type " << EnumToStringx(enumtype) << " found in tria");

	GaussPenta* gauss=new GaussPenta();
	gauss->GaussVertex(this->GetNodeIndex(node));

	input->GetInputValue(pvalue,gauss);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::NormalSection{{{*/
void Penta::NormalSection(IssmDouble* normal,IssmDouble* xyz_list){

	/*Build unit outward pointing vector*/
	IssmDouble AB[3];
	IssmDouble AC[3];
	IssmDouble norm;

	AB[0]=xyz_list[1*3+0] - xyz_list[0*3+0];
	AB[1]=xyz_list[1*3+1] - xyz_list[0*3+1];
	AB[2]=xyz_list[1*3+2] - xyz_list[0*3+2];
	AC[0]=xyz_list[2*3+0] - xyz_list[0*3+0];
	AC[1]=xyz_list[2*3+1] - xyz_list[0*3+1];
	AC[2]=xyz_list[2*3+2] - xyz_list[0*3+2];

	cross(normal,AB,AC);
	norm=sqrt(pow(normal[0],2.0)+pow(normal[1],2.0)+pow(normal[2],2.0));

	for(int i=0;i<3;i++) normal[i]=normal[i]/norm;
}
/*}}}*/
/*FUNCTION Penta::StabilizationParameter {{{*/
IssmDouble Penta::StabilizationParameter(IssmDouble u, IssmDouble v, IssmDouble w, IssmDouble diameter, IssmDouble kappa){
	/*Compute stabilization parameter*/
	/*kappa=thermalconductivity/(rho_ice*hearcapacity) for thermal model*/
	/*kappa=enthalpydiffusionparameter for enthalpy model*/

	IssmDouble normu;
	IssmDouble tau_parameter;

	normu=pow(pow(u,2)+pow(v,2)+pow(w,2),0.5);
	if(normu*diameter/(3*2*kappa)<1){ 
		tau_parameter=pow(diameter,2)/(3*2*2*kappa);
	}
	else tau_parameter=diameter/(2*normu);

	return tau_parameter;
}
/*}}}*/
/*FUNCTION Penta::GetVectorFromInputs{{{*/
void  Penta::GetVectorFromInputs(Vector<IssmDouble>* vector,int input_enum){

	int vertexpidlist[NUMVERTICES];

	/*Get out if this is not an element input*/
	if (!IsInput(input_enum)) return;

	/*Prepare index list*/
	this->GetVertexPidList(&vertexpidlist[0]);

	/*Get input (either in element or material)*/
	Input* input=inputs->GetInput(input_enum);
	if(!input) _error_("Input " << EnumToStringx(input_enum) << " not found in element");

	/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
	input->GetVectorFromInputs(vector,&vertexpidlist[0]);
}
/*}}}*/
/*FUNCTION Penta::GetXcoord {{{*/
IssmDouble Penta::GetXcoord(Gauss* gauss){

	int    i;
	IssmDouble x;
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble x_list[NUMVERTICES];

	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<NUMVERTICES;i++) x_list[i]=xyz_list[i][0];
	PentaRef::GetInputValue(&x,x_list,gauss,P1Enum);

	return x;
}
/*}}}*/
/*FUNCTION Penta::GetYcoord {{{*/
IssmDouble Penta::GetYcoord(Gauss* gauss){

	int    i;
	IssmDouble y;
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble y_list[NUMVERTICES];

	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<NUMVERTICES;i++) y_list[i]=xyz_list[i][1];
	PentaRef::GetInputValue(&y,y_list,gauss,P1Enum);

	return y;
}
/*}}}*/
/*FUNCTION Penta::GetZcoord {{{*/
IssmDouble Penta::GetZcoord(Gauss* gauss){

	int    i;
	IssmDouble z;
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble z_list[NUMVERTICES];

	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<NUMVERTICES;i++) z_list[i]=xyz_list[i][2];
	PentaRef::GetInputValue(&z,z_list,gauss,P1Enum);

	return z;
}
/*}}}*/
/*FUNCTION Penta::ZeroLevelsetCoordinates{{{*/
void Penta::ZeroLevelsetCoordinates(IssmDouble** pxyz_zero,IssmDouble* xyz_list,int levelsetenum){
	/*Compute portion of the element that is grounded*/ 

	int         normal_orientation=0;
	IssmDouble  s1,s2;
	IssmDouble  levelset[NUMVERTICES];
	IssmDouble* xyz_zero = xNew<IssmDouble>(4*3);

	/*Recover parameters and values*/
	GetInputListOnVertices(&levelset[0],levelsetenum);

	if(levelset[0]*levelset[1]>0.){ //Nodes 0 and 1 are similar, so points must be found on segment 0-2 and 1-2
		/*Portion of the segments*/
		s1=levelset[2]/(levelset[2]-levelset[1]);
		s2=levelset[2]/(levelset[2]-levelset[0]);

		if(levelset[2]<0.) normal_orientation=1; //orientation of quadrangle at base and top, depending on distribution of levelsetfunction
		/*New point 1*/
		xyz_zero[3*normal_orientation+0]=xyz_list[2*3+0]+s1*(xyz_list[1*3+0]-xyz_list[2*3+0]);
		xyz_zero[3*normal_orientation+1]=xyz_list[2*3+1]+s1*(xyz_list[1*3+1]-xyz_list[2*3+1]);
		xyz_zero[3*normal_orientation+2]=xyz_list[2*3+2]+s1*(xyz_list[1*3+2]-xyz_list[2*3+2]);

		/*New point 0*/
		xyz_zero[3*(1-normal_orientation)+0]=xyz_list[2*3+0]+s2*(xyz_list[0*3+0]-xyz_list[2*3+0]);
		xyz_zero[3*(1-normal_orientation)+1]=xyz_list[2*3+1]+s2*(xyz_list[0*3+1]-xyz_list[2*3+1]);
		xyz_zero[3*(1-normal_orientation)+2]=xyz_list[2*3+2]+s2*(xyz_list[0*3+2]-xyz_list[2*3+2]);

		/*New point 3*/
		xyz_zero[3*(2+1-normal_orientation)+0]=xyz_list[5*3+0]+s1*(xyz_list[4*3+0]-xyz_list[5*3+0]);
		xyz_zero[3*(2+1-normal_orientation)+1]=xyz_list[5*3+1]+s1*(xyz_list[4*3+1]-xyz_list[5*3+1]);
		xyz_zero[3*(2+1-normal_orientation)+2]=xyz_list[5*3+2]+s1*(xyz_list[4*3+2]-xyz_list[5*3+2]);

		/*New point 4*/
		xyz_zero[3*(2+normal_orientation)+0]=xyz_list[5*3+0]+s2*(xyz_list[3*3+0]-xyz_list[5*3+0]);
		xyz_zero[3*(2+normal_orientation)+1]=xyz_list[5*3+1]+s2*(xyz_list[3*3+1]-xyz_list[5*3+1]);
		xyz_zero[3*(2+normal_orientation)+2]=xyz_list[5*3+2]+s2*(xyz_list[3*3+2]-xyz_list[5*3+2]);
	}
	else if(levelset[1]*levelset[2]>0.){ //Nodes 1 and 2 are similar, so points must be found on segment 0-1 and 0-2
		/*Portion of the segments*/
		s1=levelset[0]/(levelset[0]-levelset[2]);
		s2=levelset[0]/(levelset[0]-levelset[1]);

		if(levelset[0]<0.) normal_orientation=1;
		/*New point 1*/
		xyz_zero[3*normal_orientation+0]=xyz_list[0*3+0]+s1*(xyz_list[2*3+0]-xyz_list[0*3+0]);
		xyz_zero[3*normal_orientation+1]=xyz_list[0*3+1]+s1*(xyz_list[2*3+1]-xyz_list[0*3+1]);
		xyz_zero[3*normal_orientation+2]=xyz_list[0*3+2]+s1*(xyz_list[2*3+2]-xyz_list[0*3+2]);

		/*New point 2*/
		xyz_zero[3*(1-normal_orientation)+0]=xyz_list[0*3+0]+s2*(xyz_list[1*3+0]-xyz_list[0*3+0]);
		xyz_zero[3*(1-normal_orientation)+1]=xyz_list[0*3+1]+s2*(xyz_list[1*3+1]-xyz_list[0*3+1]);
		xyz_zero[3*(1-normal_orientation)+2]=xyz_list[0*3+2]+s2*(xyz_list[1*3+2]-xyz_list[0*3+2]);

		/*New point 3*/
		xyz_zero[3*(2+1-normal_orientation)+0]=xyz_list[3*3+0]+s1*(xyz_list[5*3+0]-xyz_list[3*3+0]);
		xyz_zero[3*(2+1-normal_orientation)+1]=xyz_list[3*3+1]+s1*(xyz_list[5*3+1]-xyz_list[3*3+1]);
		xyz_zero[3*(2+1-normal_orientation)+2]=xyz_list[3*3+2]+s1*(xyz_list[5*3+2]-xyz_list[3*3+2]);

		/*New point 4*/
		xyz_zero[3*(2+normal_orientation)+0]=xyz_list[3*3+0]+s2*(xyz_list[4*3+0]-xyz_list[3*3+0]);
		xyz_zero[3*(2+normal_orientation)+1]=xyz_list[3*3+1]+s2*(xyz_list[4*3+1]-xyz_list[3*3+1]);
		xyz_zero[3*(2+normal_orientation)+2]=xyz_list[3*3+2]+s2*(xyz_list[4*3+2]-xyz_list[3*3+2]);
	}
	else if(levelset[0]*levelset[2]>0.){ //Nodes 0 and 2 are similar, so points must be found on segment 1-0 and 1-2
		/*Portion of the segments*/
		s1=levelset[1]/(levelset[1]-levelset[0]);
		s2=levelset[1]/(levelset[1]-levelset[2]);

		if(levelset[1]<0.) normal_orientation=1;
		/*New point 0*/
		xyz_zero[3*normal_orientation+0]=xyz_list[1*3+0]+s1*(xyz_list[0*3+0]-xyz_list[1*3+0]);
		xyz_zero[3*normal_orientation+1]=xyz_list[1*3+1]+s1*(xyz_list[0*3+1]-xyz_list[1*3+1]);
		xyz_zero[3*normal_orientation+2]=xyz_list[1*3+2]+s1*(xyz_list[0*3+2]-xyz_list[1*3+2]);

		/*New point 2*/
		xyz_zero[3*(1-normal_orientation)+0]=xyz_list[1*3+0]+s2*(xyz_list[2*3+0]-xyz_list[1*3+0]);
		xyz_zero[3*(1-normal_orientation)+1]=xyz_list[1*3+1]+s2*(xyz_list[2*3+1]-xyz_list[1*3+1]);
		xyz_zero[3*(1-normal_orientation)+2]=xyz_list[1*3+2]+s2*(xyz_list[2*3+2]-xyz_list[1*3+2]);

		/*New point 3*/
		xyz_zero[3*(2+1-normal_orientation)+0]=xyz_list[4*3+0]+s1*(xyz_list[3*3+0]-xyz_list[4*3+0]);
		xyz_zero[3*(2+1-normal_orientation)+1]=xyz_list[4*3+1]+s1*(xyz_list[3*3+1]-xyz_list[4*3+1]);
		xyz_zero[3*(2+1-normal_orientation)+2]=xyz_list[4*3+2]+s1*(xyz_list[3*3+2]-xyz_list[4*3+2]);

		/*New point 4*/
		xyz_zero[3*(2+normal_orientation)+0]=xyz_list[4*3+0]+s2*(xyz_list[5*3+0]-xyz_list[4*3+0]);
		xyz_zero[3*(2+normal_orientation)+1]=xyz_list[4*3+1]+s2*(xyz_list[5*3+1]-xyz_list[4*3+1]);
		xyz_zero[3*(2+normal_orientation)+2]=xyz_list[4*3+2]+s2*(xyz_list[5*3+2]-xyz_list[4*3+2]);
	}
	else if(levelset[0]==0. && levelset[1]==0.){ //front is on point 0 and 1
		xyz_zero[3*0+0]=xyz_list[0*3+0];
		xyz_zero[3*0+1]=xyz_list[0*3+1];
		xyz_zero[3*0+2]=xyz_list[0*3+2];

		/*New point 2*/
		xyz_zero[3*1+0]=xyz_list[1*3+0];
		xyz_zero[3*1+1]=xyz_list[1*3+1];
		xyz_zero[3*1+2]=xyz_list[1*3+2];

		/*New point 3*/
		xyz_zero[3*2+0]=xyz_list[4*3+0];
		xyz_zero[3*2+1]=xyz_list[4*3+1];
		xyz_zero[3*2+2]=xyz_list[4*3+2];

		/*New point 4*/
		xyz_zero[3*3+0]=xyz_list[3*3+0];
		xyz_zero[3*3+1]=xyz_list[3*3+1];
		xyz_zero[3*3+2]=xyz_list[3*3+2];
	}
	else if(levelset[0]==0. && levelset[2]==0.){ //front is on point 0 and 1
		xyz_zero[3*0+0]=xyz_list[2*3+0];
		xyz_zero[3*0+1]=xyz_list[2*3+1];
		xyz_zero[3*0+2]=xyz_list[2*3+2];

		/*New point 2*/
		xyz_zero[3*1+0]=xyz_list[0*3+0];
		xyz_zero[3*1+1]=xyz_list[0*3+1];
		xyz_zero[3*1+2]=xyz_list[0*3+2];

		/*New point 3*/
		xyz_zero[3*2+0]=xyz_list[3*3+0];
		xyz_zero[3*2+1]=xyz_list[3*3+1];
		xyz_zero[3*2+2]=xyz_list[3*3+2];

		/*New point 4*/
		xyz_zero[3*3+0]=xyz_list[5*3+0];
		xyz_zero[3*3+1]=xyz_list[5*3+1];
		xyz_zero[3*3+2]=xyz_list[5*3+2];
	}
	else if(levelset[1]==0. && levelset[2]==0.){ //front is on point 0 and 1
		xyz_zero[3*0+0]=xyz_list[1*3+0];
		xyz_zero[3*0+1]=xyz_list[1*3+1];
		xyz_zero[3*0+2]=xyz_list[1*3+2];

		/*New point 2*/
		xyz_zero[3*1+0]=xyz_list[2*3+0];
		xyz_zero[3*1+1]=xyz_list[2*3+1];
		xyz_zero[3*1+2]=xyz_list[2*3+2];

		/*New point 3*/
		xyz_zero[3*2+0]=xyz_list[5*3+0];
		xyz_zero[3*2+1]=xyz_list[5*3+1];
		xyz_zero[3*2+2]=xyz_list[5*3+2];

		/*New point 4*/
		xyz_zero[3*3+0]=xyz_list[4*3+0];
		xyz_zero[3*3+1]=xyz_list[4*3+1];
		xyz_zero[3*3+2]=xyz_list[4*3+2];
	}
	else _error_("Case not covered");

	/*Assign output pointer*/
	*pxyz_zero= xyz_zero;
}
/*}}}*/
/*FUNCTION Penta::Sid {{{*/
int    Penta::Sid(){

	return sid;

}
/*}}}*/
/*FUNCTION Penta::Id {{{*/
int    Penta::Id(void){
	return id; 
}
/*}}}*/
/*FUNCTION Penta::InputChangeName{{{*/
void  Penta::InputChangeName(int original_enum,int new_enum){

	/*Call inputs method*/
	this->inputs->ChangeEnum(original_enum,new_enum);
}
/*}}}*/
/*FUNCTION Penta::InputCreate(IssmDouble* vector,IoModel* iomodel,int M,int N,int vector_type,int vector_enum,int code){{{*/
void Penta::InputCreate(IssmDouble* vector,IoModel* iomodel,int M,int N,int vector_type,int vector_enum,int code){

	/*Intermediaries*/
	int             i,t;
	int             penta_vertex_ids[6];
	int             row;
	IssmDouble      nodeinputs[6];
	IssmDouble      time;
	TransientInput *transientinput      = NULL;

	/*Branch on type of vector: nodal or elementary: */
	if(vector_type==1){ //nodal vector

		/*Recover vertices ids needed to initialize inputs*/
		for(i=0;i<6;i++){ 
			_assert_(iomodel->elements);
			penta_vertex_ids[i]=iomodel->elements[6*this->sid+i]; //ids for vertices are in the elements array from Matlab
		}

		/*Are we in transient or static? */
		if(M==iomodel->numberofvertices){

			/*create input values: */
			for(i=0;i<6;i++)nodeinputs[i]=(IssmDouble)vector[penta_vertex_ids[i]-1];

			/*create static input: */
			this->inputs->AddInput(new PentaInput(vector_enum,nodeinputs,P1Enum));
		}
		else if(M==iomodel->numberofvertices+1){
			/*create transient input: */
			IssmDouble* times = xNew<IssmDouble>(N);
			for(t=0;t<N;t++) times[t] = vector[(M-1)*N+t];
			transientinput=new TransientInput(vector_enum,times,N);
			for(t=0;t<N;t++){
				for(i=0;i<NUMVERTICES;i++) nodeinputs[i]=vector[N*(penta_vertex_ids[i]-1)+t];
				transientinput->AddTimeInput(new PentaInput(vector_enum,nodeinputs,P1Enum));
			}
			this->inputs->AddInput(transientinput);
			xDelete<IssmDouble>(times);
		}
		else _error_("nodal vector is either numberofvertices (" << iomodel->numberofvertices << "), or numberofvertices+1 long. Field provided is " << M << " long. Enum " << EnumToStringx(vector_enum));
	}
	else if(vector_type==2){ //element vector
		/*Are we in transient or static? */
		if(M==iomodel->numberofelements){

			/*static mode: create an input out of the element value: */

			if (code==5){ //boolean
				this->inputs->AddInput(new BoolInput(vector_enum,reCast<bool,IssmDouble>(vector[this->sid])));
			}
			else if (code==6){ //integer
				this->inputs->AddInput(new IntInput(vector_enum,reCast<int,IssmDouble>(vector[this->sid])));
			}
			else if (code==7){ //IssmDouble
				this->inputs->AddInput(new DoubleInput(vector_enum,vector[this->sid]));
			}
			else _error_("could not recognize nature of vector from code " << code);
		}
		else {
			_error_("transient elementary inputs not supported yet!");
		}
	}
	else{
		_error_("Cannot add input for vector type " << vector_type << " (not supported)");
	}

}
/*}}}*/
/*FUNCTION Penta::InputDepthAverageAtBase{{{*/
void  Penta::InputDepthAverageAtBase(int enum_type,int average_enum_type,int object_enum){

	int  step,i;
	IssmDouble  xyz_list[NUMVERTICES][3];
	IssmDouble  Helem_list[NUMVERTICES];
	IssmDouble  zeros_list[NUMVERTICES]={0.0};
	Penta* penta=NULL;
	Input* original_input=NULL;
	Input* element_integrated_input=NULL;
	Input* total_integrated_input=NULL;
	Input* element_thickness_input=NULL;
	Input* total_thickness_input=NULL;
	Input* depth_averaged_input=NULL;

	/*recover parameters: */

	/*Are we on the base? If not, return*/
	if(!IsOnBed()) return;

	/*OK, we are on bed. Initialize global inputs as 0*/
	total_thickness_input =new PentaInput(ThicknessEnum,zeros_list,P1Enum);

	/*Now follow all the upper element from the base to the surface to integrate the input*/
	penta=this;
	step =0;
	for(;;){

		/*Step1: Get original input (to be depth avegaged): */
		if (object_enum==MeshElementsEnum)
		 original_input=(Input*)penta->inputs->GetInput(enum_type);
		else if (object_enum==MaterialsEnum)
		 original_input=(Input*)penta->material->inputs->GetInput(enum_type);
		else
		 _error_("object " << EnumToStringx(object_enum) << " not supported yet");
		if(!original_input) _error_("could not find input with enum " << EnumToStringx(enum_type));

		/*If first time, initialize total_integrated_input*/
		if (step==0){
			if (original_input->ObjectEnum()==PentaInputEnum)
			 total_integrated_input=new PentaInput(average_enum_type,zeros_list,P1Enum);
			else if (original_input->ObjectEnum()==ControlInputEnum)
			 total_integrated_input=new PentaInput(average_enum_type,zeros_list,P1Enum);
			else if (original_input->ObjectEnum()==DoubleInputEnum)
			 total_integrated_input=new DoubleInput(average_enum_type,0.0);
			else{
			 _error_("object " << EnumToStringx(original_input->ObjectEnum()) << " not supported yet");
			}
		}

		/*Step2: Create element thickness input*/
		::GetVerticesCoordinates(&xyz_list[0][0],penta->vertices,NUMVERTICES);
		for(i=0;i<3;i++){
			Helem_list[i]=xyz_list[i+3][2]-xyz_list[i][2];
			Helem_list[i+3]=Helem_list[i];
		}
		element_thickness_input=new PentaInput(ThicknessEnum,Helem_list,P1Enum);

		/*Step3: Vertically integrate A COPY of the original*/
		element_integrated_input=(Input*)original_input->copy();
		element_integrated_input->VerticallyIntegrate(element_thickness_input);

		/*Add contributions to global inputs*/
		total_integrated_input->AXPY(element_integrated_input,1.0);
		total_thickness_input ->AXPY(element_thickness_input,1.0);

		/*Clean up*/
		delete element_thickness_input;
		delete element_integrated_input;

		/*Stop if we have reached the surface, else, take upper penta*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperPenta();
		_assert_(penta->Id()!=this->id);

		/*increase couter*/
		step++;
	}

	/*OK, now we only need to divide the depth integrated input by the total thickness!*/
	depth_averaged_input=total_integrated_input->PointwiseDivide(total_thickness_input);
	depth_averaged_input->ChangeEnum(average_enum_type);

	/*Clean up*/
	delete total_thickness_input;
	delete total_integrated_input;

	/*Finally, add to inputs*/
	if (object_enum==MeshElementsEnum)
	 this->inputs->AddInput((Input*)depth_averaged_input);
	else if (object_enum==MaterialsEnum)
	 this->material->inputs->AddInput((Input*)depth_averaged_input);
	else
	 _error_("object " << EnumToStringx(object_enum) << " not supported yet");
}
/*}}}*/
/*FUNCTION Penta::InputDuplicate{{{*/
void  Penta::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Penta::InputExtrude {{{*/
void  Penta::InputExtrude(int enum_type,int object_type){

	int     i,num_inputs;
	Penta  *penta       = NULL;
	Input  *copy        = NULL;
	Input **base_inputs = NULL;

	/*Are we on the base, not on the surface?:*/
	if(!IsOnBed()) return;

	/*Step1: Get and Extrude original input: */
	if(object_type==ElementEnum){
		num_inputs=1;
		base_inputs=xNew<Input*>(num_inputs);
		base_inputs[0]=(Input*)this->inputs->GetInput(enum_type);
	}
	else if(object_type==MaterialsEnum){
		num_inputs=1;
		base_inputs=xNew<Input*>(num_inputs);
		base_inputs[0]=(Input*)material->inputs->GetInput(enum_type);
	}
	else{
		_error_("object of type " << EnumToStringx(object_type) << " not supported yet");
	}
	for(i=0;i<num_inputs;i++){
		if(!base_inputs[i]) _error_("could not find input with enum " << EnumToStringx(enum_type) << " in object " << EnumToStringx(object_type));
		base_inputs[i]->Extrude();
	}

	/*Stop if there is only one layer of element*/
	if (this->IsOnSurface()) return;

	/*Step 2: this input has been extruded for this element, now follow the upper element*/
	penta=this;
	for(;;){
		/* get upper Penta*/
		penta=penta->GetUpperPenta();
		_assert_(penta->Id()!=this->id);

		/*Add input of the basal element to penta->inputs*/
		for(i=0;i<num_inputs;i++){
			copy=(Input*)base_inputs[i]->copy();
			if (object_type==ElementEnum){
				penta->inputs->AddInput((Input*)copy);
			}
			else if(object_type==MaterialsEnum){
				penta->material->inputs->AddInput((Input*)copy);
			}
			else{
				_error_("object of type " << EnumToStringx(object_type) << " not supported yet");
			}
		}

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;
	}

	/*clean-up and return*/
	xDelete<Input*>(base_inputs);
}
/*}}}*/
/*FUNCTION Penta::InputScale{{{*/
void  Penta::InputScale(int enum_type,IssmDouble scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)_error_("could not find old input with enum: " << EnumToStringx(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(bool value, int name);{{{*/
void  Penta::InputUpdateFromConstant(bool constant, int name){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(IssmDouble value, int name);{{{*/
void  Penta::InputUpdateFromConstant(IssmDouble constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromConstant(int value, int name);{{{*/
void  Penta::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromIoModel {{{*/
void Penta::InputUpdateFromIoModel(int index,IoModel* iomodel){ 

	/*Intermediaries*/
	IssmInt i,j;
	int     penta_vertex_ids[6];
	IssmDouble  nodeinputs[6];
	IssmDouble  cmmininputs[6];
	IssmDouble  cmmaxinputs[6];

	IssmDouble  yts;
	bool    control_analysis;
	int     num_control_type,num_responses;

	/*Fetch parameters: */
	iomodel->Constant(&yts,ConstantsYtsEnum);
	iomodel->Constant(&control_analysis,InversionIscontrolEnum);
	if(control_analysis) iomodel->Constant(&num_control_type,InversionNumControlParametersEnum);
	if(control_analysis) iomodel->Constant(&num_responses,InversionNumCostFunctionsEnum);

	/*Checks if debuging*/
	/*{{{*/
	_assert_(iomodel->elements);
	/*}}}*/

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<6;i++){ 
		penta_vertex_ids[i]=iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Control Inputs*/
	if (control_analysis && iomodel->Data(InversionControlParametersEnum)){
		for(i=0;i<num_control_type;i++){
			switch(reCast<int,IssmDouble>(iomodel->Data(InversionControlParametersEnum)[i])){
				case BalancethicknessThickeningRateEnum:
					if (iomodel->Data(BalancethicknessThickeningRateEnum)){
						for(j=0;j<6;j++)nodeinputs[j]=iomodel->Data(BalancethicknessThickeningRateEnum)[penta_vertex_ids[j]-1]/yts;
						for(j=0;j<6;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						for(j=0;j<6;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						this->inputs->AddInput(new ControlInput(BalancethicknessThickeningRateEnum,PentaInputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
					}
					break;
				case VxEnum:
					if (iomodel->Data(VxEnum)){
						for(j=0;j<6;j++)nodeinputs[j]=iomodel->Data(VxEnum)[penta_vertex_ids[j]-1]/yts;
						for(j=0;j<6;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						for(j=0;j<6;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						this->inputs->AddInput(new ControlInput(VxEnum,PentaInputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
					}
					break;
				case VyEnum:
					if (iomodel->Data(VyEnum)){
						for(j=0;j<6;j++)nodeinputs[j]=iomodel->Data(VyEnum)[penta_vertex_ids[j]-1]/yts;
						for(j=0;j<6;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						for(j=0;j<6;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i]/yts;
						this->inputs->AddInput(new ControlInput(VyEnum,PentaInputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
					}
					break;
				case FrictionCoefficientEnum:
					if (iomodel->Data(FrictionCoefficientEnum)){
						for(j=0;j<6;j++)nodeinputs[j]=iomodel->Data(FrictionCoefficientEnum)[penta_vertex_ids[j]-1];
						for(j=0;j<6;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i];
						for(j=0;j<6;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[(penta_vertex_ids[j]-1)*num_control_type+i];
						this->inputs->AddInput(new ControlInput(FrictionCoefficientEnum,PentaInputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
					}
					break;
				/*Material will take care of it*/ 
				case MaterialsRheologyBbarEnum: break;
				case DamageDbarEnum:break;
				default:
					_error_("Control " << EnumToStringx(reCast<int,IssmDouble>(iomodel->Data(InversionControlParametersEnum)[i])) << " not implemented yet");
			}
		}
	}

	/*Need to know the type of approximation for this element*/
	if(iomodel->Data(FlowequationElementEquationEnum)){
		this->inputs->AddInput(new IntInput(ApproximationEnum,reCast<int>(iomodel->Data(FlowequationElementEquationEnum)[index])));
	}

	/*DatasetInputs*/
	if (control_analysis && iomodel->Data(InversionCostFunctionsCoefficientsEnum)) {

		/*Create inputs and add to DataSetInput*/
		DatasetInput* datasetinput=new DatasetInput(InversionCostFunctionsCoefficientsEnum);
		for(i=0;i<num_responses;i++){
			for(j=0;j<6;j++)nodeinputs[j]=iomodel->Data(InversionCostFunctionsCoefficientsEnum)[(penta_vertex_ids[j]-1)*num_responses+i];
			datasetinput->AddInput(new PentaInput(InversionCostFunctionsCoefficientsEnum,nodeinputs,P1Enum),reCast<int>(iomodel->Data(InversionCostFunctionsEnum)[i]));
		}

		/*Add datasetinput to element inputs*/
		this->inputs->AddInput(datasetinput);
	}
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDof{{{*/
void  Penta::InputUpdateFromSolutionOneDof(IssmDouble* solution,int enum_type){

	/*Intermediary*/
	int* doflist = NULL;

	/*Fetch number of nodes for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Fetch dof list and allocate solution vector*/
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numnodes;i++){
		values[i]=solution[doflist[i]];
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
	}

	/*Add input to the element: */
	this->inputs->AddInput(new PentaInput(enum_type,values,P1Enum));

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<int>(doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromSolutionOneDofCollpased{{{*/
void  Penta::InputUpdateFromSolutionOneDofCollapsed(IssmDouble* solution,int enum_type){

	const int  numdof   = NDOF1*NUMVERTICES;
	const int  numdof2d = NDOF1*NUMVERTICES2D;

	IssmDouble  values[numdof];
	int*    doflist = NULL;
	Penta  *penta   = NULL;

	/*If not on bed, return*/
	if (!IsOnBed()) return;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector and extrude it */
	for(int i=0;i<numdof2d;i++){
		values[i]         =solution[doflist[i]];
		values[i+numdof2d]=values[i];
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
	}

	/*Start looping over all elements above current element and update all inputs*/
	penta=this;
	for(;;){
		/*Add input to the element: */
		penta->inputs->AddInput(new PentaInput(enum_type,values,P1Enum));

		/*Stop if we have reached the surface*/
		if (penta->IsOnSurface()) break;

		/* get upper Penta*/
		penta=penta->GetUpperPenta(); _assert_(penta->Id()!=this->id);
	}

	/*Free ressources:*/
	xDelete<int>(doflist);
}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromVector(IssmDouble* vector, int name, int type);{{{*/
void  Penta::InputUpdateFromVector(IssmDouble* vector, int name, int type){

	const int   numdof         = NDOF1 *NUMVERTICES;
	int        *doflist        = NULL;
	IssmDouble  values[numdof];

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){
		case VertexPIdEnum: 
			for (int i=0;i<NUMVERTICES;i++){
				values[i]=vector[this->vertices[i]->Pid()];
			}
			/*update input*/
			this->inputs->AddInput(new PentaInput(name,values,P1Enum));
			return;

		case VertexSIdEnum: 
			for (int i=0;i<NUMVERTICES;i++){
				values[i]=vector[this->vertices[i]->Sid()];
			}
			/*update input*/
			this->inputs->AddInput(new PentaInput(name,values,P1Enum));
			return;

		case NodesEnum:
			/*Get dof list: */
			GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

			/*Use the dof list to index into the vector: */
			for(int i=0;i<numdof;i++){
				values[i]=vector[doflist[i]];
				if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
			}
			/*Add input to the element: */
			this->inputs->AddInput(new PentaInput(name,values,P1Enum));

			/*Free ressources:*/
			xDelete<int>(doflist);
			return;

	  case NodeSIdEnum:
			for(int i=0;i<NUMVERTICES;i++){
				values[i]=vector[nodes[i]->Sid()];
				if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
			}
			/*Add input to the element: */
			this->inputs->AddInput(new PentaInput(name,values,P1Enum));

			/*Free ressources:*/
			xDelete<int>(doflist);
			return;

	  default:
			_error_("type " << type << " (" << EnumToStringx(type) << ") not implemented yet");
	}
}
/*}}}*/
/*FUNCTION Penta::IsOnBed{{{*/
bool Penta::IsOnBed(void){

	bool onbed;
	inputs->GetInputValue(&onbed,MeshElementonbedEnum);
	return onbed;
}
/*}}}*/
/*FUNCTION Penta::IsOnSurface{{{*/
bool Penta::IsOnSurface(void){

	bool onsurface;
	inputs->GetInputValue(&onsurface,MeshElementonsurfaceEnum);
	return onsurface;
}
/*}}}*/
/*FUNCTION Penta::IsInput{{{*/
bool Penta::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==BathymetryEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==SurfaceforcingsMassBalanceEnum ||
				name==BasalforcingsMeltingRateEnum ||
				name==BasalforcingsGeothermalfluxEnum ||
				name==SurfaceAreaEnum||
				name==PressureEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==VzEnum ||
				name==VxMeshEnum ||
				name==VyMeshEnum ||
				name==VzMeshEnum ||
				name==InversionVxObsEnum ||
				name==InversionVyObsEnum ||
				name==InversionVzObsEnum ||
				name==TemperatureEnum ||
				name==EnthalpyEnum ||
				name==EnthalpyPicardEnum ||
				name==WaterfractionEnum||
				name==FrictionCoefficientEnum ||
				name==MaskGroundediceLevelsetEnum ||
				name==MaskIceLevelsetEnum ||
				name==GradientEnum ||
				name==OldGradientEnum  ||
				name==ConvergedEnum || 
				name==QmuVxEnum ||
				name==QmuVyEnum ||
				name==QmuVzEnum ||
				name==QmuVxMeshEnum ||
				name==QmuVyMeshEnum ||
				name==QmuVzMeshEnum ||
				name==QmuPressureEnum ||
				name==QmuBedEnum ||
				name==QmuThicknessEnum ||
				name==QmuSurfaceEnum ||
				name==QmuTemperatureEnum ||
				name==QmuMeltingEnum ||
				name==QmuMaskGroundediceLevelsetEnum ||
				name==QmuMaskIceLevelsetEnum ||
				name==GiaWEnum || 
				name==GiadWdtEnum ||
				name==SedimentHeadEnum ||
				name==EplHeadEnum ||
				name==SedimentHeadOldEnum ||
				name==EplHeadOldEnum ||
				name==HydrologydcEplThicknessOldEnum ||
				name==HydrologydcEplInitialThicknessEnum ||
				name==HydrologydcEplThicknessEnum ||
				name==HydrologydcMaskEplactiveNodeEnum ||
				name==WaterTransferEnum

				) {
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Penta::IsNodeOnShelfFromFlags {{{*/
bool   Penta::IsNodeOnShelfFromFlags(IssmDouble* flags){

	int  i;
	bool shelf=false;

	for(i=0;i<NUMVERTICES;i++){
		if (flags[vertices[i]->Pid()]<0.){
			shelf=true;
			break;
		}
	}
	return shelf;
}
/*}}}*/
/*FUNCTION Penta::JacobianDeterminant{{{*/
void Penta::JacobianDeterminant(IssmDouble* pJdet,IssmDouble* xyz_list,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetJacobianDeterminant(pJdet,xyz_list,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::JacobianDeterminantBase{{{*/
void Penta::JacobianDeterminantBase(IssmDouble* pJdet,IssmDouble* xyz_list_base,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetTriaJacobianDeterminant(pJdet,xyz_list_base,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::JacobianDeterminantLine{{{*/
void Penta::JacobianDeterminantLine(IssmDouble* pJdet,IssmDouble* xyz_list_line,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetSegmentJacobianDeterminant(pJdet,xyz_list_line,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::JacobianDeterminantTop{{{*/
void Penta::JacobianDeterminantTop(IssmDouble* pJdet,IssmDouble* xyz_list_top,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetTriaJacobianDeterminant(pJdet,xyz_list_top,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::JacobianDeterminantSurface{{{*/
void Penta::JacobianDeterminantSurface(IssmDouble* pJdet,IssmDouble* xyz_list_quad,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetQuadJacobianDeterminant(pJdet,xyz_list_quad,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::IsIceInElement {{{*/
bool   Penta::IsIceInElement(){

	/*Get levelset*/
	IssmDouble ls[NUMVERTICES];
	GetInputListOnVertices(&ls[0],MaskIceLevelsetEnum);

	/*If the level set one one of the nodes is <0, ice is present in this element*/
	if(ls[0]<0. || ls[1]<0. || ls[2]<0.) 
		return true;
	else
		return false;
}
/*}}}*/
/*FUNCTION Penta::MinEdgeLength{{{*/
IssmDouble Penta::MinEdgeLength(IssmDouble* xyz_list){
	/*Return the minimum lenght of the nine egdes of the penta*/

	int    i,node0,node1;
	int    edges[9][2]={{0,1},{0,2},{1,2},{3,4},{3,5},{4,5},{0,3},{1,4},{2,5}}; //list of the nine edges
	IssmDouble length;
	IssmDouble minlength=-1;

	for(i=0;i<9;i++){
		/*Find the two nodes for this edge*/
		node0=edges[i][0];
		node1=edges[i][1];

		/*Compute the length of this edge and compare it to the minimal length*/
		length=sqrt(pow(xyz_list[node0*3+0]-xyz_list[node1*3+0],2)+pow(xyz_list[node0*3+1]-xyz_list[node1*3+1],2)+pow(xyz_list[node0*3+2]-xyz_list[node1*3+2],2));
		if(length<minlength || minlength<0) minlength=length;
	}

	return minlength;
}
/*}}}*/
/*FUNCTION Penta::NodalValue {{{*/
int    Penta::NodalValue(IssmDouble* pvalue, int index, int natureofdataenum){

	int i;
	int found=0;
	IssmDouble value;
	Input* data=NULL;
	GaussPenta* gauss=NULL;

	/*First, serarch the input: */
	data=inputs->GetInput(natureofdataenum); 

	/*figure out if we have the vertex id: */
	found=0;
	for(i=0;i<NUMVERTICES;i++){
		if(index==vertices[i]->Id()){
			/*Do we have natureofdataenum in our inputs? :*/
			if(data){
				/*ok, we are good. retrieve value of input at vertex :*/
				gauss=new GaussPenta(); gauss->GaussVertex(i);
				data->GetInputValue(&value,gauss);
				found=1;
				break;
			}
		}
	}

	delete gauss;
	if(found)*pvalue=value;
	return found;
}
/*}}}*/
/*FUNCTION Penta::NewGauss(){{{*/
Gauss* Penta::NewGauss(void){
	return new GaussPenta();
}
/*}}}*/
/*FUNCTION Penta::NewGauss(int order){{{*/
Gauss* Penta::NewGauss(int order){
	return new GaussPenta(order,order);
}
/*}}}*/
/*FUNCTION Penta::NewGauss(IssmDouble* xyz_list, IssmDouble* xyz_list_front,int order_horiz,int order_vert){{{*/
Gauss* Penta::NewGauss(IssmDouble* xyz_list, IssmDouble* xyz_list_front,int order_horiz,int order_vert){

	IssmDouble  area_coordinates[4][3];

	GetAreaCoordinates(&area_coordinates[0][0],xyz_list_front,xyz_list,4);

	return new GaussPenta(area_coordinates,order_horiz,order_vert);
}
/*}}}*/
/*FUNCTION Penta::NewGaussBase(int order){{{*/
Gauss* Penta::NewGaussBase(int order){
	return new GaussPenta(0,1,2,order);
}
/*}}}*/
/*FUNCTION Penta::NewGaussLine(int vertex1,int vertex2,int order){{{*/
Gauss* Penta::NewGaussLine(int vertex1,int vertex2,int order){
	return new GaussPenta(vertex1,vertex2,order);
}
/*}}}*/
/*FUNCTION Penta::NewGaussTop(int order){{{*/
Gauss* Penta::NewGaussTop(int order){
	return new GaussPenta(3,4,5,order);
}
/*}}}*/
/*FUNCTION Penta::NewElementVector{{{*/
ElementVector* Penta::NewElementVector(int approximation_enum){
	return new ElementVector(nodes,this->NumberofNodes(),this->parameters,approximation_enum);
}
/*}}}*/
/*FUNCTION Penta::NewElementMatrix{{{*/
ElementMatrix* Penta::NewElementMatrix(int approximation_enum){
	return new ElementMatrix(nodes,this->NumberofNodes(),this->parameters,approximation_enum);
}
/*}}}*/
/*FUNCTION Penta::NewElementMatrixCoupling{{{*/
ElementMatrix* Penta::NewElementMatrixCoupling(int number_nodes,int approximation_enum){
	return new ElementMatrix(nodes,number_nodes,this->parameters,approximation_enum);
}
/*}}}*/
/*FUNCTION Penta::NodalFunctions{{{*/
void Penta::NodalFunctions(IssmDouble* basis, Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctions(basis,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsP1{{{*/
void Penta::NodalFunctionsP1(IssmDouble* basis, Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsP1(basis,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsDerivatives{{{*/
void Penta::NodalFunctionsDerivatives(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsDerivatives(dbasis,xyz_list,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsP1Derivatives{{{*/
void Penta::NodalFunctionsP1Derivatives(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsP1Derivatives(dbasis,xyz_list,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsMINIDerivatives{{{*/
void Penta::NodalFunctionsMINIDerivatives(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsMINIDerivatives(dbasis,xyz_list,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsDerivativesVelocity{{{*/
void Penta::NodalFunctionsDerivativesVelocity(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsDerivativesVelocity(dbasis,xyz_list,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsVelocity{{{*/
void Penta::NodalFunctionsVelocity(IssmDouble* basis, Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsVelocity(basis,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NodalFunctionsPressure{{{*/
void Penta::NodalFunctionsPressure(IssmDouble* basis, Gauss* gauss){

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->GetNodalFunctionsPressure(basis,(GaussPenta*)gauss);

}
/*}}}*/
/*FUNCTION Penta::NormalBase {{{*/
void Penta::NormalBase(IssmDouble* bed_normal,IssmDouble* xyz_list){

	int i;
	IssmDouble v13[3],v23[3];
	IssmDouble normal[3];
	IssmDouble normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0*3+i]-xyz_list[2*3+i];
		v23[i]=xyz_list[1*3+i]-xyz_list[2*3+i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];
	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	/*Bed normal is opposite to surface normal*/
	bed_normal[0]=-normal[0]/normal_norm;
	bed_normal[1]=-normal[1]/normal_norm;
	bed_normal[2]=-normal[2]/normal_norm;
}
/*}}}*/
/*FUNCTION Penta::NormalTop {{{*/
void Penta::NormalTop(IssmDouble* top_normal,IssmDouble* xyz_list){

	int i;
	IssmDouble v13[3],v23[3];
	IssmDouble normal[3];
	IssmDouble normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0*3+i]-xyz_list[2*3+i];
		v23[i]=xyz_list[1*3+i]-xyz_list[2*3+i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];
	normal_norm=sqrt(normal[0]*normal[0] + normal[1]*normal[1] + normal[2]*normal[2]);

	top_normal[0]=normal[0]/normal_norm;
	top_normal[1]=normal[1]/normal_norm;
	top_normal[2]=normal[2]/normal_norm;
}
/*}}}*/
/*FUNCTION Penta::NumberofNodesPressure{{{*/
int Penta::NumberofNodesPressure(void){
	return PentaRef::NumberofNodesPressure();
}
/*}}}*/
/*FUNCTION Penta::NumberofNodesVelocity{{{*/
int Penta::NumberofNodesVelocity(void){
	return PentaRef::NumberofNodesVelocity();
}
/*}}}*/
/*FUNCTION Penta::PositiveDegreeDay{{{*/
void  Penta::PositiveDegreeDay(IssmDouble* pdds,IssmDouble* pds,IssmDouble signorm){

   IssmDouble agd[NUMVERTICES];             // surface mass balance
   IssmDouble monthlytemperatures[NUMVERTICES][12],monthlyprec[NUMVERTICES][12];
   IssmDouble h[NUMVERTICES],s[NUMVERTICES]; // ,b
   IssmDouble rho_water,rho_ice,desfac,s0p;

   /*Recover monthly temperatures and precipitation*/
   Input*     input=inputs->GetInput(SurfaceforcingsMonthlytemperaturesEnum); _assert_(input);
   Input*     input2=inputs->GetInput(SurfaceforcingsPrecipitationEnum); _assert_(input2);
   GaussPenta* gauss=new GaussPenta();
   IssmDouble time,yts;
   this->parameters->FindParam(&time,TimeEnum);
   this->parameters->FindParam(&yts,ConstantsYtsEnum);
   for(int month=0;month<12;month++) {
     for(int iv=0;iv<NUMVERTICES;iv++) {
       gauss->GaussVertex(iv);
       input->GetInputValue(&monthlytemperatures[iv][month],gauss,time+month/12.*yts);
       monthlytemperatures[iv][month]=monthlytemperatures[iv][month]-273.15; // conversion from Kelvin to celcius
       input2->GetInputValue(&monthlyprec[iv][month],gauss,time+month/12.*yts);
       monthlyprec[iv][month]=monthlyprec[iv][month]*yts; // convertion to m/yr
     }
   } 

  /*Recover info at the vertices: */
  GetInputListOnVertices(&h[0],ThicknessEnum);
  GetInputListOnVertices(&s[0],SurfaceEnum);

  /*Get material parameters :*/
  rho_ice=matpar->GetRhoIce();
  rho_water=matpar->GetRhoFreshwater();

  /*Get desertification effect parameters*/
  desfac=matpar->GetDesFac();
  s0p=matpar->GetS0p();

   /*measure the surface mass balance*/
   for (int iv = 0; iv < NUMVERTICES; iv++){
     agd[iv]=PddSurfaceMassBlance(&monthlytemperatures[iv][0], &monthlyprec[iv][0], pdds, pds, 
				  signorm, yts, h[iv], s[iv], rho_ice, rho_water, desfac, s0p);
   }

   /*Update inputs*/    
   this->inputs->AddInput(new PentaInput(SurfaceforcingsMassBalanceEnum,&agd[0],P1Enum));
   //this->inputs->AddInput(new PentaVertexInput(ThermalSpcTemperatureEnum,&Tsurf[0]));
   this->InputExtrude(SurfaceforcingsMassBalanceEnum,ElementEnum);

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::PureIceEnthalpy{{{*/
IssmDouble Penta::PureIceEnthalpy(IssmDouble pressure){

	return this->matpar->PureIceEnthalpy(pressure);
}
/*}}}*/
/*FUNCTION Penta::ReduceMatrices{{{*/
void Penta::ReduceMatrices(ElementMatrix* Ke,ElementVector* pe){

	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	if(pe){
		if(analysis_type==StressbalanceAnalysisEnum){
			if(this->element_type==MINIcondensedEnum){
				int approximation;
				inputs->GetInputValue(&approximation,ApproximationEnum);
				if(approximation==HOFSApproximationEnum || approximation==SSAFSApproximationEnum){
					//Do nothing, condensation already done in PVectorCoupling
				}
				else{
					int indices[3]={18,19,20};
					pe->StaticCondensation(Ke,3,&indices[0]);
				}
			}
			else if(this->element_type==P1bubblecondensedEnum){
				int size   = nodes[6]->GetNumberOfDofs(NoneApproximationEnum,GsetEnum);
				int offset = 0;
				for(int i=0;i<6;i++) offset+=nodes[i]->GetNumberOfDofs(NoneApproximationEnum,GsetEnum);
				int* indices=xNew<int>(size);
				for(int i=0;i<size;i++) indices[i] = offset+i;
				pe->StaticCondensation(Ke,size,indices);
				xDelete<int>(indices);
			}
		}
	}

	if(Ke){
		if(analysis_type==StressbalanceAnalysisEnum){
			int approximation;
			inputs->GetInputValue(&approximation,ApproximationEnum);
			if(approximation==HOFSApproximationEnum || approximation==SSAFSApproximationEnum){
				//Do nothing condensatino already done for Stokes part
			}
			else{
				if(this->element_type==MINIcondensedEnum){
					int indices[3]={18,19,20};
					Ke->StaticCondensation(3,&indices[0]);
				}
				else if(this->element_type==P1bubblecondensedEnum){
					int size   = nodes[6]->GetNumberOfDofs(NoneApproximationEnum,GsetEnum);
					int offset = 0;
					for(int i=0;i<6;i++) offset+=nodes[i]->GetNumberOfDofs(NoneApproximationEnum,GsetEnum);
					int* indices=xNew<int>(size);
					for(int i=0;i<size;i++) indices[i] = offset+i;
					Ke->StaticCondensation(size,indices);
					xDelete<int>(indices);
				}
			}
		}
	}
}
/*}}}*/
/*FUNCTION Penta::ResetCoordinateSystem{{{*/
void  Penta::ResetCoordinateSystem(void){

	int        approximation;
	int        numindices;
	int       *indices = NULL;
	IssmDouble slopex,slopey;
	IssmDouble xz_plane[6];

	/*For FS only: we want the CS to be tangential to the bedrock*/
	inputs->GetInputValue(&approximation,ApproximationEnum);
	if(IsFloating() || !IsOnBed() || (approximation!=FSApproximationEnum && approximation!=SSAFSApproximationEnum &&  approximation!=HOFSApproximationEnum)) return;

	/*Get number of nodes for velocity only and base*/
	BasalNodeIndices(&numindices,&indices,this->VelocityInterpolation());

	/*Get inputs*/
	Input* slopex_input=inputs->GetInput(BedSlopeXEnum); _assert_(slopex_input);
	Input* slopey_input=inputs->GetInput(BedSlopeYEnum); _assert_(slopey_input);

	/*Loop over basal nodes and update their CS*/
	GaussPenta* gauss = new GaussPenta();
	for(int i=0;i<numindices;i++){//FIXME
		gauss->GaussNode(this->VelocityInterpolation(),indices[i]);

		slopex_input->GetInputValue(&slopex,gauss);
		slopey_input->GetInputValue(&slopey,gauss);

		/*New X axis          New Z axis*/
		xz_plane[0]=1.;       xz_plane[3]=-slopex;  
		xz_plane[1]=0.;       xz_plane[4]=-slopey;  
		xz_plane[2]=slopex;   xz_plane[5]=1.;          

		XZvectorsToCoordinateSystem(&this->nodes[indices[i]]->coord_system[0][0],&xz_plane[0]);
	}

	/*cleanup*/
	xDelete<int>(indices);
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::SetClone {{{*/
void  Penta::SetClone(int* minranks){

	_error_("not implemented yet");
}
/*}}}*/
/*FUNCTION Penta::SetCurrentConfiguration {{{*/
void  Penta::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, Nodes* nodesin, Materials* materialsin, Parameters* parametersin){

	int analysis_counter;

	/*go into parameters and get the analysis_counter: */
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Pick up nodes */
	if (this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
}
/*}}}*/
/*FUNCTION Penta::SetTemporaryElementType{{{*/
void Penta::SetTemporaryElementType(int element_type_in){
	this->element_type=element_type_in;
}
/*}}}*/
/*FUNCTION Penta::SetwiseNodeConnectivity{{{*/
void Penta::SetwiseNodeConnectivity(int* pd_nz,int* po_nz,Node* node,bool* flags,int* flagsindices,int set1_enum,int set2_enum){

	/*Intermediaries*/
	const int numnodes = this->NumberofNodes();

	/*Output */
	int d_nz = 0;
	int o_nz = 0;

	/*Loop over all nodes*/
	for(int i=0;i<numnodes;i++){

		if(!flags[this->nodes[i]->Lid()]){

			/*flag current node so that no other element processes it*/
			flags[this->nodes[i]->Lid()]=true;

			int counter=0;
			while(flagsindices[counter]>=0) counter++;
			flagsindices[counter]=this->nodes[i]->Lid();

			/*if node is clone, we have an off-diagonal non-zero, else it is a diagonal non-zero*/
			switch(set2_enum){
				case FsetEnum:
					if(nodes[i]->indexing.fsize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				case GsetEnum:
					if(nodes[i]->indexing.gsize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				case SsetEnum:
					if(nodes[i]->indexing.ssize){
						if(this->nodes[i]->IsClone())
						 o_nz += 1;
						else
						 d_nz += 1;
					}
					break;
				default: _error_("not supported");
			}
		}
	}

	/*Special case: 2d/3d coupling, the node of this element might be connected
	 *to the basal element*/
	int analysis_type,approximation,numlayers;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	if(analysis_type==StressbalanceAnalysisEnum){
		inputs->GetInputValue(&approximation,ApproximationEnum);
		if(approximation==SSAHOApproximationEnum || approximation==SSAFSApproximationEnum){
			parameters->FindParam(&numlayers,MeshNumberoflayersEnum);
			o_nz += numlayers*3;
			d_nz += numlayers*3;
		}
	}

	/*Assign output pointers: */
	*pd_nz=d_nz;
	*po_nz=o_nz;
}
/*}}}*/
/*FUNCTION Penta::SpawnTria {{{*/
Tria*  Penta::SpawnTria(int location){

	int analysis_counter;

	/*go into parameters and get the analysis_counter: */
	this->parameters->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Create Tria*/
	Tria* tria=new Tria();
	tria->id=this->id;
	tria->inputs=(Inputs*)this->inputs->SpawnTriaInputs(location);
	tria->parameters=this->parameters;
	tria->element_type=P1Enum; //Only P1 CG for now (TO BE CHANGED)
	this->SpawnTriaHook(dynamic_cast<ElementHook*>(tria),location);

	/*Spawn material*/
	tria->material=(Material*)this->material->copy();
	delete tria->material->inputs;
	tria->material->inputs=(Inputs*)this->material->inputs->SpawnTriaInputs(location);

	/*recover nodes, material and matpar: */
	tria->nodes=(Node**)tria->hnodes[analysis_counter]->deliverp();
	tria->vertices=(Vertex**)tria->hvertices->deliverp();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();

	/*Return new Tria*/
	return tria;
}
/*}}}*/
/*FUNCTION Penta::SpawnBasalElement{{{*/
Element*  Penta::SpawnBasalElement(void){

	_assert_(this->IsOnBed());

	this->InputDepthAverageAtBase(MaterialsRheologyBEnum,MaterialsRheologyBbarEnum,MaterialsEnum);
	this->InputDepthAverageAtBase(DamageDEnum,DamageDbarEnum,MaterialsEnum);
	if(this->inputs->GetInput(VxEnum)) this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	if(this->inputs->GetInput(VyEnum)) this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	this->material->inputs->DeleteInput(MaterialsRheologyBbarEnum);
	this->material->inputs->DeleteInput(DamageDbarEnum);
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	return tria;
}
/*}}}*/
/*FUNCTION Penta::SpawnTopElement{{{*/
Element*  Penta::SpawnTopElement(void){

	_assert_(this->IsOnSurface());

	Tria* tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.

	return tria;
}
/*}}}*/
/*FUNCTION Penta::SmbGradients{{{*/
void Penta::SmbGradients(void){

	int i;

	// input
   IssmDouble h[NUMVERTICES];					// ice thickness (m)		
	IssmDouble s[NUMVERTICES];					// surface elevation (m)
	IssmDouble b_pos[NUMVERTICES];				// Hs-SMB relation parameter
	IssmDouble b_neg[NUMVERTICES];				// Hs-SMB relation paremeter
	IssmDouble Href[NUMVERTICES];					// reference elevation from which deviations are used to calculate the SMB adjustment
	IssmDouble Smbref[NUMVERTICES];				// reference SMB to which deviations are added
   IssmDouble rho_water;                   // density of fresh water
	IssmDouble rho_ice;                     // density of ice

	// output
	IssmDouble smb[NUMVERTICES];					// surface mass balance (m/yr ice)

	/*Recover SmbGradients*/
	GetInputListOnVertices(&Href[0],SurfaceforcingsHrefEnum);
	GetInputListOnVertices(&Smbref[0],SurfaceforcingsSmbrefEnum);
	GetInputListOnVertices(&b_pos[0],SurfaceforcingsBPosEnum);
	GetInputListOnVertices(&b_neg[0],SurfaceforcingsBNegEnum);

   /*Recover surface elevatio at vertices: */
	GetInputListOnVertices(&h[0],ThicknessEnum);
	GetInputListOnVertices(&s[0],SurfaceEnum);

   /*Get material parameters :*/
   rho_ice=matpar->GetRhoIce();
   rho_water=matpar->GetRhoFreshwater();

   // loop over all vertices
 for(i=0;i<NUMVERTICES;i++){
     if(Smbref[i]>0){
		  smb[i]=Smbref[i]+b_pos[i]*(s[i]-Href[i]);
	  }
	  else{
		  smb[i]=Smbref[i]+b_neg[i]*(s[i]-Href[i]);
	  }
	  smb[i]=smb[i]/rho_ice;      // SMB in m/y ice		
	}  //end of the loop over the vertices
	  /*Update inputs*/
	  this->inputs->AddInput(new PentaInput(SurfaceforcingsMassBalanceEnum,&smb[0],P1Enum));
}
/*}}}*/
/*FUNCTION Penta::SurfaceArea {{{*/
IssmDouble Penta::SurfaceArea(void){

	int    approximation;
	IssmDouble S;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceArea*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		S=tria->SurfaceArea();
		delete tria->material; delete tria;
		return S;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		S=tria->SurfaceArea();
		delete tria->material; delete tria;
		return S;
	}
}
/*}}}*/
/*FUNCTION Penta::TimeAdapt{{{*/
IssmDouble  Penta::TimeAdapt(void){

	int    i;
	IssmDouble C,dx,dy,dz,dt;
	IssmDouble maxabsvx,maxabsvy,maxabsvz;
	IssmDouble maxx,minx,maxy,miny,maxz,minz;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*get CFL coefficient:*/
	this->parameters->FindParam(&C,TimesteppingCflCoefficientEnum);

	/*Get for Vx and Vy, the max of abs value: */
	this->MaxAbsVx(&maxabsvx);
	this->MaxAbsVy(&maxabsvy);
	this->MaxAbsVz(&maxabsvz);

	/* Get node coordinates and dof list: */
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	minx=xyz_list[0][0];
	maxx=xyz_list[0][0];
	miny=xyz_list[0][1];
	maxy=xyz_list[0][1];
	minz=xyz_list[0][2];
	maxz=xyz_list[0][2];

	for(i=1;i<NUMVERTICES;i++){
		if (xyz_list[i][0]<minx)minx=xyz_list[i][0];
		if (xyz_list[i][0]>maxx)maxx=xyz_list[i][0];
		if (xyz_list[i][1]<miny)miny=xyz_list[i][1];
		if (xyz_list[i][1]>maxy)maxy=xyz_list[i][1];
		if (xyz_list[i][2]<minz)minz=xyz_list[i][2];
		if (xyz_list[i][2]>maxz)maxz=xyz_list[i][2];
	}
	dx=maxx-minx;
	dy=maxy-miny;
	dz=maxz-minz;

	/*CFL criterion: */
	dt=C/(maxabsvy/dx+maxabsvy/dy+maxabsvz/dz);

	return dt;
}/*}}}*/
/*FUNCTION Penta::Update {{{*/
void Penta::Update(int index,IoModel* iomodel,int analysis_counter,int analysis_type,int finiteelement_type){ 

	/*Intermediaries*/
	int        i;
	int        penta_vertex_ids[6];
	IssmDouble nodeinputs[6];
	IssmDouble yts;
	bool       dakota_analysis;
	bool       isFS;
	int        numnodes;
	int*       penta_node_ids = NULL;

	/*Fetch parameters: */
	iomodel->Constant(&yts,ConstantsYtsEnum);
	iomodel->Constant(&dakota_analysis,QmuIsdakotaEnum);
	iomodel->Constant(&isFS,FlowequationIsFSEnum);

	/*Checks if debuging*/
	_assert_(iomodel->elements);

	/*Recover element type*/
	this->SetElementType(finiteelement_type,analysis_counter);

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<6;i++) penta_vertex_ids[i]=iomodel->elements[6*index+i]; //ids for vertices are in the elements array from Matlab

	/*Recover nodes ids needed to initialize the node hook.*/
	switch(finiteelement_type){
		case P1Enum:
			numnodes         = 6;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			break;
		case P1bubbleEnum: case P1bubblecondensedEnum:
			numnodes         = 7;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[6]=iomodel->nodecounter+iomodel->numberofvertices+index+1;
			break;
		case P1xP2Enum:
			numnodes         = 9;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+0]+1;
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+1]+1;
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+2]+1;
			break;
		case P1xP3Enum:
			numnodes         = 12;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+0]+1;
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+1]+1;
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+2]+1;
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+0]+2;
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+1]+2;
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+2]+2;
			break;
		case P2xP1Enum:
			numnodes         = 12;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+3]+1;
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+4]+1;
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+5]+1;
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+6]+1;
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+7]+1;
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+8]+1;
			break;
		case P2xP4Enum:
			numnodes         = 30;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0]; /*Vertex 1*/
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1]; /*Vertex 2*/
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2]; /*Vertex 3*/
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3]; /*Vertex 4*/
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4]; /*Vertex 5*/
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5]; /*Vertex 6*/
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+1; /*mid vertical edge 1*/
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+1; /*mid vertical edge 2*/
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+1; /*mid vertical edge 3*/
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+3]+1; /*mid basal edge 1*/
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+4]+1; /*mid basal edge 2*/
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+5]+1; /*mid basal edge 3*/
			penta_node_ids[12]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+6]+1; /*mid top edge 1*/
			penta_node_ids[13]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+7]+1; /*mid top edge 2*/
			penta_node_ids[14]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+8]+1; /*mid top edge 3*/
			penta_node_ids[15]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+2; /* 1/4 vertical edge 1*/
			penta_node_ids[16]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+2; /* 1/4 vertical edge 2*/
			penta_node_ids[17]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+2; /* 1/4 vertical edge 3*/
			penta_node_ids[18]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+3; /* 2/4 vertical edge 1*/
			penta_node_ids[19]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+3; /* 2/4 vertical edge 2*/
			penta_node_ids[20]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+3; /* 2/4 vertical edge 3*/
			penta_node_ids[21]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+1; /* 1/4 vertical face 1*/
			penta_node_ids[22]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+1; /* 1/4 vertical face 2*/
			penta_node_ids[23]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+1; /* 1/4 vertical face 3*/
			penta_node_ids[24]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+2; /* 2/4 vertical face 1*/
			penta_node_ids[25]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+2; /* 2/4 vertical face 2*/
			penta_node_ids[26]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+2; /* 2/4 vertical face 3*/
			penta_node_ids[27]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+3; /* 3/4 vertical face 1*/
			penta_node_ids[28]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+3; /* 3/4 vertical face 2*/
			penta_node_ids[29]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+3; /* 3/4 vertical face 3*/
			break;
		case P2Enum:
			numnodes         = 15;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+0]+1;
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+1]+1;
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+2]+1;
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+3]+1;
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+4]+1;
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+5]+1;
			penta_node_ids[12]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+6]+1;
			penta_node_ids[13]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+7]+1;
			penta_node_ids[14]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+8]+1;
			break;
		case P1P1Enum: case P1P1GLSEnum:
			numnodes         = 12;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];

			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+0];
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+1];
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+2];
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+3];
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+4];
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elements[6*index+5];
			break;
		case MINIEnum: case MINIcondensedEnum:
			numnodes         = 13;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+index+1;

			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+0];
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+1];
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+2];
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+3];
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+4];
			penta_node_ids[12]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofelements+iomodel->elements[6*index+5];
			break;
		case TaylorHoodEnum:
			numnodes         = 21;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0];
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1];
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2];
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3];
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4];
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5];
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+0]+1;
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+1]+1;
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+2]+1;
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+3]+1;
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+4]+1;
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+5]+1;
			penta_node_ids[12]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+6]+1;
			penta_node_ids[13]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+7]+1;
			penta_node_ids[14]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->elementtoedgeconnectivity[9*index+8]+1;

			penta_node_ids[15]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+0];
			penta_node_ids[16]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+1];
			penta_node_ids[17]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+2];
			penta_node_ids[18]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+3];
			penta_node_ids[19]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+4];
			penta_node_ids[20]=iomodel->nodecounter+iomodel->numberofvertices+iomodel->numberofedges+iomodel->elements[6*index+5];
			break;
		case OneLayerP4zEnum:
			numnodes         = 30+12;
			penta_node_ids   = xNew<int>(numnodes);
			penta_node_ids[ 0]=iomodel->nodecounter+iomodel->elements[6*index+0]; /*Vertex 1*/
			penta_node_ids[ 1]=iomodel->nodecounter+iomodel->elements[6*index+1]; /*Vertex 2*/
			penta_node_ids[ 2]=iomodel->nodecounter+iomodel->elements[6*index+2]; /*Vertex 3*/
			penta_node_ids[ 3]=iomodel->nodecounter+iomodel->elements[6*index+3]; /*Vertex 4*/
			penta_node_ids[ 4]=iomodel->nodecounter+iomodel->elements[6*index+4]; /*Vertex 5*/
			penta_node_ids[ 5]=iomodel->nodecounter+iomodel->elements[6*index+5]; /*Vertex 6*/
			penta_node_ids[ 6]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+1; /*mid vertical edge 1*/
			penta_node_ids[ 7]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+1; /*mid vertical edge 2*/
			penta_node_ids[ 8]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+1; /*mid vertical edge 3*/
			penta_node_ids[ 9]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+3]+1; /*mid basal edge 1*/
			penta_node_ids[10]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+4]+1; /*mid basal edge 2*/
			penta_node_ids[11]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+5]+1; /*mid basal edge 3*/
			penta_node_ids[12]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+6]+1; /*mid top edge 1*/
			penta_node_ids[13]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+7]+1; /*mid top edge 2*/
			penta_node_ids[14]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+8]+1; /*mid top edge 3*/
			penta_node_ids[15]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+2; /* 1/4 vertical edge 1*/
			penta_node_ids[16]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+2; /* 1/4 vertical edge 2*/
			penta_node_ids[17]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+2; /* 1/4 vertical edge 3*/
			penta_node_ids[18]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+0]+3; /* 2/4 vertical edge 1*/
			penta_node_ids[19]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+1]+3; /* 2/4 vertical edge 2*/
			penta_node_ids[20]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->elementtoedgeconnectivity[9*index+2]+3; /* 2/4 vertical edge 3*/
			penta_node_ids[21]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+1; /* 1/4 vertical face 1*/
			penta_node_ids[22]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+1; /* 1/4 vertical face 2*/
			penta_node_ids[23]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+1; /* 1/4 vertical face 3*/
			penta_node_ids[24]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+2; /* 2/4 vertical face 1*/
			penta_node_ids[25]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+2; /* 2/4 vertical face 2*/
			penta_node_ids[26]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+2; /* 2/4 vertical face 3*/
			penta_node_ids[27]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+2]+3; /* 3/4 vertical face 1*/
			penta_node_ids[28]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+3]+3; /* 3/4 vertical face 2*/
			penta_node_ids[29]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->elementtofaceconnectivity[5*index+4]+3; /* 3/4 vertical face 3*/

			penta_node_ids[30]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+0];
			penta_node_ids[31]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+1];
			penta_node_ids[32]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+2];
			penta_node_ids[33]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+3];
			penta_node_ids[34]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+4];
			penta_node_ids[35]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->elements[6*index+5];
			penta_node_ids[36]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+0]+1;
			penta_node_ids[37]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+1]+1;
			penta_node_ids[38]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+2]+1;
			penta_node_ids[39]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+0]+2;
			penta_node_ids[40]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+1]+2;
			penta_node_ids[41]=iomodel->nodecounter+iomodel->numberofvertices+3*iomodel->numberofedges+3*iomodel->numberoffaces+iomodel->numberofvertices+2*iomodel->elementtoedgeconnectivity[9*index+2]+2;
			break;
		default:
			_error_("Finite element "<<EnumToStringx(finiteelement_type)<<" not supported yet");
	}

	/*hooks: */
	this->SetHookNodes(penta_node_ids,numnodes,analysis_counter); this->nodes=NULL;
	xDelete<int>(penta_node_ids);

	/*Fill with IoModel*/
	this->InputUpdateFromIoModel(index,iomodel);

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case StressbalanceAnalysisEnum:

			if(*(iomodel->Data(FlowequationElementEquationEnum)+index)==HOFSApproximationEnum){
				/*Create VzHO and VzFS Enums*/
				if(iomodel->Data(VzEnum) && iomodel->Data(FlowequationBorderFSEnum)){
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->Data(VzEnum)[penta_vertex_ids[i]-1]*iomodel->Data(FlowequationBorderFSEnum)[penta_vertex_ids[i]-1];
					this->inputs->AddInput(new PentaInput(VzFSEnum,nodeinputs,P1Enum));
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->Data(VzEnum)[penta_vertex_ids[i]-1]*(1-iomodel->Data(FlowequationBorderFSEnum)[penta_vertex_ids[i]-1]);
					this->inputs->AddInput(new PentaInput(VzHOEnum,nodeinputs,P1Enum));
				}
				else{
					for(i=0;i<6;i++)nodeinputs[i]=0;
					this->inputs->AddInput(new PentaInput(VzFSEnum,nodeinputs,P1Enum));
					this->inputs->AddInput(new PentaInput(VzHOEnum,nodeinputs,P1Enum));
				}
			}
			if(*(iomodel->Data(FlowequationElementEquationEnum)+index)==SSAFSApproximationEnum){
				/*Create VzSSA and VzFS Enums*/
				if(iomodel->Data(VzEnum) && iomodel->Data(FlowequationBorderFSEnum)){
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->Data(VzEnum)[penta_vertex_ids[i]-1]*iomodel->Data(FlowequationBorderFSEnum)[penta_vertex_ids[i]-1];
					this->inputs->AddInput(new PentaInput(VzFSEnum,nodeinputs,P1Enum));
					for(i=0;i<6;i++) nodeinputs[i]=iomodel->Data(VzEnum)[penta_vertex_ids[i]-1]*(1-iomodel->Data(FlowequationBorderFSEnum)[penta_vertex_ids[i]-1]);
					this->inputs->AddInput(new PentaInput(VzSSAEnum,nodeinputs,P1Enum));
				}
				else{
					for(i=0;i<6;i++)nodeinputs[i]=0;
					this->inputs->AddInput(new PentaInput(VzFSEnum,nodeinputs,P1Enum));
					this->inputs->AddInput(new PentaInput(VzSSAEnum,nodeinputs,P1Enum));
				}
			}
			break;
		default:
			/*No update for other solution types*/
			break;
	}
}
/*}}}*/
/*FUNCTION Penta::UpdateConstraintsExtrudeFromBase{{{*/
void  Penta::UpdateConstraintsExtrudeFromBase(void){

	if(!IsOnBed()) return;

	int        extrusioninput;
	IssmDouble value,isonbed;

	this->parameters->FindParam(&extrusioninput,InputToExtrudeEnum);
	Input* input = inputs->GetInput(extrusioninput);      _assert_(extrusioninput);
	Input* onbed = inputs->GetInput(MeshVertexonbedEnum); _assert_(onbed);

	GaussPenta* gauss=new GaussPenta();
	for(int iv=0;iv<this->NumberofNodes();iv++){
		gauss->GaussNode(this->element_type,iv);
		onbed->GetInputValue(&isonbed,gauss);
		if(isonbed==1.){
			input->GetInputValue(&value,gauss);
			this->nodes[iv]->ApplyConstraint(1,value);
		}
	}
	delete gauss;

}
/*}}}*/
/*FUNCTION Penta::UpdateConstraintsExtrudeFromTop{{{*/
void  Penta::UpdateConstraintsExtrudeFromTop(void){

	if(!IsOnSurface()) return;

	int extrusioninput;
	int indices[3]={3,4,5};
	IssmDouble value;

	this->parameters->FindParam(&extrusioninput,InputToExtrudeEnum);
	Input* input = inputs->GetInput(extrusioninput); _assert_(extrusioninput);

	GaussPenta* gauss=new GaussPenta();
	for(int i=0;i<3;i++){
		gauss->GaussNode(P1Enum,indices[i]);
		input->GetInputValue(&value,gauss);
		this->nodes[indices[i]]->ApplyConstraint(1,value);
	}
	delete gauss;

}
/*}}}*/
/*FUNCTION Penta::ValueP1OnGauss{{{*/
void Penta::ValueP1OnGauss(IssmDouble* pvalue,IssmDouble* values,Gauss* gauss){
	PentaRef::GetInputValue(pvalue,values,gauss,P1Enum);
}
/*}}}*/
/*FUNCTION Penta::ValueP1DerivativesOnGauss{{{*/
void Penta::ValueP1DerivativesOnGauss(IssmDouble* dvalue,IssmDouble* values,IssmDouble* xyz_list,Gauss* gauss){
	PentaRef::GetInputDerivativeValue(dvalue,values,xyz_list,gauss);
}
/*}}}*/
/*FUNCTION Penta::VertexConnectivity{{{*/
int Penta::VertexConnectivity(int vertexindex){
	_assert_(this->vertices);
	return this->vertices[vertexindex]->Connectivity();
}
/*}}}*/
/*FUNCTION Penta::VerticalSegmentIndices{{{*/
void Penta::VerticalSegmentIndices(int** pindices,int* pnumseg){

	/*output*/
	int *indices = xNew<int>(3*2);
	indices[0*2 + 0] = 0; indices[0*2 + 1] = 3;
	indices[1*2 + 0] = 1; indices[1*2 + 1] = 4;
	indices[2*2 + 0] = 2; indices[2*2 + 1] = 5;

	/*Assign output pointers*/
	*pindices = indices;
	*pnumseg  = 3;
}
/*}}}*/
/*FUNCTION Penta::ViscousHeating{{{*/
void Penta::ViscousHeating(IssmDouble* pphi,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input){

	/*Intermediaries*/
	IssmDouble phi;
	IssmDouble viscosity;
	IssmDouble epsilon[6];

	_assert_(gauss->Enum()==GaussPentaEnum);
	this->StrainRateFS(&epsilon[0],xyz_list,(GaussPenta*)gauss,vx_input,vy_input,vz_input);
	material->GetViscosity3dFS(&viscosity,&epsilon[0]);
	GetPhi(&phi,&epsilon[0],viscosity);

	/*Assign output pointer*/
	*pphi = phi;
}
/*}}}*/
/*FUNCTION Penta::VelocityInterpolation{{{*/
int Penta::VelocityInterpolation(void){
	return PentaRef::VelocityInterpolation();
}
/*}}}*/
/*FUNCTION Penta::PressureInterpolation{{{*/
int Penta::PressureInterpolation(void){
	return PentaRef::PressureInterpolation();
}
/*}}}*/
/*FUNCTION Penta::IsZeroLevelset{{{*/
bool Penta::IsZeroLevelset(int levelset_enum){

	bool        iszerols;
	IssmDouble  ls[NUMVERTICES];

	/*Retrieve all inputs and parameters*/
	GetInputListOnVertices(&ls[0],levelset_enum);

	/*If the level set has always same sign, there is no ice front here*/
	iszerols = false;
	if(IsIceInElement()){
		if(ls[0]*ls[1]<0. || ls[0]*ls[2]<0. || (ls[0]*ls[1]+ls[0]*ls[2]+ls[1]*ls[2]==0.)){
			iszerols = true;
		}
	}
	return iszerols;
}
/*}}}*/

/*FUNCTION Penta::AverageOntoPartition {{{*/
void  Penta::AverageOntoPartition(Vector<IssmDouble>* partition_contributions,Vector<IssmDouble>* partition_areas,IssmDouble* vertex_response,IssmDouble* qmu_part){
	_error_("Not supported yet!");
}
/*}}}*/
/*FUNCTION Penta::IceVolume {{{*/
IssmDouble Penta::IceVolume(void){

	/*The volume of a troncated prism is base * 1/3 sum(length of edges)*/
	IssmDouble base,height;
	IssmDouble xyz_list[NUMVERTICES][3];

	if(!IsIceInElement())return 0;

	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	/*First calculate the area of the base (cross section triangle)
	 * http://en.wikipedia.org/wiki/Pentangle
	 * base = 1/2 abs((xA-xC)(yB-yA)-(xA-xB)(yC-yA))*/
	base = 1./2.*fabs((xyz_list[0][0]-xyz_list[2][0])*(xyz_list[1][1]-xyz_list[0][1]) - (xyz_list[0][0]-xyz_list[1][0])*(xyz_list[2][1]-xyz_list[0][1]));

	/*Now get the average height*/
	height = 1./3.*((xyz_list[3][2]-xyz_list[0][2])+(xyz_list[4][2]-xyz_list[1][2])+(xyz_list[5][2]-xyz_list[2][2]));

	/*Return: */
	return base*height;
}
/*}}}*/
/*FUNCTION Penta::IceVolumeAboveFloatation {{{*/
IssmDouble Penta::IceVolumeAboveFloatation(void){

	/*Volume above floatation: H + rho_water/rho_ice*bathymetry for nodes on the bed*/
	IssmDouble rho_ice,rho_water;
	IssmDouble base,bed,surface,bathymetry;
	IssmDouble xyz_list[NUMVERTICES][3];

	if(!IsIceInElement() || IsFloating() || !IsOnBed())return 0;

	rho_ice=matpar->GetRhoIce();
	rho_water=matpar->GetRhoWater();
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	/*First calculate the area of the base (cross section triangle)
	 * http://en.wikipedia.org/wiki/Pentangle
	 * base = 1/2 abs((xA-xC)(yB-yA)-(xA-xB)(yC-yA))*/
	base = 1./2.*fabs((xyz_list[0][0]-xyz_list[2][0])*(xyz_list[1][1]-xyz_list[0][1]) - (xyz_list[0][0]-xyz_list[1][0])*(xyz_list[2][1]-xyz_list[0][1]));

	/*Now get the average height above floatation*/
	Input* surface_input    = inputs->GetInput(SurfaceEnum);    _assert_(surface_input);
	Input* bed_input        = inputs->GetInput(BedEnum);        _assert_(bed_input);
	Input* bathymetry_input = inputs->GetInput(BathymetryEnum); _assert_(bathymetry_input);
	surface_input->GetInputAverage(&surface);
	bed_input->GetInputAverage(&bed);
	bathymetry_input->GetInputAverage(&bathymetry);

	/*Return: */
	return base*(surface - bed + min( rho_water/rho_ice * bathymetry, 0.) );
}
/*}}}*/
/*FUNCTION Penta::MinVel{{{*/
void  Penta::MinVel(IssmDouble* pminvel){

	/*Get minimum:*/
	IssmDouble minvel=this->inputs->Min(VelEnum);

	/*Assign output pointers:*/
	*pminvel=minvel;
}
/*}}}*/
/*FUNCTION Penta::MinVx{{{*/
void  Penta::MinVx(IssmDouble* pminvx){

	/*Get minimum:*/
	IssmDouble minvx=this->inputs->Min(VxEnum);

	/*Assign output pointers:*/
	*pminvx=minvx;
}
/*}}}*/
/*FUNCTION Penta::MinVy{{{*/
void  Penta::MinVy(IssmDouble* pminvy){

	/*Get minimum:*/
	IssmDouble minvy=this->inputs->Min(VyEnum);

	/*Assign output pointers:*/
	*pminvy=minvy;
}
/*}}}*/
/*FUNCTION Penta::MinVz{{{*/
void  Penta::MinVz(IssmDouble* pminvz){

	/*Get minimum:*/
	IssmDouble minvz=this->inputs->Min(VzEnum);

	/*Assign output pointers:*/
	*pminvz=minvz;
}
/*}}}*/
/*FUNCTION Penta::MassFlux {{{*/
IssmDouble Penta::MassFlux( IssmDouble* segment){

	IssmDouble mass_flux=0;

	if(!IsOnBed()) return mass_flux;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	mass_flux=tria->MassFlux(segment);
	delete tria->material; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*clean up and return*/
	return mass_flux;
}
/*}}}*/
/*FUNCTION Penta::MassFlux {{{*/
IssmDouble Penta::MassFlux( IssmDouble x1, IssmDouble y1, IssmDouble x2, IssmDouble y2,int segment_id){

	IssmDouble mass_flux=0;

	if(!IsOnBed()) return mass_flux;

	/*Depth Averaging Vx and Vy*/
	this->InputDepthAverageAtBase(VxEnum,VxAverageEnum);
	this->InputDepthAverageAtBase(VyEnum,VyAverageEnum);

	/*Spawn Tria element from the base of the Penta: */
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	mass_flux=tria->MassFlux(x1,y1,x2,y2,segment_id);
	delete tria->material; delete tria;

	/*Delete Vx and Vy averaged*/
	this->inputs->DeleteInput(VxAverageEnum);
	this->inputs->DeleteInput(VyAverageEnum);

	/*clean up and return*/
	return mass_flux;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVx{{{*/
void  Penta::MaxAbsVx(IssmDouble* pmaxabsvx){

	/*Get maximum:*/
	IssmDouble maxabsvx=this->inputs->MaxAbs(VxEnum);

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVy{{{*/
void  Penta::MaxAbsVy(IssmDouble* pmaxabsvy){

	/*Get maximum:*/
	IssmDouble maxabsvy=this->inputs->MaxAbs(VyEnum);

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Penta::MaxAbsVz{{{*/
void  Penta::MaxAbsVz(IssmDouble* pmaxabsvz){

	/*Get maximum:*/
	IssmDouble maxabsvz=this->inputs->MaxAbs(VzEnum);

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Penta::MaxVel{{{*/
void  Penta::MaxVel(IssmDouble* pmaxvel){

	/*Get maximum:*/
	IssmDouble maxvel=this->inputs->Max(VelEnum);

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Penta::MaxVx{{{*/
void  Penta::MaxVx(IssmDouble* pmaxvx){

	/*Get maximum:*/
	IssmDouble maxvx=this->inputs->Max(VxEnum);

	/*Assign output pointers:*/
	*pmaxvx=maxvx;
}
/*}}}*/
/*FUNCTION Penta::MaxVy{{{*/
void  Penta::MaxVy(IssmDouble* pmaxvy){

	/*Get maximum:*/
	IssmDouble maxvy=this->inputs->Max(VyEnum);

	/*Assign output pointers:*/
	*pmaxvy=maxvy;
}
/*}}}*/
/*FUNCTION Penta::MaxVz{{{*/
void  Penta::MaxVz(IssmDouble* pmaxvz){

	/*Get maximum:*/
	IssmDouble maxvz=this->inputs->Max(VzEnum);

	/*Assign output pointers:*/
	*pmaxvz=maxvz;
}
/*}}}*/
/*FUNCTION Penta::ElementResponse{{{*/
void Penta::ElementResponse(IssmDouble* presponse,int response_enum){

	switch(response_enum){
		case MaterialsRheologyBbarEnum:
			*presponse=this->material->GetBbar();
			break;
		case DamageDbarEnum:
			*presponse=this->material->GetDbar();
			break;
		case VelEnum:
			{

				/*Get input:*/
				IssmDouble vel;
				Input* vel_input;

				vel_input=this->inputs->GetInput(VelEnum); _assert_(vel_input);
				vel_input->GetInputAverage(&vel);

				/*Assign output pointers:*/
				*presponse=vel;
			}
			break;
		default:  
			_error_("Response type " << EnumToStringx(response_enum) << " not supported yet!");
	}

}
/*}}}*/
/*FUNCTION Penta::TotalSmb {{{*/
IssmDouble Penta::TotalSmb(void){

	/*The smb[Gt yr-1] of one element is area[m2] * smb [ m ice yr^-1] * rho_ice [kg m-3] / 1e+10^12 */
	IssmDouble base,smb,rho_ice;
	IssmDouble Total_Smb=0;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*Get material parameters :*/
	rho_ice=matpar->GetRhoIce();

	if(!IsIceInElement() || !IsOnSurface()) return 0.;

	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	/*First calculate the area of the base (cross section triangle)
	 * http://en.wikipedia.org/wiki/Triangle
	 * base = 1/2 abs((xA-xC)(yB-yA)-(xA-xB)(yC-yA))*/
	base = 1./2. * fabs((xyz_list[0][0]-xyz_list[2][0])*(xyz_list[1][1]-xyz_list[0][1]) - (xyz_list[0][0]-xyz_list[1][0])*(xyz_list[2][1]-xyz_list[0][1]));

	/*Now get the average SMB over the element*/
	Input* smb_input = inputs->GetInput(SurfaceforcingsMassBalanceEnum); _assert_(smb_input);

	smb_input->GetInputAverage(&smb);
	Total_Smb=rho_ice*base*smb;// smb on element in kg s-1

	/*Return: */
	return Total_Smb;
}
/*}}}*/

#ifdef _HAVE_GIA_
/*FUNCTION Penta::GiaDeflection {{{*/
void Penta::GiaDeflection(Vector<IssmDouble>* wg,Vector<IssmDouble>* dwgdt,IssmDouble* x,IssmDouble* y){
	_error_("GIA deflection not implemented yet!");
}
/*}}}*/
#endif

/*FUNCTION Penta::ControlInputGetGradient{{{*/
void Penta::ControlInputGetGradient(Vector<IssmDouble>* gradient,int enum_type,int control_index){

	int vertexpidlist[NUMVERTICES];
	Input* input=NULL;

	if(enum_type==MaterialsRheologyBbarEnum){
		if(!IsOnBed()) return;
		input=(Input*)material->inputs->GetInput(MaterialsRheologyBEnum);
	}
	else if(enum_type==DamageDbarEnum){
		if(!IsOnBed()) return;
		input=(Input*)material->inputs->GetInput(DamageDEnum);
	}
	else{
		input=inputs->GetInput(enum_type);
	}
	if (!input) _error_("Input " << EnumToStringx(enum_type) << " not found");
	if (input->ObjectEnum()!=ControlInputEnum) _error_("Input " << EnumToStringx(enum_type) << " is not a ControlInput");

	GradientIndexing(&vertexpidlist[0],control_index);
	((ControlInput*)input)->GetGradient(gradient,&vertexpidlist[0]);

}/*}}}*/
/*FUNCTION Penta::ControlInputScaleGradient{{{*/
void Penta::ControlInputScaleGradient(int enum_type,IssmDouble scale){

	Input* input=NULL;

	if(enum_type==MaterialsRheologyBbarEnum){
		input=(Input*)material->inputs->GetInput(MaterialsRheologyBEnum);
	}
	else if(enum_type==DamageDbarEnum){
		input=(Input*)material->inputs->GetInput(DamageDEnum);
	}
	else{
		input=inputs->GetInput(enum_type);
	}
	if (!input) _error_("Input " << EnumToStringx(enum_type) << " not found");
	if (input->ObjectEnum()!=ControlInputEnum) _error_("Input " << EnumToStringx(enum_type) << " is not a ControlInput");

	((ControlInput*)input)->ScaleGradient(scale);
}/*}}}*/
/*FUNCTION Penta::ControlInputSetGradient{{{*/
void Penta::ControlInputSetGradient(IssmDouble* gradient,int enum_type,int control_index){

	int    vertexpidlist[NUMVERTICES];
	IssmDouble grad_list[NUMVERTICES];
	Input* grad_input=NULL;
	Input* input=NULL;

	if(enum_type==MaterialsRheologyBbarEnum){
		input=(Input*)material->inputs->GetInput(MaterialsRheologyBEnum);
	}
	else if(enum_type==DamageDbarEnum){
		input=(Input*)material->inputs->GetInput(DamageDEnum);
	}
	else{
		input=inputs->GetInput(enum_type);
	}
	if (!input) _error_("Input " << EnumToStringx(enum_type) << " not found");
	if (input->ObjectEnum()!=ControlInputEnum) _error_("Input " << EnumToStringx(enum_type) << " is not a ControlInput");

	GradientIndexing(&vertexpidlist[0],control_index);
	for(int i=0;i<NUMVERTICES;i++) grad_list[i]=gradient[vertexpidlist[i]];
	grad_input=new PentaInput(GradientEnum,grad_list,P1Enum);
	((ControlInput*)input)->SetGradient(grad_input);

}/*}}}*/
/*FUNCTION Penta::ControlToVectors{{{*/
void Penta::ControlToVectors(Vector<IssmPDouble>* vector_control, Vector<IssmPDouble>* vector_gradient,int control_enum){

	Input* input=NULL;

	if(control_enum==MaterialsRheologyBbarEnum){
		input=(Input*)material->inputs->GetInput(MaterialsRheologyBEnum);
	}
	else if(control_enum==DamageDbarEnum){
		input=(Input*)material->inputs->GetInput(DamageDEnum);
	}
	else{
		input=inputs->GetInput(control_enum);
	}
	if (!input) _error_("Input " << EnumToStringx(control_enum) << " not found");
	if (input->ObjectEnum()!=ControlInputEnum) _error_("Input " << EnumToStringx(control_enum) << " is not a ControlInput");

	int         sidlist[NUMVERTICES];
	int         connectivity[NUMVERTICES];
	IssmPDouble values[NUMVERTICES];
	IssmPDouble gradients[NUMVERTICES]; 
	IssmDouble  value,gradient;

	this->GetVerticesConnectivityList(&connectivity[0]);
	this->GetVerticesSidList(&sidlist[0]);

	GaussPenta* gauss=new GaussPenta();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);

		((ControlInput*)input)->GetInputValue(&value,gauss);
		((ControlInput*)input)->GetGradientValue(&gradient,gauss);

		values[iv]    = reCast<IssmPDouble>(value)/reCast<IssmPDouble>(connectivity[iv]);
		gradients[iv] = reCast<IssmPDouble>(gradient)/reCast<IssmPDouble>(connectivity[iv]);
	}
	delete gauss;

	vector_control->SetValues(NUMVERTICES,&sidlist[0],&values[0],ADD_VAL);
	vector_gradient->SetValues(NUMVERTICES,&sidlist[0],&gradients[0],ADD_VAL);

}/*}}}*/
/*FUNCTION Penta::GradientIndexing{{{*/
void Penta::GradientIndexing(int* indexing,int control_index){

	/*Get some parameters*/
	int num_controls;
	parameters->FindParam(&num_controls,InversionNumControlParametersEnum);

	/*get gradient indices*/
	for(int i=0;i<NUMVERTICES;i++){
		indexing[i]=num_controls*this->vertices[i]->Pid() + control_index;
	}

}
/*}}}*/
/*FUNCTION Penta::Gradj {{{*/
void  Penta::Gradj(Vector<IssmDouble>* gradient,int control_type,int control_index){
	/*dJ/dalpha = ∂L/∂alpha = ∂J/∂alpha + ∂/∂alpha(KU-F)*/

	int   approximation;
	Tria* tria=NULL;

	/*If on water, skip grad (=0): */
	if(!IsIceInElement())return;
					
	/*First deal with ∂/∂alpha(KU-F)*/
	switch(control_type){

		case FrictionCoefficientEnum:
			inputs->GetInputValue(&approximation,ApproximationEnum);
			switch(approximation){
				case SSAApproximationEnum:
					GradjDragSSA(gradient,control_index);
					break;
				case HOApproximationEnum:
					GradjDragHO(gradient,control_index);
					break;
				case FSApproximationEnum:
					GradjDragFS(gradient,control_index);
					break;
				case NoneApproximationEnum:
					/*Gradient is 0*/
					break;
				default:
					_error_("approximation " << EnumToStringx(approximation) << " not supported yet");
			}
			break;

		case MaterialsRheologyBbarEnum:
			inputs->GetInputValue(&approximation,ApproximationEnum);
			switch(approximation){
				case SSAApproximationEnum:
					GradjBbarSSA(gradient,control_index);
					break;
				case HOApproximationEnum:
					GradjBbarHO(gradient,control_index);
					break;
				case FSApproximationEnum:
					GradjBbarFS(gradient,control_index);
					break;
				case NoneApproximationEnum:
					/*Gradient is 0*/
					break;
				default:
					_error_("approximation " << EnumToStringx(approximation) << " not supported yet");
			}
			break;

		default:
			_error_("control type " << EnumToStringx(control_type) << " not supported yet: ");
	}

	/*Now deal with ∂J/∂alpha*/
	int *responses = NULL;
	int num_responses,resp;
	this->parameters->FindParam(&num_responses,InversionNumCostFunctionsEnum);
	this->parameters->FindParam(&responses,NULL,InversionCostFunctionsEnum);

	for(resp=0;resp<num_responses;resp++) switch(responses[resp]){

		case ThicknessAbsMisfitEnum:
		case SurfaceAbsVelMisfitEnum:
		case SurfaceRelVelMisfitEnum:
		case SurfaceLogVelMisfitEnum:
		case SurfaceLogVxVyMisfitEnum:
		case SurfaceAverageVelMisfitEnum:
			/*Nothing, J does not depends on the parameter being inverted for*/
			break;
		case DragCoefficientAbsGradientEnum:
			if(IsOnBed()){
				tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
				tria->GradjDragGradient(gradient,control_index);
				delete tria->material; delete tria;
			}
			break;
		case RheologyBbarAbsGradientEnum:
			if(IsOnBed()){
				tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
				tria->GradjBGradient(gradient,control_index);
				delete tria->material; delete tria;
			}
			break;
		default:
			_error_("response " << EnumToStringx(responses[resp]) << " not supported yet");
	}
	xDelete<int>(responses);
}
/*}}}*/
/*FUNCTION Penta::GradjDragSSA {{{*/
void  Penta::GradjDragSSA(Vector<IssmDouble>* gradient,int control_index){

	/*Gradient is 0 if on shelf or not on bed*/
	if(IsFloating() || !IsOnBed()) return;

	/*Spawn tria*/
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->GradjDragSSA(gradient,control_index);
	delete tria->material; delete tria;

} /*}}}*/
/*FUNCTION Penta::GradjDragHO {{{*/
void  Penta::GradjDragHO(Vector<IssmDouble>* gradient,int control_index){

	int        i,j;
	int        analysis_type;
	int        vertexpidlist[NUMVERTICES];
	IssmDouble vx,vy,lambda,mu,alpha_complement,Jdet;
	IssmDouble bed,thickness,Neff,drag;
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble xyz_list_tria[NUMVERTICES2D][3]={0.0};
	IssmDouble dk[NDOF3]; 
	IssmDouble grade_g[NUMVERTICES]={0.0};
	IssmDouble grade_g_gaussian[NUMVERTICES];
	IssmDouble basis[6];
	Friction*  friction=NULL;
	GaussPenta *gauss=NULL;

	/*Gradient is 0 if on shelf or not on bed*/
	if(IsFloating() || !IsOnBed()) return;

	/*Retrieve all inputs and parameters*/
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	GradientIndexing(&vertexpidlist[0],control_index);
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<NUMVERTICES2D;i++) for(j=0;j<3;j++) xyz_list_tria[i][j]=xyz_list[i][j];
	Input* adjointx_input=inputs->GetInput(AdjointxEnum);               _assert_(adjointx_input);
	Input* adjointy_input=inputs->GetInput(AdjointyEnum);               _assert_(adjointy_input);
	Input* vx_input=inputs->GetInput(VxEnum);                           _assert_(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);                           _assert_(vy_input);
	Input* dragcoefficient_input=inputs->GetInput(FrictionCoefficientEnum); _assert_(dragcoefficient_input);

	/*Build frictoin element, needed later: */
	friction=new Friction(this,2);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(0,1,2,4);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetTriaJacobianDeterminant(&Jdet, &xyz_list_tria[0][0],gauss);
		GetNodalFunctionsP1(basis,gauss);

		/*Build alpha_complement_list: */
		friction->GetAlphaComplement(&alpha_complement,gauss,vx_input,vy_input,NULL);

		dragcoefficient_input->GetInputValue(&drag, gauss);
		adjointx_input->GetInputValue(&lambda, gauss);
		adjointy_input->GetInputValue(&mu, gauss);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		dragcoefficient_input->GetInputDerivativeValue(&dk[0],&xyz_list[0][0],gauss);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<NUMVERTICES;i++){
			grade_g_gaussian[i]=-2*drag*alpha_complement*((lambda*vx+mu*vy))*Jdet*gauss->weight*basis[i];
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for(i=0;i<NUMVERTICES;i++){
			_assert_(!xIsNan<IssmDouble>(grade_g[i]));
			grade_g[i]+=grade_g_gaussian[i];
		}
	}
	gradient->SetValues(NUMVERTICES,vertexpidlist,grade_g,ADD_VAL);

	/*Clean up and return*/
	delete gauss;
	delete friction;
}
/*}}}*/
/*FUNCTION Penta::GradjDragFS {{{*/
void  Penta::GradjDragFS(Vector<IssmDouble>* gradient,int control_index){

	int        i,j;
	int        analysis_type;
	int        vertexpidlist[NUMVERTICES];
	IssmDouble bed,thickness,Neff;
	IssmDouble lambda,mu,xi,Jdet,vx,vy,vz;
	IssmDouble alpha_complement,drag;
	IssmDouble surface_normal[3],bed_normal[3];
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble xyz_list_tria[NUMVERTICES2D][3]={0.0};
	IssmDouble dk[NDOF3]; 
	IssmDouble basis[6];
	IssmDouble grade_g[NUMVERTICES]={0.0};
	IssmDouble grade_g_gaussian[NUMVERTICES];
	Friction*  friction=NULL;
	GaussPenta* gauss=NULL;

	/*Gradient is 0 if on shelf or not on bed*/
	if(IsFloating() || !IsOnBed()) return;

	/*Retrieve all inputs and parameters*/
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	::GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	for(i=0;i<NUMVERTICES2D;i++) for(j=0;j<3;j++) xyz_list_tria[i][j]=xyz_list[i][j];
	GradientIndexing(&vertexpidlist[0],control_index);
	Input* drag_input    =inputs->GetInput(FrictionCoefficientEnum); _assert_(drag_input);
	Input* vx_input      =inputs->GetInput(VxEnum);                  _assert_(vx_input);
	Input* vy_input      =inputs->GetInput(VyEnum);                  _assert_(vy_input);
	Input* vz_input      =inputs->GetInput(VzEnum);                  _assert_(vz_input);
	Input* adjointx_input=inputs->GetInput(AdjointxEnum);            _assert_(adjointx_input);
	Input* adjointy_input=inputs->GetInput(AdjointyEnum);            _assert_(adjointy_input);
	Input* adjointz_input=inputs->GetInput(AdjointzEnum);            _assert_(adjointz_input);

	/*Build frictoin element, needed later: */
	friction=new Friction(this,3);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussPenta(0,1,2,4);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Recover alpha_complement and drag: */
		friction->GetAlphaComplement(&alpha_complement,gauss,vx_input,vy_input,vz_input);
		drag_input->GetInputValue(&drag,gauss);

		/*recover lambda mu and xi: */
		adjointx_input->GetInputValue(&lambda,gauss);
		adjointy_input->GetInputValue(&mu    ,gauss);
		adjointz_input->GetInputValue(&xi    ,gauss);

		/*recover vx vy and vz: */
		vx_input->GetInputValue(&vx, gauss);
		vy_input->GetInputValue(&vy, gauss);
		vz_input->GetInputValue(&vz, gauss);

		/*Get normal vector to the bed */
		NormalTop(&surface_normal[0],&xyz_list_tria[0][0]);

		bed_normal[0]=-surface_normal[0]; //Function is for upper surface, so the normal to the bed is the opposite of the result
		bed_normal[1]=-surface_normal[1];
		bed_normal[2]=-surface_normal[2];

		/* Get Jacobian determinant: */
		GetTriaJacobianDeterminant(&Jdet,&xyz_list_tria[0][0],gauss);
		GetNodalFunctionsP1(basis, gauss);

		/*Get k derivative: dk/dx */
		drag_input->GetInputDerivativeValue(&dk[0],&xyz_list[0][0],gauss);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<NUMVERTICES;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=(
						-lambda*(2*drag*alpha_complement*(vx - vz*bed_normal[0]*bed_normal[2]))
						-mu    *(2*drag*alpha_complement*(vy - vz*bed_normal[1]*bed_normal[2]))
						-xi    *(2*drag*alpha_complement*(-vx*bed_normal[0]*bed_normal[2]-vy*bed_normal[1]*bed_normal[2]))
						)*Jdet*gauss->weight*basis[i]; 
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<NUMVERTICES; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	gradient->SetValues(NUMVERTICES,vertexpidlist,grade_g,ADD_VAL);

	delete friction;
	delete gauss;
}
/*}}}*/
/*FUNCTION Penta::GradjBbarSSA {{{*/
void  Penta::GradjBbarSSA(Vector<IssmDouble>* gradient,int control_index){

	/*This element should be collapsed into a tria element at its base*/
	if (!IsOnBed()) return; 

	/*Depth Average B*/
	this->InputDepthAverageAtBase(MaterialsRheologyBEnum,MaterialsRheologyBbarEnum,MaterialsEnum);
	this->InputDepthAverageAtBase(DamageDEnum,DamageDbarEnum,MaterialsEnum);

	/*Collapse element to the base*/
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->GradjBSSA(gradient,control_index);
	delete tria->material; delete tria;

	/*delete Average B*/
	this->material->inputs->DeleteInput(MaterialsRheologyBbarEnum);
	this->material->inputs->DeleteInput(DamageDbarEnum);

} /*}}}*/
/*FUNCTION Penta::GradjBbarHO {{{*/
void  Penta::GradjBbarHO(Vector<IssmDouble>* gradient,int control_index){

	/*Gradient is computed on bed only (Bbar)*/
	if (!IsOnBed()) return;

	/*Depth Average B and D*/
	this->InputDepthAverageAtBase(MaterialsRheologyBEnum,MaterialsRheologyBbarEnum,MaterialsEnum);
	this->InputDepthAverageAtBase(DamageDEnum,DamageDbarEnum,MaterialsEnum);

	/*Collapse element to the base*/
	Tria* tria=(Tria*)SpawnTria(0);
	tria->GradjBSSA(gradient,control_index);    //We use SSA as an estimate for now
	delete tria->material; delete tria;

	/*delete Average B*/
	this->material->inputs->DeleteInput(MaterialsRheologyBbarEnum);
	this->material->inputs->DeleteInput(DamageDbarEnum);
} /*}}}*/
/*FUNCTION Penta::GradjBbarFS {{{*/
void  Penta::GradjBbarFS(Vector<IssmDouble>* gradient,int control_index){

	/*Gradient is computed on bed only (Bbar)*/
	if (!IsOnBed()) return;

	/*Depth Average B and D*/
	this->InputDepthAverageAtBase(MaterialsRheologyBEnum,MaterialsRheologyBbarEnum,MaterialsEnum);
	this->InputDepthAverageAtBase(DamageDEnum,DamageDbarEnum,MaterialsEnum);

	/*Collapse element to the base*/
	Tria* tria=(Tria*)SpawnTria(0);
	tria->GradjBSSA(gradient,control_index);    //We use SSA as an estimate for now
	delete tria->material; delete tria;

	/*delete Average B*/
	this->material->inputs->DeleteInput(MaterialsRheologyBbarEnum);
	this->material->inputs->DeleteInput(DamageDbarEnum);
} /*}}}*/
/*FUNCTION Penta::InputControlUpdate{{{*/
void  Penta::InputControlUpdate(IssmDouble scalar,bool save_parameter){

	/*Intermediary*/
	int    num_controls;
	int*   control_type=NULL;
	Input* input=NULL;

	/*retrieve some parameters: */
	this->parameters->FindParam(&num_controls,InversionNumControlParametersEnum);
	this->parameters->FindParam(&control_type,NULL,InversionControlParametersEnum);

	for(int i=0;i<num_controls;i++){

		if(control_type[i]==MaterialsRheologyBbarEnum){
			if (!IsOnBed()) goto cleanup_and_return;
			input=(Input*)material->inputs->GetInput(MaterialsRheologyBEnum); _assert_(input);
		}
		else if(control_type[i]==DamageDbarEnum){
			if (!IsOnBed()) goto cleanup_and_return;
			input=(Input*)material->inputs->GetInput(DamageDEnum); _assert_(input);
		}
		else{
			input=(Input*)this->inputs->GetInput(control_type[i]); _assert_(input);
		}

		if (input->ObjectEnum()!=ControlInputEnum) _error_("input " << EnumToStringx(control_type[i]) << " is not a ControlInput");

		((ControlInput*)input)->UpdateValue(scalar);
		((ControlInput*)input)->Constrain();
		if (save_parameter) ((ControlInput*)input)->SaveValue();

		if(control_type[i]==MaterialsRheologyBbarEnum){
			this->InputExtrude(MaterialsRheologyBEnum,MaterialsEnum);
		}
		else if(control_type[i]==DamageDbarEnum){
			this->InputExtrude(DamageDEnum,MaterialsEnum);
		}
	}

	/*Clean up and return*/
cleanup_and_return:
	xDelete<int>(control_type);
}
/*}}}*/
/*FUNCTION Penta::SurfaceAverageVelMisfit {{{*/
IssmDouble Penta::SurfaceAverageVelMisfit(void){

	int    approximation;
	IssmDouble J;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA and not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAverageVelMisfit*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		J=tria->SurfaceAverageVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		J=tria->SurfaceAverageVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceAbsVelMisfit {{{*/
IssmDouble Penta::SurfaceAbsVelMisfit(void){

	int    approximation;
	IssmDouble J;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA and not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceAbsVelMisfit*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		J=tria->SurfaceAbsVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		J=tria->SurfaceAbsVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVelMisfit {{{*/
IssmDouble Penta::SurfaceLogVelMisfit(void){

	int    approximation;
	IssmDouble J;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA and not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVelMisfit*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		J=tria->SurfaceLogVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		J=tria->SurfaceLogVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceLogVxVyMisfit {{{*/
IssmDouble Penta::SurfaceLogVxVyMisfit(void){

	IssmDouble J;
	Tria* tria=NULL;

	/*inputs: */
	int  approximation;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA and not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceLogVxVyMisfit*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		J=tria->SurfaceLogVxVyMisfit();
		delete tria->material; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		J=tria->SurfaceLogVxVyMisfit();
		delete tria->material; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::SurfaceRelVelMisfit {{{*/
IssmDouble Penta::SurfaceRelVelMisfit(void){

	int    approximation;
	IssmDouble J;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;

	/*Bail out if this element if:
	 * -> Non SSA and not on the surface
	 * -> SSA (2d model) and not on bed) */
	if ((approximation!=SSAApproximationEnum && !IsOnSurface()) || (approximation==SSAApproximationEnum && !IsOnBed())){
		return 0;
	}
	else if (approximation==SSAApproximationEnum){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute SurfaceRelVelMisfit*/
		tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
		J=tria->SurfaceRelVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(1); //lower face is 0, upper face is 1.
		J=tria->SurfaceRelVelMisfit();
		delete tria->material; delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION Penta::ThicknessAbsGradient{{{*/
IssmDouble Penta::ThicknessAbsGradient(void){

	_error_("Not implemented yet");
}
/*}}}*/
/*FUNCTION Penta::ThicknessAbsMisfit {{{*/
IssmDouble Penta::ThicknessAbsMisfit(void){

	int    approximation;
	IssmDouble J;
	Tria*  tria=NULL;

	/*retrieve inputs :*/
	inputs->GetInputValue(&approximation,ApproximationEnum);

	/*If on water, return 0: */
	if(!IsIceInElement())return 0;
	_error_("Not implemented yet");

	tria=(Tria*)SpawnTria(0);
	J=tria->ThicknessAbsMisfit();
	delete tria->material; delete tria;
	return J;
}
/*}}}*/
/*FUNCTION Penta::DragCoefficientAbsGradient{{{*/
IssmDouble Penta::DragCoefficientAbsGradient(void){

	IssmDouble J;
	Tria*  tria=NULL;

	/*If on water, on shelf or not on bed, skip: */
	if(!IsIceInElement()|| IsFloating() || !IsOnBed()) return 0;

	tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1
	J=tria->DragCoefficientAbsGradient();
	delete tria->material; delete tria;
	return J;
}
/*}}}*/
/*FUNCTION Penta::RheologyBbarAbsGradient{{{*/
IssmDouble Penta::RheologyBbarAbsGradient(void){

	IssmDouble J;
	Tria*  tria=NULL;

	/*If on water, on shelf or not on bed, skip: */
	if(!IsIceInElement() || !IsOnBed()) return 0;

	tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1
	J=tria->RheologyBbarAbsGradient();
	delete tria->material; delete tria;
	return J;
}
/*}}}*/
/*FUNCTION Penta::GetVectorFromControlInputs{{{*/
void  Penta::GetVectorFromControlInputs(Vector<IssmDouble>* vector,int control_enum,int control_index,const char* data){

	int vertexpidlist[NUMVERTICES];

	/*Get out if this is not an element input*/
	if(!IsInput(control_enum)) return;

	/*Prepare index list*/
	GradientIndexing(&vertexpidlist[0],control_index);

	/*Get input (either in element or material)*/
	Input* input=inputs->GetInput(control_enum);
	if(!input) _error_("Input " << EnumToStringx(control_enum) << " not found in element");

	/*Check that it is a ControlInput*/
	if (input->ObjectEnum()!=ControlInputEnum){
		_error_("input " << EnumToStringx(control_enum) << " is not a ControlInput");
	}

	((ControlInput*)input)->GetVectorFromInputs(vector,&vertexpidlist[0],data);
}
/*}}}*/
/*FUNCTION Penta::SetControlInputsFromVector{{{*/
void  Penta::SetControlInputsFromVector(IssmDouble* vector,int control_enum,int control_index){

	IssmDouble  values[NUMVERTICES];
	int     vertexpidlist[NUMVERTICES];
	Input  *input     = NULL;
	Input  *new_input = NULL;

	/*Get out if this is not an element input*/
	if(!IsInput(control_enum)) return;

	/*Prepare index list*/
	GradientIndexing(&vertexpidlist[0],control_index);

	/*Get values on vertices*/
	for (int i=0;i<NUMVERTICES;i++){
		values[i]=vector[vertexpidlist[i]];
	}
	new_input = new PentaInput(control_enum,values,P1Enum);

	if(control_enum==MaterialsRheologyBbarEnum){
		input=(Input*)material->inputs->GetInput(control_enum); _assert_(input);
	}
	else{
		input=(Input*)this->inputs->GetInput(control_enum);   _assert_(input);
	}

	if (input->ObjectEnum()!=ControlInputEnum){
		_error_("input " << EnumToStringx(control_enum) << " is not a ControlInput");
	}

	((ControlInput*)input)->SetInput(new_input);
}
/*}}}*/

#ifdef _HAVE_DAKOTA_
/*FUNCTION Penta::InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type);{{{*/
void  Penta::InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type){

	int i,j;

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			/*New PentaInput*/
			IssmDouble values[6];

			/*Get values on the 6 vertices*/
			for (i=0;i<6;i++){
				values[i]=vector[this->vertices[i]->Sid()]; //careful, vector of values here is not parallel distributed, but serial distributed (from a serial Dakota core!)
			}

			/*Branch on the specified type of update: */
			switch(name){
				case ThicknessEnum:
					/*Update thickness + surface: assume bed is constant. On ice shelves, takes hydrostatic equilibrium*/
					IssmDouble  thickness[6];
					IssmDouble  thickness_init[6];
					IssmDouble  hydrostatic_ratio[6];
					IssmDouble  surface[6];
					IssmDouble  bed[6];

					/*retrieve inputs: */
					GetInputListOnVertices(&thickness_init[0],ThicknessEnum);
					GetInputListOnVertices(&hydrostatic_ratio[0],GeometryHydrostaticRatioEnum);
					GetInputListOnVertices(&bed[0],BedEnum);
					GetInputListOnVertices(&surface[0],SurfaceEnum);

					/*build new thickness: */
//					for(j=0;j<6;j++)thickness[j]=values[j];

					/*build new bed and surface: */
					if (this->IsFloating()){
						/*hydrostatic equilibrium: */
						IssmDouble rho_ice,rho_water,di;
						rho_ice=this->matpar->GetRhoIce();
						rho_water=this->matpar->GetRhoWater();

						di=rho_ice/rho_water;

						/*build new thickness: */
						for (j=0; j<6; j++) {
						/*  for observed/interpolated/hydrostatic thickness, remove scaling from any hydrostatic thickness  */
							if     (hydrostatic_ratio[j] >= 0.)
								thickness[j]=values[j]-(values[j]/thickness_init[j]-1.)*hydrostatic_ratio[j]*surface[j]/(1.-di);
						/*  for minimum thickness, don't scale  */
							else
								thickness[j]=thickness_init[j];

						/*  check the computed thickness and update bed  */
							if (thickness[j] < 0.)
								thickness[j]=1./(1.-di);
							bed[j]=surface[j]-thickness[j];
						}

//						for(j=0;j<6;j++){
//							surface[j]=(1-di)*thickness[j];
//							bed[j]=-di*thickness[j];
//						}
					}
					else{
						/*build new thickness: */
						for (j=0; j<6; j++) {
						/*  for observed thickness, use scaled value  */
							if(hydrostatic_ratio[j] >= 0.)
								thickness[j]=values[j];
						/*  for minimum thickness, don't scale  */
							else
								thickness[j]=thickness_init[j];
						}

						/*update bed on grounded ice: */
//						for(j=0;j<6;j++)surface[j]=bed[j]+thickness[j];
						for(j=0;j<6;j++)bed[j]=surface[j]-thickness[j];
					}

					/*Add new inputs: */
					this->inputs->AddInput(new PentaInput(ThicknessEnum,thickness,P1Enum));
					this->inputs->AddInput(new PentaInput(BedEnum,bed,P1Enum));
					this->inputs->AddInput(new PentaInput(SurfaceEnum,surface,P1Enum));

					break;
				default:
					this->inputs->AddInput(new PentaInput(name,values,P1Enum));
			}
			break;

		default:
			_error_("type " << type << " (" << EnumToStringx(type) << ") not implemented yet");
	}

}
/*}}}*/
/*FUNCTION Penta::InputUpdateFromMatrixDakota(IssmDouble* matrix, int nrows, int ncols, int name, int type);{{{*/
void  Penta::InputUpdateFromMatrixDakota(IssmDouble* matrix, int nrows, int ncols, int name, int type){

	int             i,t,row;
	IssmDouble      time;
	TransientInput *transientinput = NULL;
	IssmDouble      values[6];

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:
			/*Create transient input: */
			for(t=0;t<ncols;t++){ //ncols is the number of times

				/*create input values: */
				for(i=0;i<6;i++){
					row=this->vertices[i]->Sid();
					values[i]=matrix[ncols*row+t];
				}

				/*time:*/
				time=matrix[(nrows-1)*ncols+t];

				if(t==0) transientinput=new TransientInput(name);
				transientinput->AddTimeInput(new PentaInput(name,values,P1Enum),time);
				transientinput->Configure(parameters);
			}
			this->inputs->AddInput(transientinput);
			break;

		default:
			_error_("type " << type << " (" << EnumToStringx(type) << ") not implemented yet");
	}

}
/*}}}*/
#endif

/*FUNCTION Penta::CreateEPLDomainMassMatrix {{{*/
ElementMatrix* Penta::CreateEPLDomainMassMatrix(void){

	if (!IsOnBed()) return NULL;

	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	ElementMatrix* Ke=tria->CreateEPLDomainMassMatrix();
	delete tria->material; delete tria;

	/*clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Penta::GetHydrologyDCInefficientHmax{{{*/
void  Penta::GetHydrologyDCInefficientHmax(IssmDouble* ph_max, Node* innode){

	if (!IsOnBed()) return;

	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->GetHydrologyDCInefficientHmax(ph_max,innode);
	delete tria->material; delete tria;
}
/*}}}*/
/*FUNCTION Penta::GetHydrologyTransfer{{{*/
void  Penta::GetHydrologyTransfer(Vector<IssmDouble>* transfer){

	if (!IsOnBed()) return;

	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->GetHydrologyTransfer(transfer);
	delete tria->material; delete tria;
}
/*}}}*/
/*FUNCTION Penta::GetSolutionFromInputsOneDof {{{*/
void Penta::GetSolutionFromInputsOneDof(Vector<IssmDouble>* solution, int enum_type){

	const int    numdof=NDOF1*NUMVERTICES;

	int          i;
	int*         doflist=NULL;
	IssmDouble   values[numdof];
	IssmDouble   enum_value;
	GaussPenta   *gauss=NULL;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	Input* enum_input=inputs->GetInput(enum_type); _assert_(enum_input);

	gauss=new GaussPenta();
	for(i=0;i<NUMVERTICES;i++){
		/*Recover temperature*/
		gauss->GaussVertex(i);
		enum_input->GetInputValue(&enum_value,gauss);
		values[i]=enum_value;
	}

	/*Add value to global vector*/
	solution->SetValues(numdof,doflist,values,INS_VAL);

	/*Free ressources:*/
	delete gauss;
	xDelete<int>(doflist);
}
/*}}}*/
/*FUNCTION Penta::HydrologyEPLGetActive {{{*/
void Penta::HydrologyEPLGetActive(Vector<IssmDouble>* active_vec){

	if (!IsOnBed()){
		return;
	}
	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->HydrologyEPLGetActive(active_vec);
	delete tria->material; delete tria;

}
/*}}}*/
/*FUNCTION Penta::HydrologyEPLGetMask{{{*/
void  Penta::HydrologyEPLGetMask(Vector<IssmDouble>* vec_mask){

	if (!IsOnBed())return;

	Tria* tria=(Tria*)SpawnTria(0); //lower face is 0, upper face is 1.
	tria->HydrologyEPLGetMask(vec_mask);
	delete tria->material; delete tria;

}
/*}}}*/
/*FUNCTION Penta::ComputeEPLThickness{{{*/
void  Penta::ComputeEPLThickness(void){

	int         i;
	const int   numdof   = NDOF1 *NUMVERTICES;
	const int   numdof2d = NDOF1 *NUMVERTICES2D;
	bool        isefficientlayer;
	IssmDouble  n,A,dt,init_thick;
	IssmDouble  rho_water,rho_ice;
	IssmDouble  gravity,latentheat,EPLgrad;
	IssmDouble  EPL_N,epl_conductivity;
	IssmDouble  activeEpl[numdof],thickness[numdof];
	IssmDouble  eplhead[numdof], old_thickness[numdof];
	IssmDouble  epl_slopeX[numdof],epl_slopeY[numdof];
	IssmDouble  ice_thickness[numdof],bed[numdof];
	Penta       *penta = NULL;
	/*If not on bed, return*/
	if (!IsOnBed())return;

	/*Get the flag to know if the efficient layer is present*/
	this->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	this->parameters->FindParam(&dt,TimesteppingTimeStepEnum);

	if(isefficientlayer){
		/*For now, assuming just one way to compute EPL thickness*/
		rho_water        = matpar->GetRhoWater();
		rho_ice          = matpar->GetRhoIce();
		gravity          = matpar->GetG();
		latentheat       = matpar->GetLatentHeat();
		epl_conductivity = matpar->GetEplConductivity();
		init_thick       = matpar->GetEplInitialThickness();
		n                = material->GetN();
		A                = material->GetA();
		
		GetInputListOnVertices(&activeEpl[0],HydrologydcMaskEplactiveNodeEnum);
		GetInputListOnVertices(&eplhead[0],EplHeadEnum);
		GetInputListOnVertices(&epl_slopeX[0],EplHeadSlopeXEnum); 
		GetInputListOnVertices(&epl_slopeY[0],EplHeadSlopeYEnum);
		GetInputListOnVertices(&old_thickness[0],HydrologydcEplThicknessOldEnum);
		GetInputListOnVertices(&ice_thickness[0],ThicknessEnum);
		GetInputListOnVertices(&bed[0],BedEnum);
		
		for(int i=0;i<numdof2d;i++){
			/*Keeping thickness to 1 if EPL is not active*/
			if(activeEpl[i]==0.0){
				thickness[i]=init_thick;
				thickness[i+numdof2d]=thickness[i];
			}
			else{

				/*Compute first the effective pressure in the EPL*/
				EPL_N=gravity*((rho_ice*ice_thickness[i])-(rho_water*(eplhead[i]-bed[i])));
				if(EPL_N<0.0)EPL_N=0.0;
				/*Get then the gradient of EPL heads*/
				EPLgrad = epl_slopeX[i]+epl_slopeY[i];
				
				/*And proceed to the real thing*/
				thickness[i] = old_thickness[i]*(1+((rho_water*gravity*dt)/(rho_ice*latentheat))*epl_conductivity*pow(EPLgrad,2.0)-2.0*(A*dt/(pow(n,n)))*(pow(EPL_N,n)));
				thickness[i+numdof2d]=thickness[i];
			}
		}
		penta=this;
		for(;;){

			/*Add input to the element: */			
			penta->inputs->AddInput(new PentaInput(HydrologydcEplThicknessEnum,thickness,P1Enum));
			
			/*Stop if we have reached the surface*/
			if (penta->IsOnSurface()) break;
			
			/* get upper Penta*/
			penta=penta->GetUpperPenta(); _assert_(penta->Id()!=this->id);
		}
	}
}
/*}}}*/

/*FUNCTION Penta::MigrateGroundingLine{{{*/
void  Penta::MigrateGroundingLine(IssmDouble* phi_ungrounding){

	int        i,migration_style;
	bool       oldfloating;
	bool       floatingelement = false;
	bool       groundedelement = false;
	IssmDouble bed_hydro,yts,gl_melting_rate;
	IssmDouble rho_water,rho_ice,density;
	IssmDouble h[NUMVERTICES],s[NUMVERTICES],b[NUMVERTICES],r[NUMVERTICES];
	IssmDouble melting[NUMVERTICES],phi[NUMVERTICES];

	if(!IsOnBed()) return;

	/*Recover info at the vertices: */
	parameters->FindParam(&migration_style,GroundinglineMigrationEnum);
	parameters->FindParam(&gl_melting_rate,GroundinglineMeltingRateEnum);
	parameters->FindParam(&yts,ConstantsYtsEnum);
	GetInputListOnVertices(&h[0],ThicknessEnum);
	GetInputListOnVertices(&s[0],SurfaceEnum);
	GetInputListOnVertices(&b[0],BedEnum);
	GetInputListOnVertices(&r[0],BathymetryEnum);
	GetInputListOnVertices(&phi[0],MaskGroundediceLevelsetEnum);
	rho_water   = matpar->GetRhoWater();
	rho_ice     = matpar->GetRhoIce();
	density     = rho_ice/rho_water;
	oldfloating = this->IsFloating(); //figure out if the element is floating before starting to change everything

	/*go through vertices, and update inputs, considering them to be PentaVertex type: */
	for(i=0;i<NUMVERTICES;i++){
		/*Ice shelf: if bed below bathymetry, impose it at the bathymetry and update surface, elso do nothing */
		if(phi[i]<=0){
			if(b[i]<=r[i]){ 
				b[i]        = r[i];
				s[i]        = b[i]+h[i];
			}
		}
		/*Ice sheet: if hydrostatic bed above bathymetry, ice sheet starts to unground, elso do nothing */
		/*Change only if AgressiveMigration or if the ice sheet is in contact with the ocean*/
		else{
			bed_hydro=-density*h[i];
			if(bed_hydro>r[i]){
				/*Unground only if the element is connected to the ice shelf*/
				if(migration_style==AgressiveMigrationEnum || migration_style==SubelementMigrationEnum || migration_style==SubelementMigration2Enum){
					s[i]        = (1-density)*h[i];
					b[i]        = -density*h[i];
				}
				else if(migration_style==SoftMigrationEnum && phi_ungrounding[vertices[i]->Pid()]<0.){
					s[i]        = (1-density)*h[i];
					b[i]        = -density*h[i];
				}
				else{
					if(migration_style!=SoftMigrationEnum) _error_("Error: migration should be Aggressive, Soft or Subelement");
				}
			}
		}
	}

	/*Recalculate phi*/
	for(i=0;i<NUMVERTICES;i++){
		if(migration_style==SoftMigrationEnum){
			bed_hydro=-density*h[i];
			if(phi[i]<0. || bed_hydro<=r[i] || phi_ungrounding[vertices[i]->Pid()]<0.){
				phi[i]=h[i]+r[i]/density;
			}
		}
		else phi[i]=h[i]+r[i]/density;
	}
	this->inputs->AddInput(new PentaInput(MaskGroundediceLevelsetEnum,&phi[0],P1Enum));
	this->InputExtrude(MaskGroundediceLevelsetEnum,ElementEnum);

	/*SubelementMigrationEnum: if one grounded, all grounded*/
	if(migration_style==SubelementMigrationEnum || migration_style==SubelementMigration2Enum){
		for(i=0;i<NUMVERTICES;i++){
			if(phi[i]>0.){
				groundedelement=true;
				break;
			}
		}
		floatingelement=!groundedelement;
	}
	else{
		for(i=0;i<NUMVERTICES;i++){
			if(phi[i]<=0.){
				floatingelement=true;
				break;
			}
		}
	}

   /*Add basal melting rate if element just ungrounded*/
	if(!oldfloating && floatingelement==true){
		for(i=0;i<NUMVERTICES;i++)melting[i]=gl_melting_rate/yts;
		this->inputs->AddInput(new PentaInput(BasalforcingsMeltingRateEnum,&melting[0],P1Enum));
	} 

	/*Update inputs*/
	this->inputs->AddInput(new PentaInput(SurfaceEnum,&s[0],P1Enum));
	this->inputs->AddInput(new PentaInput(BedEnum,&b[0],P1Enum));

	/*Extrude inputs*/
	this->InputExtrude(SurfaceEnum,ElementEnum);
	this->InputExtrude(BedEnum,ElementEnum);
}
/*}}}*/
/*FUNCTION Penta::PotentialUngrounding{{{*/
void  Penta::PotentialUngrounding(Vector<IssmDouble>* potential_ungrounding){

	IssmDouble  h[NUMVERTICES],r[NUMVERTICES],gl[NUMVERTICES];
	IssmDouble  bed_hydro;
	IssmDouble  rho_water,rho_ice,density;

	/*material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	density=rho_ice/rho_water;
	GetInputListOnVertices(&h[0],ThicknessEnum);
	GetInputListOnVertices(&r[0],BathymetryEnum);
	GetInputListOnVertices(&gl[0],MaskGroundediceLevelsetEnum);

	/*go through vertices, and figure out which ones are on the ice sheet, and want to unground: */
	for(int i=0;i<NUMVERTICES;i++){
		/*Find if grounded vertices want to start floating*/
		if (gl[i]>0.){
			bed_hydro=-density*h[i];
			if(bed_hydro>r[i]){
				/*Vertex that could potentially unground, flag it*/
				potential_ungrounding->SetValue(vertices[i]->Pid(),1,INS_VAL);
			}
		}
	}
}
/*}}}*/
/*FUNCTION Penta::UpdatePotentialUngrounding{{{*/
int Penta::UpdatePotentialUngrounding(IssmDouble* vertices_potentially_ungrounding,Vector<IssmDouble>* vec_nodes_on_iceshelf,IssmDouble* nodes_on_iceshelf){

	int i;
	int nflipped=0;

	/*Go through nodes, and whoever is on the potential_ungrounding, ends up in nodes_on_iceshelf: */
	for(i=0;i<NUMVERTICES;i++){
		if (reCast<bool,IssmDouble>(vertices_potentially_ungrounding[vertices[i]->Pid()])){
			vec_nodes_on_iceshelf->SetValue(vertices[i]->Pid(),-1.,INS_VAL);

			/*If node was not on ice shelf, we flipped*/
			if(nodes_on_iceshelf[vertices[i]->Pid()]>=0.){
				nflipped++;
			}
		}
	}
	return nflipped;
}
/*}}}*/
