%TRANSIENT class definition
%
%   Usage:
%      transient=transient();

classdef transient
	properties (SetAccess=public) 
		ismasstransport   = 0;
		isstressbalance   = 0;
		isthermal         = 0;
		isgroundingline   = 0;
		isgia             = 0;
		isdamage          = 0;
		islevelset        = 0;
		requested_outputs = {};
	end
	methods
		function obj = transient(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
			obj.ismasstransport = 1;
			obj.isstressbalance = 1;
			obj.isthermal       = 1;
			obj.isgroundingline = 0;
			obj.isgia           = 0;
			obj.isdamage        = 0;
			obj.islevelset      = 0;

			%default output
			obj.requested_outputs={'default'};
		end % }}}
		function list = defaultoutputs(self,md) % {{{

			list = {'SurfaceforcingsMassBalance'};

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if solution~=TransientSolutionEnum(), return; end

			md = checkfield(md,'fieldname','transient.ismasstransport','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isstressbalance','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isthermal','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgroundingline','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgia','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isdamage','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.islevelset','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.requested_outputs','stringrow',1);

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   transient solution parameters:'));

			fielddisplay(obj,'ismasstransport','indicates whether a masstransport solution is used in the transient');
			fielddisplay(obj,'isstressbalance','indicates whether a stressbalance solution is used in the transient');
			fielddisplay(obj,'isthermal','indicates whether a thermal solution is used in the transient');
			fielddisplay(obj,'isgroundingline','indicates whether a groundingline migration is used in the transient');
			fielddisplay(obj,'isgia','indicates whether a postglacial rebound model is used in the transient');
			fielddisplay(obj,'isdamage','indicates whether damage evolution is used in the transient');
			fielddisplay(obj,'islevelset','LEVEL SET DESCRIPTION...');
			fielddisplay(obj,'requested_outputs','list of additional outputs requested');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','ismasstransport','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isstressbalance','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isthermal','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isgroundingline','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isgia','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','isdamage','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','islevelset','format','Boolean');

			%process requested outputs
			outputs = obj.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(obj,md)]; %add defaults
			end
			WriteData(fid,'data',outputs,'enum',TransientRequestedOutputsEnum,'format','StringArray');
		end % }}}
	end
end
