%SMBpdd Class definition
%
%   Usage:
%      SMBpdd=SMBpdd();

classdef SMBpdd
	properties (SetAccess=public) 
		precipitation             = NaN;
		monthlytemperatures       = NaN;
		desfac                    = 0.5;
		s0p                       = 0;
		isdelta18o                = 0;
		delta18o                  = NaN;
		delta18o_surface          = NaN;
		temperatures_presentday   = NaN;
		temperatures_lgm          = NaN;
		precipitations_presentday = NaN;
	end
	methods
		function obj = SMBpdd(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = initialize(self,md) % {{{

			if isnan(self.precipitation),
				self.precipitation=zeros(md.mesh.numberofvertices,1);
				disp('      no SMBpdd.precipitation specified: values set as zero');
			end

		end % }}}
		function obj = setdefaultparameters(obj) % {{{

		  obj.isdelta18o = 0;
		  obj.desfac     = 0.5;
		  obj.s0p        = 0;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			if ismember(MasstransportAnalysisEnum(),analyses),
				md = checkfield(md,'fieldname','surfaceforcings.desfac','<=',1,'numel',1);
				md = checkfield(md,'fieldname','surfaceforcings.s0p','>=',0,'numel',1);
				if(obj.isdelta18o==0)
					md = checkfield(md,'fieldname','surfaceforcings.monthlytemperatures','forcing',1,'NaN',1);
					md = checkfield(md,'fieldname','surfaceforcings.precipitation','forcing',1,'NaN',1);
				else
					md = checkfield(md,'fieldname','surfaceforcings.delta18o','NaN',1);
					md = checkfield(md,'fieldname','surfaceforcings.delta18o_surface','NaN',1);
					md = checkfield(md,'fieldname','surfaceforcings.temperatures_presentday','size',[md.mesh.numberofvertices+1 12],'NaN',1);
					md = checkfield(md,'fieldname','surfaceforcings.temperatures_lgm','size',[md.mesh.numberofvertices+1 12],'NaN',1);
					md = checkfield(md,'fieldname','surfaceforcings.precipitations_presentday','size',[md.mesh.numberofvertices+1 12],'NaN',1);
				end
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   surface forcings parameters:'));

			disp(sprintf('\n   PDD and deltaO18 parameters:'));
			fielddisplay(obj,'isdelta18o','is temperature and precipitation delta18o parametrisation activated (0 or 1, default is 0)');
			fielddisplay(obj,'desfac','desertification elevation factor (between 0 and 1, default is 0.5) [m]');
			fielddisplay(obj,'s0p','should be set to elevation from precip source (between 0 and a few 1000s m, default is 0) [m]');
			fielddisplay(obj,'monthlytemperatures','monthly surface temperatures [K], required if pdd is activated and delta18o not activated');
			fielddisplay(obj,'precipitation','surface precipitation [m/yr water eq]');
			fielddisplay(obj,'temperatures_presentday','monthly present day surface temperatures [K], required if pdd is activated and delta18o activated');
			fielddisplay(obj,'temperatures_lgm','monthly LGM surface temperatures [K], required if pdd is activated and delta18o activated');
			fielddisplay(obj,'precipitations_presentday','monthly surface precipitation [m/yr water eq], required if pdd is activated and delta18o activated');
			fielddisplay(obj,'delta18o','delta18o, required if pdd is activated and delta18o activated');
			fielddisplay(obj,'delta18o_surface','surface elevation of the delta18o site, required if pdd is activated and delta18o activated');

		end % }}}
		function marshall(obj,md,fid) % {{{

			yts=365.0*24.0*3600.0;

			WriteData(fid,'enum',SurfaceforcingsEnum(),'data',SMBpddEnum(),'format','Integer');

			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','precipitation','format','DoubleMat','mattype',1,'scale',1./yts,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','desfac','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','s0p','format','Double');
			WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','isdelta18o','format','Boolean');
			if obj.isdelta18o
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','temperatures_presentday','format','DoubleMat','mattype',1);
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','temperatures_lgm','format','DoubleMat','mattype',1);
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','precipitations_presentday','format','DoubleMat','mattype',1);
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','delta18o_surface','format','DoubleMat','mattype',1);
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','delta18o','format','DoubleMat','mattype',1);
			else
				WriteData(fid,'object',obj,'class','surfaceforcings','fieldname','monthlytemperatures','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1);
			end
		end % }}}
	end
end
