#include "./StressbalanceSIAAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

/*Model processing*/
int  StressbalanceSIAAnalysis::DofsPerNode(int** doflist,int meshtype,int approximation){/*{{{*/
	return 2;
}/*}}}*/
void StressbalanceSIAAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	/*No specific parameters*/

}/*}}}*/
void StressbalanceSIAAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	/*Fetch data needed: */
	bool   isSIA;
	iomodel->Constant(&isSIA,FlowequationIsSIAEnum);

	/*Now, is the flag SIA on? otherwise, do nothing: */
	if (!isSIA)return;

	iomodel->FetchData(1,FlowequationElementEquationEnum);

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	iomodel->FetchDataToInput(elements,ThicknessEnum);
	iomodel->FetchDataToInput(elements,FrictionCoefficientEnum);

	/*Free data: */
	iomodel->DeleteData(1,FlowequationElementEquationEnum);
}/*}}}*/
void StressbalanceSIAAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	/*Intermediaries*/
	bool  isSIA;
	Node* node = NULL;

	/*Fetch parameters: */
	iomodel->Constant(&isSIA,FlowequationIsSIAEnum);

	/*Now, is the flag isSIA on? otherwise, do nothing: */
	if(!isSIA) return;

	/*First create nodes*/
	int    lid=0;
	iomodel->FetchData(6,MeshVertexonbedEnum,MeshVertexonsurfaceEnum,FlowequationBorderSSAEnum,FlowequationBorderFSEnum,
				FlowequationVertexEquationEnum,StressbalanceReferentialEnum);

	for(int i=0;i<iomodel->numberofvertices;i++){
		if(iomodel->my_vertices[i]){

			/*Create new node if is in this processor's partition*/
			node = new Node(iomodel->nodecounter+i+1,i,lid++,i,iomodel,StressbalanceSIAAnalysisEnum,reCast<int>(iomodel->Data(FlowequationVertexEquationEnum)[i]));

			/*Deactivate node if not SIA*/
			if(reCast<int>(iomodel->Data(FlowequationVertexEquationEnum)[i])!=SIAApproximationEnum){
				node->Deactivate();
			}

			/*Add to Nodes dataset*/
			nodes->AddObject(node);
		}
	}

	iomodel->DeleteData(6,MeshVertexonbedEnum,MeshVertexonsurfaceEnum,FlowequationBorderSSAEnum,FlowequationBorderFSEnum,
				FlowequationVertexEquationEnum,StressbalanceReferentialEnum);

}/*}}}*/
void StressbalanceSIAAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	/*Intermediary*/
	int        count;
	IssmDouble yts;
	bool       isSIA;

	/*Fetch parameters: */
	iomodel->Constant(&yts,ConstantsYtsEnum);
	iomodel->Constant(&isSIA,FlowequationIsSIAEnum);

	/*Now, is the flag isSIA on? otherwise, do nothing: */
	if (!isSIA) return;

	/*Fetch data: */
	iomodel->FetchData(3,StressbalanceSpcvxEnum,StressbalanceSpcvyEnum,FlowequationVertexEquationEnum);

	/*Initialize conunter*/
	count=0;

	/*vx and vy are spc'd if we are not on nodeonSIA: */
	for(int i=0;i<iomodel->numberofvertices;i++){
		/*keep only this partition's nodes:*/
		if((iomodel->my_vertices[i])){
			if (reCast<int,IssmDouble>(iomodel->Data(FlowequationVertexEquationEnum)[i])!=SIAApproximationEnum){

				constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,1,0,StressbalanceSIAAnalysisEnum));
				count++;

				constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,2,0,StressbalanceSIAAnalysisEnum));
				count++;
			}
			else{
				if (!xIsNan<IssmDouble>(iomodel->Data(StressbalanceSpcvxEnum)[i])){
					constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,1,iomodel->Data(StressbalanceSpcvxEnum)[i]/yts,StressbalanceSIAAnalysisEnum)); //add count'th spc, on node i+1, setting dof 1 to vx.
					count++;
				}

				if (!xIsNan<IssmDouble>(iomodel->Data(StressbalanceSpcvyEnum)[i])){
					constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,2,iomodel->Data(StressbalanceSpcvyEnum)[i]/yts,StressbalanceSIAAnalysisEnum)); //add count'th spc, on node i+1, setting dof 2 to vy
					count++;
				}
			}
		}
	}

	/*Free data: */
	iomodel->DeleteData(3,StressbalanceSpcvxEnum,StressbalanceSpcvyEnum,FlowequationVertexEquationEnum);

}/*}}}*/
void StressbalanceSIAAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	/*No loads*/

}/*}}}*/

/*Finite Element Analysis*/
void           StressbalanceSIAAnalysis::Core(FemModel* femmodel){/*{{{*/

		if(VerboseSolution()) _printf0_("   computing SIA velocities\n");
		femmodel->SetCurrentConfiguration(StressbalanceSIAAnalysisEnum);
		solutionsequence_linear(femmodel);
}/*}}}*/
ElementVector* StressbalanceSIAAnalysis::CreateDVector(Element* element){/*{{{*/
	/*Default, return NULL*/
	return NULL;
}/*}}}*/
ElementMatrix* StressbalanceSIAAnalysis::CreateJacobianMatrix(Element* element){/*{{{*/
_error_("Not implemented");
}/*}}}*/
ElementMatrix* StressbalanceSIAAnalysis::CreateKMatrix(Element* element){/*{{{*/
	int meshtype;
	element->FindParam(&meshtype,MeshTypeEnum);
	switch(meshtype){
		case Mesh2DhorizontalEnum:
			return CreateKMatrix2D(element);
		case Mesh3DEnum:
			return CreateKMatrix3D(element);
		default: _error_("mesh "<<EnumToStringx(meshtype)<<" not supported yet");
	}
}/*}}}*/
ElementMatrix* StressbalanceSIAAnalysis::CreateKMatrix2D(Element* element){/*{{{*/

	/*Intermediaries */
	IssmDouble connectivity;

	/*Fetch number vertices for this element*/
	int numvertices = element->GetNumberOfVertices();

	/*Initialize Element vector*/
	ElementMatrix* Ke=element->NewElementMatrix();
	for(int iv=0;iv<numvertices;iv++){
		connectivity=(IssmDouble)element->VertexConnectivity(iv);
		Ke->values[(2*iv+0)*2*numvertices+(2*iv+0)]=1./connectivity;
		Ke->values[(2*iv+1)*2*numvertices+(2*iv+1)]=1./connectivity;
	}

	/*Clean up and return*/
	return Ke;
}/*}}}*/
ElementMatrix* StressbalanceSIAAnalysis::CreateKMatrix3D(Element* element){/*{{{*/

	/*Intermediaries */
	int         i0,i1,j0,j1,nodeup,nodedown,numsegments;
	IssmDouble  slope[2],connectivity[2],one0,one1;
	int        *pairindices = NULL;

	/*Fetch number vertices for this element*/
	int numvertices = element->GetNumberOfVertices();
	int numdof      = 2*numvertices;

	/*Initialize Element vector*/
	ElementMatrix* Ke=element->NewElementMatrix();

	element->VerticalSegmentIndices(&pairindices,&numsegments);
	for(int is=0;is<numsegments;is++){
		nodedown = pairindices[is*2+0];
		nodeup   = pairindices[is*2+1];
		connectivity[0]=(IssmDouble)element->VertexConnectivity(nodedown);
		connectivity[1]=(IssmDouble)element->VertexConnectivity(nodeup);
		one0=1./connectivity[0];
		one1=1./connectivity[1];

		/*2 dofs of first node*/
		i0=2*nodedown;  i1=2*nodedown+1;
		/*2 dofs of second node*/
		j0=2*nodeup;    j1=2*nodeup+1;

		/*Create matrix for these two nodes*/
		if(element->IsOnBed() && element->IsOnSurface()){
			Ke->values[i0*numdof+i0] = +one0;
			Ke->values[i1*numdof+i1] = +one0;
			Ke->values[j0*numdof+i0] = -one1;
			Ke->values[j0*numdof+j0] = +one1;
			Ke->values[j1*numdof+i1] = -one1;
			Ke->values[j1*numdof+j1] = +one1;
		}
		else if(element->IsOnBed()){
			Ke->values[i0*numdof+i0] = one0;
			Ke->values[i1*numdof+i1] = one0;
			Ke->values[j0*numdof+i0] = -2.*one1;
			Ke->values[j0*numdof+j0] = +2.*one1;
			Ke->values[j1*numdof+i1] = -2.*one1;
			Ke->values[j1*numdof+j1] = +2.*one1;
		}
		else if(element->IsOnSurface()){
			Ke->values[j0*numdof+i0] = -one1;
			Ke->values[j0*numdof+j0] = +one1;
			Ke->values[j1*numdof+i1] = -one1;
			Ke->values[j1*numdof+j1] = +one1;
		}
		else{ //node is on two horizontal layers and beams include the values only once, so the have to use half of the connectivity
			Ke->values[j0*numdof+i0] = -2.*one1;
			Ke->values[j0*numdof+j0] = +2.*one1;
			Ke->values[j1*numdof+i1] = -2.*one1;
			Ke->values[j1*numdof+j1] = +2.*one1;
		}
	}

	/*Clean up and return*/
	xDelete<int>(pairindices);
	return Ke;
}/*}}}*/
ElementVector* StressbalanceSIAAnalysis::CreatePVector(Element* element){/*{{{*/

	int meshtype;
	element->FindParam(&meshtype,MeshTypeEnum);
	switch(meshtype){
		case Mesh2DhorizontalEnum:
			return CreatePVector2D(element);
		case Mesh3DEnum:
			return CreatePVector3D(element);
		default: _error_("mesh "<<EnumToStringx(meshtype)<<" not supported yet");
	}
}/*}}}*/
ElementVector* StressbalanceSIAAnalysis::CreatePVector2D(Element* element){/*{{{*/

	/*Intermediaries */
	IssmDouble ub,vb,slope2,drag,thickness,connectivity;
	IssmDouble slope[2];

	/*Fetch number vertices for this element*/
	int numvertices = element->GetNumberOfVertices();

	/*Initialize Element vector*/
	ElementVector* pe=element->NewElementVector();

	/*Retrieve all inputs and parameters*/
	IssmDouble  rho_ice    = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  gravity    = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble  n          = element->GetMaterialParameter(MaterialsRheologyNEnum);
	IssmDouble  B          = element->GetMaterialParameter(MaterialsRheologyBbarEnum);
	Input* slopex_input    = element->GetInput(SurfaceSlopeXEnum);        _assert_(slopex_input);
	Input* slopey_input    = element->GetInput(SurfaceSlopeYEnum);        _assert_(slopey_input);
	Input* thickness_input = element->GetInput(ThicknessEnum);            _assert_(thickness_input);
	Input* drag_input      = element->GetInput(FrictionCoefficientEnum);  _assert_(drag_input);

	Gauss* gauss=element->NewGauss();
	for(int iv=0;iv<numvertices;iv++){
		gauss->GaussVertex(iv);

		connectivity=(IssmDouble)element->VertexConnectivity(iv);

		thickness_input->GetInputValue(&thickness,gauss);
		drag_input->GetInputValue(&drag,gauss);
		slopex_input->GetInputValue(&slope[0],gauss);
		slopey_input->GetInputValue(&slope[1],gauss);
		slope2=slope[0]*slope[0]+slope[1]*slope[1];

		/*Payne 1995 (m = 1, B = 5e-3/yts = 1.58e-10  m/s/Pa*/
		ub=-1.58*1.e-10*rho_ice*gravity*thickness*slope[0];
		vb=-1.58*1.e-10*rho_ice*gravity*thickness*slope[1];
		///*Ritz et al. 1996*/
		//ub=drag*(rho_ice*gravity*thickness)*(rho_ice*gravity*thickness)*slope[0]/sqrt(slope2);
		//vb=drag*(rho_ice*gravity*thickness)*(rho_ice*gravity*thickness)*slope[1]/sqrt(slope2);
		///*Rutt et al. 2009*/
		//ub=-drag*rho_ice*gravity*thickness*slope[0];
		//vb=-drag*rho_ice*gravity*thickness*slope[1];

		pe->values[2*iv+0]=(ub-2.*pow(rho_ice*gravity,n)*pow(slope2,((n-1.)/2.))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[0])/connectivity;
		pe->values[2*iv+1]=(vb-2.*pow(rho_ice*gravity,n)*pow(slope2,((n-1.)/2.))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[1])/connectivity;
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}/*}}}*/
ElementVector* StressbalanceSIAAnalysis::CreatePVector3D(Element* element){/*{{{*/

	/*Intermediaries */
	int         nodeup,nodedown,numsegments;
	IssmDouble  ub,vb,slope2,drag,surface,thickness,constant_part,z,Jdet;
	IssmDouble  slope[2],connectivity[2],xyz_list_line[2][3];
	IssmDouble *xyz_list = NULL;
	int        *pairindices = NULL;

	/*Fetch number vertices for this element*/
	int numvertices = element->GetNumberOfVertices();

	/*Initialize Element vector*/
	ElementVector* pe=element->NewElementVector();

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	IssmDouble  rho_ice    = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  gravity    = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble  n          = element->GetMaterialParameter(MaterialsRheologyNEnum);
	IssmDouble  B          = element->GetMaterialParameter(MaterialsRheologyBEnum);
	Input* surface_input   = element->GetInput(SurfaceEnum);              _assert_(surface_input);
	Input* slopex_input    = element->GetInput(SurfaceSlopeXEnum);        _assert_(slopex_input);
	Input* slopey_input    = element->GetInput(SurfaceSlopeYEnum);        _assert_(slopey_input);
	Input* thickness_input = element->GetInput(ThicknessEnum);            _assert_(thickness_input);
	Input* drag_input      = element->GetInput(FrictionCoefficientEnum);  _assert_(drag_input);

	/*Get Vertical segment indices*/
	element->VerticalSegmentIndices(&pairindices,&numsegments);
	for(int is=0;is<numsegments;is++){
		nodedown = pairindices[is*2+0];
		nodeup   = pairindices[is*2+1];
		connectivity[0]=(IssmDouble)element->VertexConnectivity(nodedown);
		connectivity[1]=(IssmDouble)element->VertexConnectivity(nodeup);
		for(int i=0;i<3;i++){
			xyz_list_line[0][i]=xyz_list[nodedown*3+i];
			xyz_list_line[1][i]=xyz_list[nodeup*3+i];
		}

		Gauss* gauss=element->NewGaussLine(nodedown,nodeup,3);
		for(int ig=gauss->begin();ig<gauss->end();ig++){
			gauss->GaussPoint(ig);

			slopex_input->GetInputValue(&slope[0],gauss);
			slopey_input->GetInputValue(&slope[1],gauss);
			surface_input->GetInputValue(&surface,gauss);
			thickness_input->GetInputValue(&thickness,gauss);

			slope2=slope[0]*slope[0]+slope[1]*slope[1];
			constant_part=-2.*pow(rho_ice*gravity,n)*pow(slope2,((n-1.)/2.));

			z = element->GetZcoord(gauss);
			element->JacobianDeterminantLine(&Jdet,&xyz_list_line[0][0],gauss);

			if(element->IsOnSurface()){
				pe->values[2*nodeup+0]+=constant_part*pow((surface-z)/B,n)*slope[0]*Jdet*gauss->weight/connectivity[1];
				pe->values[2*nodeup+1]+=constant_part*pow((surface-z)/B,n)*slope[1]*Jdet*gauss->weight/connectivity[1];
			}
			else{/*connectivity is too large, should take only half on it*/
				pe->values[2*nodeup+0]+=constant_part*pow((surface-z)/B,n)*slope[0]*Jdet*gauss->weight*2./connectivity[1];
				pe->values[2*nodeup+1]+=constant_part*pow((surface-z)/B,n)*slope[1]*Jdet*gauss->weight*2./connectivity[1];
			}
		}

		/*Deal with basal velocities*/
		if(element->IsOnBed()){
			drag_input->GetInputValue(&drag,gauss);

			/*Payne 1995 (m = 1, B = 5e-3/yts = 1.58e-10  m/s/Pa*/
			ub=-1.58*1.e-10*rho_ice*gravity*thickness*slope[0];
			vb=-1.58*1.e-10*rho_ice*gravity*thickness*slope[1];
			///*Ritz et al. 1996*/
			//ub=drag*(rho_ice*gravity*thickness)*(rho_ice*gravity*thickness)*slope[0]/sqrt(slope2);
			//vb=drag*(rho_ice*gravity*thickness)*(rho_ice*gravity*thickness)*slope[1]/sqrt(slope2);
			///*Rutt et al. 2009*/
			//ub=-drag*rho_ice*gravity*thickness*slope[0];
			//vb=-drag*rho_ice*gravity*thickness*slope[1];

			pe->values[2*nodedown+0]+=ub/connectivity[0];
			pe->values[2*nodedown+1]+=vb/connectivity[0];
		}
		delete gauss;
	}

	/*Clean up and return*/
	xDelete<int>(pairindices);
	xDelete<IssmDouble>(xyz_list);
	return pe;

}/*}}}*/
void StressbalanceSIAAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/

	IssmDouble vx,vy;
	int       *doflist = NULL;

	/*Fetch number of nodes and initialize values*/
	int         numnodes = element->GetNumberOfNodes();
	int         numdof   = numnodes*2;
	IssmDouble* values   = xNew<IssmDouble>(numdof);

	/*Get dof list and inputs */
	element->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	Input* vx_input=element->GetInput(VxEnum); _assert_(vx_input);
	Input* vy_input=element->GetInput(VyEnum); _assert_(vy_input);

	/*Ok, we have the velocities in inputs, fill in solution */
	Gauss* gauss=element->NewGauss();
	for(int i=0;i<numnodes;i++){
		gauss->GaussVertex(i);
		vx_input->GetInputValue(&vx,gauss);
		vy_input->GetInputValue(&vy,gauss);
		values[i*2+0]=vx;
		values[i*2+1]=vy;
	}

	/*Add value to global vector*/
	solution->SetValues(numdof,doflist,values,INS_VAL);

	/*Free ressources:*/
	delete gauss;
	xDelete<int>(doflist);
	xDelete<IssmDouble>(values);
}/*}}}*/
void StressbalanceSIAAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	int         i,meshtype;
	IssmDouble  rho_ice,g;
	int*        doflist=NULL;
	IssmDouble* xyz_list=NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();
	int numdof   = numnodes*2;

	/*Fetch dof list and allocate solution vectors*/
	element->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numdof);
	IssmDouble* vx        = xNew<IssmDouble>(numdof);
	IssmDouble* vy        = xNew<IssmDouble>(numdof);
	IssmDouble* vz        = xNew<IssmDouble>(numdof);
	IssmDouble* vel       = xNew<IssmDouble>(numdof);
	IssmDouble* pressure  = xNew<IssmDouble>(numdof);
	IssmDouble* thickness = xNew<IssmDouble>(numdof);
	IssmDouble* surface   = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Transform solution in Cartesian Space*/
	element->TransformSolutionCoord(&values[0],XYEnum);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numnodes;i++){
		vx[i]=values[i*NDOF2+0];
		vy[i]=values[i*NDOF2+1];

		/*Check solution*/
		if(xIsNan<IssmDouble>(vx[i])) _error_("NaN found in solution vector");
		if(xIsNan<IssmDouble>(vy[i])) _error_("NaN found in solution vector");
	}

	/*Get Vz and compute vel*/
	element->GetInputListOnNodes(&vz[0],VzEnum,0.);
	for(i=0;i<numnodes;i++) vel[i]=sqrt(vx[i]*vx[i] + vy[i]*vy[i] + vz[i]*vz[i]);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 2D, 
	 *so the pressure is just the pressure at the bedrock: */
	rho_ice  = element->GetMaterialParameter(MaterialsRhoIceEnum);
	g        = element->GetMaterialParameter(ConstantsGEnum);
	element->FindParam(&meshtype,MeshTypeEnum);
	switch(meshtype){
		case Mesh2DhorizontalEnum:
			element->GetInputListOnNodes(&thickness[0],ThicknessEnum);
			for(i=0;i<numnodes;i++) pressure[i]=rho_ice*g*thickness[i];
			break;
		case Mesh3DEnum:   
			element->GetVerticesCoordinates(&xyz_list);
			element->GetInputListOnNodes(&surface[0],SurfaceEnum);
			for(i=0;i<numnodes;i++) pressure[i]=rho_ice*g*(surface[i]-xyz_list[i*3+2]);
			break;
		default: _error_("mesh "<<EnumToStringx(meshtype)<<" not supported yet");
	}

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	element->InputChangeName(VxEnum,VxPicardEnum);
	element->InputChangeName(VyEnum,VyPicardEnum);
	element->InputChangeName(PressureEnum,PressurePicardEnum);

	/*Add vx and vy as inputs to the tria element: */
	element->AddInput(VxEnum,vx,P1Enum);
	element->AddInput(VyEnum,vy,P1Enum);
	element->AddInput(VelEnum,vel,P1Enum);
	element->AddInput(PressureEnum,pressure,P1Enum);

	/*Free ressources:*/
	xDelete<IssmDouble>(thickness);
	xDelete<IssmDouble>(surface);
	xDelete<IssmDouble>(pressure);
	xDelete<IssmDouble>(vel);
	xDelete<IssmDouble>(vz);
	xDelete<IssmDouble>(vy);
	xDelete<IssmDouble>(vx);
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(xyz_list);
	xDelete<int>(doflist);
}/*}}}*/
