/*!\file Element.cpp
 * \brief: implementation of the Element object
 */
/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif
#include <stdio.h>
#include <string.h>
#include "../classes.h"
#include "../../shared/shared.h"
/*}}}*/

/*Constructors/destructor/copy*/
Element::Element(){/*{{{*/
	this->inputs     = NULL;
	this->nodes      = NULL;
	this->vertices   = NULL;
	this->material   = NULL;
	this->matpar     = NULL;
	this->inputs     = NULL;
	this->parameters = NULL;
}/*}}}*/
Element::~Element(){/*{{{*/
	delete inputs;
}
/*}}}*/

/*Other*/
void Element::AddInput(Input* input_in){/*{{{*/

	/*Call inputs method*/
	_assert_(this->inputs);
	this->inputs->AddInput(input_in);
}/*}}}*/
bool Element::AllActive(void){/*{{{*/

	int numnodes = this->GetNumberOfNodes();
	for(int i=0;i<numnodes;i++){
		if(!this->nodes[i]->IsActive()) return false;
	}
	return true;
}/*}}}*/
bool Element::AnyActive(void){/*{{{*/

	int numnodes = this->GetNumberOfNodes();
	for(int i=0;i<numnodes;i++){
		if(this->nodes[i]->IsActive()) return true;
	}
	return false;
}/*}}}*/
void Element::DeleteMaterials(void){/*{{{*/
	delete this->material;
}/*}}}*/
void Element::FindParam(bool* pvalue,int paramenum){/*{{{*/
	this->parameters->FindParam(pvalue,paramenum);
}/*}}}*/
void Element::FindParam(int* pvalue,int paramenum){/*{{{*/
	this->parameters->FindParam(pvalue,paramenum);
}/*}}}*/
void Element::FindParam(IssmDouble* pvalue,int paramenum){/*{{{*/
	this->parameters->FindParam(pvalue,paramenum);
}/*}}}*/
void Element::FindParam(int** pvalues,int* psize,int paramenum){/*{{{*/
	this->parameters->FindParam(pvalues,psize,paramenum);
}/*}}}*/
IssmDouble Element::GetMaterialParameter(int enum_in){/*{{{*/

	_assert_(this->matpar);
	switch(enum_in){ // FIXME: change this to material
		case MaterialsRheologyNEnum:
			return this->material->GetN();
		case MaterialsRheologyBEnum:
			return this->material->GetB();
		case MaterialsRheologyBbarEnum:
			return this->material->GetBbar();
		default:
			return this->matpar->GetMaterialParameter(enum_in);
	}
}
/*}}}*/
Input* Element::GetInput(int inputenum){/*{{{*/
	return inputs->GetInput(inputenum);
}/*}}}*/
bool Element::IsFloating(){/*{{{*/

	bool shelf;
	int  migration_style;
	parameters->FindParam(&migration_style,GroundinglineMigrationEnum);

	if(migration_style==SubelementMigrationEnum || migration_style==SubelementMigration2Enum){ //Floating if all nodes are floating
		if(this->inputs->Max(MaskGroundediceLevelsetEnum) <= 0.) shelf=true;
		else shelf=false;
	}
	else if(migration_style==NoneEnum || migration_style==AgressiveMigrationEnum || migration_style==SoftMigrationEnum){ //Floating if all nodes are floating
		if(this->inputs->Min(MaskGroundediceLevelsetEnum) > 0.) shelf=false;
		else shelf=true;
	}
	else _error_("migration_style not implemented yet");

	return shelf;
}/*}}}*/
IssmDouble Element::TMeltingPoint(IssmDouble pressure){/*{{{*/
	_assert_(matpar);
	return this->matpar->TMeltingPoint(pressure);
}/*}}}*/

void Element::ViscosityFS(IssmDouble* pviscosity,int dim,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input){/*{{{*/

	/*Intermediaries*/
	IssmDouble viscosity;
	IssmDouble epsilon3d[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	IssmDouble epsilon2d[3]; /* epsilon=[exx,eyy,exy];            */

	if(dim==3){
		/*3D*/
		this->StrainRateFS(&epsilon3d[0],xyz_list,gauss,vx_input,vy_input,vz_input);
		material->GetViscosity3dFS(&viscosity, &epsilon3d[0]);
	}
	else{
		/*2D*/
		this->StrainRateSSA(&epsilon2d[0],xyz_list,gauss,vx_input,vy_input);
		material->GetViscosity2dvertical(&viscosity,&epsilon2d[0]);
	}

	/*Assign output pointer*/
	*pviscosity=viscosity;
}
/*}}}*/
void Element::ViscosityL1L2(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* surface_input){/*{{{*/
	/*Compute the L1L2 viscosity
	 *
	 *      1
	 * mu = - A^-1 (sigma'_e)^(1-n)
	 *      2
	 *
	 * sigma'_e^2 = |sigma'_//|^2 + |sigma'_perp|^2 (see Perego 2012 eq. 17,18)
	 *
	 * L1L2 assumptions:
	 *
	 * (1) |eps_b|_// = A (|sigma'_//|^2 + |sigma'_perp|^2)^((n-1)/2) |sigma'_//|
	 * (2) |sigma'_perp|^2 = |rho g (s-z) grad(s)|^2
	 *
	 * Assuming that n = 3, we have a polynom of degree 3 to solve (the only unkown is X=|sigma'_//|)
	 *
	 * A X^3 + A |rho g (s-z) grad(s)|^2 X - |eps_b|_// = 0     */

	int        i;
	IssmDouble z,s,viscosity,p,q,delta;
	IssmDouble tau_perp,tau_par,eps_b,A;
	IssmDouble epsilonvx[5]; /*exx eyy exy exz eyz*/
	IssmDouble epsilonvy[5]; /*exx eyy exy exz eyz*/
	IssmDouble epsilon[5];   /*exx eyy exy exz eyz*/
	IssmDouble slope[3];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input || !surface_input) _error_("Input missing");

	/*Get tau_perp*/
	surface_input->GetInputValue(&s,gauss);
	surface_input->GetInputDerivativeValue(&slope[0],xyz_list,gauss);
	z=GetZcoord(gauss);
	tau_perp = matpar->GetRhoIce() * matpar->GetG() * fabs(s-z)*sqrt(slope[0]*slope[0]+slope[1]*slope[1]);

	/* Get eps_b*/
	vx_input->GetVxStrainRate3dHO(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3dHO(epsilonvy,xyz_list,gauss);
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];
	eps_b = sqrt(epsilon[0]*epsilon[0] + epsilon[1]*epsilon[1] + epsilon[0]*epsilon[1] + epsilon[2]*epsilon[2]);
	if(eps_b==0.){
		*pviscosity = 2.5e+17;
		return;
	}

	/*Get A*/
	_assert_(material->GetN()==3.0);
	A=material->GetA();

	/*Solve for tau_perp (http://fr.wikipedia.org/wiki/Méthode_de_Cardan)*/
	p     = tau_perp *tau_perp;
	q     = - eps_b/A;
	delta = q *q + p*p*p*4./27.;
	_assert_(delta>0);
	tau_par = pow(0.5*(-q+sqrt(delta)),1./3.) - pow(0.5*(q+sqrt(delta)),1./3.);

	/*Viscosity*/
	viscosity = 1./(2.*A*(tau_par*tau_par + tau_perp*tau_perp));
	_assert_(!xIsNan(viscosity));
	_assert_(viscosity > 0.);

	/*Assign output pointer*/
	*pviscosity = viscosity;
}/*}}}*/
void Element::ViscosityHO(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/

	/*Intermediaries*/
	IssmDouble viscosity;
	IssmDouble epsilon[5];/* epsilon=[exx,eyy,exy,exz,eyz];*/

	this->StrainRateHO(&epsilon[0],xyz_list,gauss,vx_input,vy_input);
	material->GetViscosity3d(&viscosity, &epsilon[0]);

	/*Assign output pointer*/
	*pviscosity=viscosity;
}/*}}}*/
void Element::ViscositySSA(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/

	/*Intermediaries*/
	IssmDouble viscosity;
	IssmDouble epsilon[3];/* epsilon=[exx,eyy,exy];    */

	this->StrainRateSSA(&epsilon[0],xyz_list,gauss,vx_input,vy_input);
	material->GetViscosity2d(&viscosity, &epsilon[0]);

	/*Assign output pointer*/
	*pviscosity=viscosity;
}/*}}}*/
void Element::ViscositySSADerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	this->material->GetViscosity2dDerivativeEpsSquare(pmu_prime,epsilon);
}/*}}}*/
void Element::ViscosityHODerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	this->material->GetViscosityDerivativeEpsSquare(pmu_prime,epsilon);
}/*}}}*/
void Element::ViscosityFSDerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	this->material->GetViscosityDerivativeEpsSquare(pmu_prime,epsilon);
}/*}}}*/
void Element::StrainRateFS(IssmDouble* epsilon,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input){/*{{{*/
	/*Compute the 3d Strain Rate (6 components):
	 *
	 * epsilon=[exx eyy ezz exy exz eyz]
	 */

	IssmDouble epsilonvx[6];
	IssmDouble epsilonvy[6];
	IssmDouble epsilonvz[6];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input || !vz_input){
		_error_("Input missing. Here are the input pointers we have for vx: " << vx_input << ", vy: " << vy_input << ", vz: " << vz_input << "\n");
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3d(epsilonvy,xyz_list,gauss);
	vz_input->GetVzStrainRate3d(epsilonvz,xyz_list,gauss);

	/*Sum all contributions*/
	for(int i=0;i<6;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i]+epsilonvz[i];
}/*}}}*/
void Element::StrainRateHO(IssmDouble* epsilon,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/
	/*Compute the 3d Blatter/HOStrain Rate (5 components):
	 *
	 * epsilon=[exx eyy exy exz eyz]
	 *
	 * with exz=1/2 du/dz
	 *      eyz=1/2 dv/dz
	 *
	 * the contribution of vz is neglected
	 */

	int i;
	IssmDouble epsilonvx[5];
	IssmDouble epsilonvy[5];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		_error_("Input missing. Here are the input pointers we have for vx: " << vx_input << ", vy: " << vy_input << "\n");
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate3dHO(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate3dHO(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];
}/*}}}*/
void Element::StrainRateSSA(IssmDouble* epsilon,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/

	int i;
	IssmDouble epsilonvx[3];
	IssmDouble epsilonvy[3];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		_error_("Input missing. Here are the input pointers we have for vx: " << vx_input << ", vy: " << vy_input << "\n");
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate2d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate2d(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<3;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];
}/*}}}*/
void Element::TransformLoadVectorCoord(ElementVector* pe,int transformenum){/*{{{*/
	::TransformLoadVectorCoord(pe,this->nodes,this->GetNumberOfNodes(),transformenum);
}/*}}}*/
void Element::TransformLoadVectorCoord(ElementVector* pe,int* transformenum_list){/*{{{*/
	::TransformLoadVectorCoord(pe,this->nodes,this->GetNumberOfNodes(),transformenum_list);
}/*}}}*/
void Element::TransformSolutionCoord(IssmDouble* values,int transformenum){/*{{{*/
	::TransformSolutionCoord(values,this->nodes,this->GetNumberOfNodes(),transformenum);
}/*}}}*/
void Element::TransformSolutionCoord(IssmDouble* values,int* transformenum_list){/*{{{*/
	::TransformSolutionCoord(values,this->nodes,this->GetNumberOfNodes(),transformenum_list);
}/*}}}*/
void Element::TransformSolutionCoord(IssmDouble* values,int numnodes,int transformenum){/*{{{*/
	::TransformSolutionCoord(values,this->nodes,numnodes,transformenum);
}/*}}}*/
void Element::TransformSolutionCoord(IssmDouble* values,int numnodes,int* transformenum_list){/*{{{*/
	::TransformSolutionCoord(values,this->nodes,numnodes,transformenum_list);
}/*}}}*/
void Element::TransformStiffnessMatrixCoord(ElementMatrix* Ke,int transformenum){/*{{{*/
	::TransformStiffnessMatrixCoord(Ke,this->nodes,this->GetNumberOfNodes(),transformenum);
}/*}}}*/
void Element::TransformStiffnessMatrixCoord(ElementMatrix* Ke,int* transformenum_list){/*{{{*/
	::TransformStiffnessMatrixCoord(Ke,this->nodes,this->GetNumberOfNodes(),transformenum_list);
}/*}}}*/
void Element::TransformStiffnessMatrixCoord(ElementMatrix* Ke,Node** nodes_list,int numnodes,int* transformenum_list){/*{{{*/
	::TransformStiffnessMatrixCoord(Ke,nodes_list,numnodes,transformenum_list);
}/*}}}*/
