#include "./EnthalpyAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  EnthalpyAnalysis::DofsPerNode(int** doflist,int meshtype,int approximation){/*{{{*/
	return 1;
}/*}}}*/
void EnthalpyAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	int     numoutputs;
	char**  requestedoutputs = NULL;

	parameters->AddObject(iomodel->CopyConstantObject(ThermalStabilizationEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalIsenthalpyEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalIsdynamicbasalspcEnum));

	iomodel->FetchData(&requestedoutputs,&numoutputs,ThermalRequestedOutputsEnum);
	parameters->AddObject(new IntParam(ThermalNumRequestedOutputsEnum,numoutputs));
	if(numoutputs)parameters->AddObject(new StringArrayParam(ThermalRequestedOutputsEnum,requestedoutputs,numoutputs));
	iomodel->DeleteData(&requestedoutputs,numoutputs,ThermalRequestedOutputsEnum);
}/*}}}*/
void EnthalpyAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	bool dakota_analysis;
	bool isenthalpy;

	/*Now, is the model 3d? otherwise, do nothing: */
	if(iomodel->meshtype==Mesh2DhorizontalEnum)return;

	/*Is enthalpy requested?*/
	iomodel->Constant(&isenthalpy,ThermalIsenthalpyEnum);
	if(!isenthalpy) return;

	/*Fetch data needed: */
	iomodel->FetchData(3,TemperatureEnum,WaterfractionEnum,PressureEnum);

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	iomodel->Constant(&dakota_analysis,QmuIsdakotaEnum);

	iomodel->FetchDataToInput(elements,ThicknessEnum);
	iomodel->FetchDataToInput(elements,SurfaceEnum);
	iomodel->FetchDataToInput(elements,BedEnum);
	iomodel->FetchDataToInput(elements,FrictionCoefficientEnum);
	iomodel->FetchDataToInput(elements,FrictionPEnum);
	iomodel->FetchDataToInput(elements,FrictionQEnum);
	iomodel->FetchDataToInput(elements,MaskIceLevelsetEnum);
	iomodel->FetchDataToInput(elements,MaskGroundediceLevelsetEnum);
	iomodel->FetchDataToInput(elements,MeshElementonbedEnum);
	iomodel->FetchDataToInput(elements,MeshElementonsurfaceEnum);
	iomodel->FetchDataToInput(elements,FlowequationElementEquationEnum);
	iomodel->FetchDataToInput(elements,MaterialsRheologyBEnum);
	iomodel->FetchDataToInput(elements,MaterialsRheologyNEnum);
	iomodel->FetchDataToInput(elements,PressureEnum);
	iomodel->FetchDataToInput(elements,TemperatureEnum);
	iomodel->FetchDataToInput(elements,WaterfractionEnum);
	iomodel->FetchDataToInput(elements,EnthalpyEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsGeothermalfluxEnum);
	iomodel->FetchDataToInput(elements,WatercolumnEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsMeltingRateEnum);
	iomodel->FetchDataToInput(elements,VxEnum);
	iomodel->FetchDataToInput(elements,VyEnum);
	iomodel->FetchDataToInput(elements,VzEnum);
	InputUpdateFromConstantx(elements,0.,VxMeshEnum);
	InputUpdateFromConstantx(elements,0.,VyMeshEnum);
	InputUpdateFromConstantx(elements,0.,VzMeshEnum);
	if(dakota_analysis){
		elements->InputDuplicate(TemperatureEnum,QmuTemperatureEnum);
		elements->InputDuplicate(BasalforcingsMeltingRateEnum,QmuMeltingEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVxMeshEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVyMeshEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVzMeshEnum);
	}

	/*Free data: */
	iomodel->DeleteData(3,TemperatureEnum,WaterfractionEnum,PressureEnum);
}/*}}}*/
void EnthalpyAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	if(iomodel->meshtype==Mesh3DEnum) iomodel->FetchData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
	::CreateNodes(nodes,iomodel,EnthalpyAnalysisEnum,P1Enum);
	iomodel->DeleteData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
}/*}}}*/
void EnthalpyAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	/*Intermediary*/
	int        count;
	int        M,N;
	bool       spcpresent = false;
	IssmDouble heatcapacity;
	IssmDouble referencetemperature;

	/*Output*/
	IssmDouble *spcvector  = NULL;
	IssmDouble* times=NULL;
	IssmDouble* values=NULL;

	/*Fetch parameters: */
	iomodel->Constant(&heatcapacity,MaterialsHeatcapacityEnum);
	iomodel->Constant(&referencetemperature,ConstantsReferencetemperatureEnum);

	/*return if 2d mesh*/
	if(iomodel->meshtype==Mesh2DhorizontalEnum) return;

	/*Fetch data: */
	iomodel->FetchData(&spcvector,&M,&N,ThermalSpctemperatureEnum);

	//FIX ME: SHOULD USE IOMODELCREATECONSTRAINTS 
	/*Transient or static?:*/
	if(M==iomodel->numberofvertices){
		/*static: just create Constraints objects*/
		count=0;

		for(int i=0;i<iomodel->numberofvertices;i++){
			/*keep only this partition's nodes:*/
			if((iomodel->my_vertices[i])){

				if (!xIsNan<IssmDouble>(spcvector[i])){

					constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,1,heatcapacity*(spcvector[i]-referencetemperature),EnthalpyAnalysisEnum));
					count++;

				}
			}
		}
	}
	else if (M==(iomodel->numberofvertices+1)){
		/*transient: create transient SpcTransient objects. Same logic, except we need to retrieve 
		 * various times and values to initialize an SpcTransient object: */
		count=0;

		/*figure out times: */
		times=xNew<IssmDouble>(N);
		for(int j=0;j<N;j++){
			times[j]=spcvector[(M-1)*N+j];
		}

		/*Create constraints from x,y,z: */
		for(int i=0;i<iomodel->numberofvertices;i++){

			/*keep only this partition's nodes:*/
			if((iomodel->my_vertices[i])){

				/*figure out times and values: */
				values=xNew<IssmDouble>(N);
				spcpresent=false;
				for(int j=0;j<N;j++){
					values[j]=heatcapacity*(spcvector[i*N+j]-referencetemperature);
					if(!xIsNan<IssmDouble>(values[j]))spcpresent=true; //NaN means no spc by default
				}

				if(spcpresent){
					constraints->AddObject(new SpcTransient(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,1,N,times,values,EnthalpyAnalysisEnum));
					count++;
				}
				xDelete<IssmDouble>(values);
			}
		}
	}
	else{
		_error_("Size of field " << EnumToStringx(ThermalSpctemperatureEnum) << " not supported");
	}

	/*Free ressources:*/
	iomodel->DeleteData(spcvector,ThermalSpctemperatureEnum);
	xDelete<IssmDouble>(times);
	xDelete<IssmDouble>(values);
}/*}}}*/
void EnthalpyAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	/*No loads */
}/*}}}*/

/*Finite Element Analysis*/
ElementMatrix* EnthalpyAnalysis::CreateKMatrix(Element* element){/*{{{*/

	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixVolume(element);
	ElementMatrix* Ke2=CreateKMatrixShelf(element);
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;
}/*}}}*/
ElementMatrix* EnthalpyAnalysis::CreateKMatrixVolume(Element* element){/*{{{*/

	/*Intermediaries */
	int         stabilization;
	IssmDouble  Jdet,dt,u,v,w,um,vm,wm,vel;
	IssmDouble  h,hx,hy,hz,vx,vy,vz;
	IssmDouble  tau_parameter,diameter;
	IssmDouble  D_scalar;
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes    = element->GetNumberOfNodes();

	/*Initialize Element vector and other vectors*/
	ElementMatrix* Ke       = element->NewElementMatrix();
	IssmDouble*    basis    = xNew<IssmDouble>(numnodes);
	IssmDouble*    dbasis   = xNew<IssmDouble>(3*numnodes);
	IssmDouble*    B        = xNew<IssmDouble>(3*numnodes);
	IssmDouble*    Bprime   = xNew<IssmDouble>(3*numnodes);
	IssmDouble     D[3][3]  = {0.};
	IssmDouble     K[3][3];

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	element->FindParam(&stabilization,ThermalStabilizationEnum);
	IssmDouble  rho_water           = element->GetMaterialParameter(MaterialsRhoWaterEnum);
	IssmDouble  rho_ice             = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  gravity             = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble  heatcapacity        = element->GetMaterialParameter(MaterialsHeatcapacityEnum);
	IssmDouble  thermalconductivity = element->GetMaterialParameter(MaterialsThermalconductivityEnum);
	Input* vx_input  = element->GetInput(VxEnum);     _assert_(vx_input);
	Input* vy_input  = element->GetInput(VyEnum);     _assert_(vy_input);
	Input* vz_input  = element->GetInput(VzEnum);     _assert_(vz_input);
	Input* vxm_input = element->GetInput(VxMeshEnum); _assert_(vxm_input);
	Input* vym_input = element->GetInput(VyMeshEnum); _assert_(vym_input);
	Input* vzm_input = element->GetInput(VzMeshEnum); _assert_(vzm_input);
	if(stabilization==2) diameter=element->MinEdgeLength(xyz_list);

	/*Enthalpy diffusion parameter*/
	IssmDouble kappa=this->EnthalpyDiffusionParameterVolume(element,EnthalpyEnum); _assert_(kappa>0.);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		D_scalar=gauss->weight*Jdet;
		if(dt!=0.) D_scalar=D_scalar*dt;

		/*Conduction: */
		GetBConduct(B,element,xyz_list,gauss); 
		D[0][0]=D_scalar*kappa/rho_ice;
		D[1][1]=D_scalar*kappa/rho_ice;
		D[2][2]=D_scalar*kappa/rho_ice;
		TripleMultiply(B,3,numnodes,1,
					&D[0][0],3,3,0,
					B,3,numnodes,0,
					&Ke->values[0],1);

		/*Advection: */
		GetBAdvec(B,element,xyz_list,gauss); 
		GetBAdvecprime(Bprime,element,xyz_list,gauss); 
		vx_input->GetInputValue(&u,gauss); vxm_input->GetInputValue(&um,gauss); vx=u-um;
		vy_input->GetInputValue(&v,gauss); vym_input->GetInputValue(&vm,gauss); vy=v-vm;
		vz_input->GetInputValue(&w,gauss); vzm_input->GetInputValue(&wm,gauss); vz=w-wm;
		D[0][0]=D_scalar*vx;
		D[1][1]=D_scalar*vy;
		D[2][2]=D_scalar*vz;
		TripleMultiply(B,3,numnodes,1,
					&D[0][0],3,3,0,
					Bprime,3,numnodes,0,
					&Ke->values[0],1);

		/*Transient: */
		if(dt!=0.){
			D_scalar=gauss->weight*Jdet;
			element->NodalFunctions(basis,gauss);
			TripleMultiply(basis,numnodes,1,0,
						&D_scalar,1,1,0,
						basis,1,numnodes,0,
						&Ke->values[0],1);
			D_scalar=D_scalar*dt;
		}

		/*Artifficial diffusivity*/
		if(stabilization==1){
			element->ElementSizes(&hx,&hy,&hz);
			vel=sqrt(vx*vx + vy*vy + vz*vz)+1.e-14;
			h=sqrt( pow(hx*vx/vel,2) + pow(hy*vy/vel,2) + pow(hz*vz/vel,2));
			K[0][0]=h/(2.*vel)*vx*vx;  K[0][1]=h/(2.*vel)*vx*vy; K[0][2]=h/(2.*vel)*vx*vz;
			K[1][0]=h/(2.*vel)*vy*vx;  K[1][1]=h/(2.*vel)*vy*vy; K[1][2]=h/(2.*vel)*vy*vz;
			K[2][0]=h/(2.*vel)*vz*vx;  K[2][1]=h/(2.*vel)*vz*vy; K[2][2]=h/(2.*vel)*vz*vz;
			for(int i=0;i<3;i++) for(int j=0;j<3;j++) K[i][j] = D_scalar*K[i][j];

			GetBAdvecprime(Bprime,element,xyz_list,gauss); 
			TripleMultiply(Bprime,3,numnodes,1,
						&K[0][0],3,3,0,
						Bprime,3,numnodes,0,
						&Ke->values[0],1);
		}
		else if(stabilization==2){
			element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);
			tau_parameter=element->StabilizationParameter(u-um,v-vm,w-wm,diameter,kappa/rho_ice);
			for(int i=0;i<numnodes;i++){
				for(int j=0;j<numnodes;j++){
					Ke->values[i*numnodes+j]+=tau_parameter*D_scalar*
					  ((u-um)*dbasis[0*numnodes+i]+(v-vm)*dbasis[1*numnodes+i]+(w-wm)*dbasis[2*numnodes+i])*((u-um)*dbasis[0*numnodes+j]+(v-vm)*dbasis[1*numnodes+j]+(w-wm)*dbasis[2*numnodes+j]);
				}
			}
			if(dt!=0.){
				for(int i=0;i<numnodes;i++){
					for(int j=0;j<numnodes;j++){
						Ke->values[i*numnodes+j]+=tau_parameter*D_scalar*basis[j]*((u-um)*dbasis[0*numnodes+i]+(v-vm)*dbasis[1*numnodes+i]+(w-wm)*dbasis[2*numnodes+i]);
					}
				}
			}
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(dbasis);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(Bprime);
	delete gauss;
	return Ke;
}/*}}}*/
ElementMatrix* EnthalpyAnalysis::CreateKMatrixShelf(Element* element){/*{{{*/

	/*Initialize Element matrix and return if necessary*/
	if(!element->IsOnBed() || !element->IsFloating()) return NULL;

	IssmDouble  dt,Jdet,D;
	IssmDouble *xyz_list_base = NULL;

	/*Get basal element*/
	if(!element->IsOnBed() || !element->IsFloating()) return NULL;

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Initialize vectors*/
	ElementMatrix* Ke    = element->NewElementMatrix();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinatesBase(&xyz_list_base);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	IssmDouble  gravity             = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble  rho_water           = element->GetMaterialParameter(MaterialsRhoWaterEnum);
	IssmDouble  rho_ice             = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  heatcapacity        = element->GetMaterialParameter(MaterialsHeatcapacityEnum);
	IssmDouble  mixed_layer_capacity= element->GetMaterialParameter(MaterialsMixedLayerCapacityEnum);
	IssmDouble  thermal_exchange_vel= element->GetMaterialParameter(MaterialsThermalExchangeVelocityEnum);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGaussBase(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminantBase(&Jdet,xyz_list_base,gauss);
		element->NodalFunctions(basis,gauss);

		D=gauss->weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_vel/(heatcapacity*rho_ice);
		if(reCast<bool,IssmDouble>(dt)) D=dt*D;
		TripleMultiply(basis,numnodes,1,0,
					&D,1,1,0,
					basis,1,numnodes,0,
					&Ke->values[0],1);

	}

	/*Clean up and return*/
	delete gauss;
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(xyz_list_base);
	return Ke;
}/*}}}*/
ElementVector* EnthalpyAnalysis::CreatePVector(Element* element){/*{{{*/

	/*compute all load vectors for this element*/
	ElementVector* pe1=CreatePVectorVolume(element);
	ElementVector* pe2=CreatePVectorSheet(element);
	ElementVector* pe3=CreatePVectorShelf(element);
	ElementVector* pe =new ElementVector(pe1,pe2,pe3);

	/*clean-up and return*/
	delete pe1;
	delete pe2;
	delete pe3;
	return pe;
}/*}}}*/
ElementVector* EnthalpyAnalysis::CreatePVectorVolume(Element* element){/*{{{*/

	/*Intermediaries*/
	int         stabilization;
	IssmDouble  Jdet,phi,dt;
	IssmDouble  enthalpy;
	IssmDouble  kappa,tau_parameter,diameter;
	IssmDouble  u,v,w;
	IssmDouble  scalar_def,scalar_transient;
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes    = element->GetNumberOfNodes();
	int numvertices = element->GetNumberOfVertices();

	/*Initialize Element vector*/
	ElementVector* pe             = element->NewElementVector();
	IssmDouble*    basis          = xNew<IssmDouble>(numnodes);
	IssmDouble*    dbasis         = xNew<IssmDouble>(3*numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	IssmDouble  rho_ice             = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  thermalconductivity = element->GetMaterialParameter(MaterialsThermalconductivityEnum);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	element->FindParam(&stabilization,ThermalStabilizationEnum);
	Input* vx_input=element->GetInput(VxEnum); _assert_(vx_input);
	Input* vy_input=element->GetInput(VyEnum); _assert_(vy_input);
	Input* vz_input=element->GetInput(VzEnum); _assert_(vz_input);
	Input* enthalpy_input = NULL;
	if(reCast<bool,IssmDouble>(dt)){enthalpy_input = element->GetInput(EnthalpyEnum); _assert_(enthalpy_input);}
	if(stabilization==2){
		diameter=element->MinEdgeLength(xyz_list);
		kappa=this->EnthalpyDiffusionParameterVolume(element,EnthalpyPicardEnum); _assert_(kappa>0.);
	}

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		element->NodalFunctions(basis,gauss);
		element->ViscousHeating(&phi,xyz_list,gauss,vx_input,vy_input,vz_input);

		scalar_def=phi/rho_ice*Jdet*gauss->weight;
		if(dt!=0.) scalar_def=scalar_def*dt;

		/*TODO: add -beta*laplace T_m(p)*/
		for(int i=0;i<numnodes;i++) pe->values[i]+=scalar_def*basis[i];

		/* Build transient now */
		if(reCast<bool,IssmDouble>(dt)){
			enthalpy_input->GetInputValue(&enthalpy, gauss);
			scalar_transient=enthalpy*Jdet*gauss->weight;
			for(int i=0;i<numnodes;i++) pe->values[i]+=scalar_transient*basis[i];
		}

		if(stabilization==2){
			element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

			vx_input->GetInputValue(&u,gauss);
			vy_input->GetInputValue(&v,gauss);
			vz_input->GetInputValue(&w,gauss);
			tau_parameter=element->StabilizationParameter(u,v,w,diameter,kappa/rho_ice);

			for(int i=0;i<numnodes;i++) pe->values[i]+=tau_parameter*scalar_def*(u*dbasis[0*numnodes+i]+v*dbasis[1*numnodes+i]+w*dbasis[2*numnodes+i]);

			if(dt!=0.){
				for(int i=0;i<numnodes;i++) pe->values[i]+=tau_parameter*scalar_transient*(u*dbasis[0*numnodes+i]+v*dbasis[1*numnodes+i]+w*dbasis[2*numnodes+i]);
			}
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(dbasis);
	xDelete<IssmDouble>(xyz_list);
	delete gauss;
	return pe;

}/*}}}*/
ElementVector* EnthalpyAnalysis::CreatePVectorSheet(Element* element){/*{{{*/

	/* Geothermal flux on ice sheet base and basal friction */
	if(!element->IsOnBed() || element->IsFloating()) return NULL;

	IssmDouble  dt,Jdet,enthalpy,pressure,watercolumn,geothermalflux,vx,vy,vz;
	IssmDouble  enthalpyup,pressureup,alpha2,scalar,basalfriction,heatflux;
	IssmDouble *xyz_list_base = NULL;

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Initialize vectors*/
	ElementVector* pe    = element->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinatesBase(&xyz_list_base);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	Input* vx_input             = element->GetInput(VxEnum);                          _assert_(vx_input);
	Input* vy_input             = element->GetInput(VyEnum);                          _assert_(vy_input);
	Input* vz_input             = element->GetInput(VzEnum);                          _assert_(vz_input);
	Input* enthalpy_input       = element->GetInput(EnthalpyPicardEnum);              _assert_(enthalpy_input);
	Input* pressure_input       = element->GetInput(PressureEnum);                    _assert_(pressure_input);
	Input* geothermalflux_input = element->GetInput(BasalforcingsGeothermalfluxEnum); _assert_(geothermalflux_input);
	Input* watercolumn_input    = element->GetInput(WatercolumnEnum);                 _assert_(watercolumn_input);
	IssmDouble  rho_ice             = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  heatcapacity        = element->GetMaterialParameter(MaterialsHeatcapacityEnum);

	/*Build friction element, needed later: */
	Friction* friction=new Friction(element,3);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss   = element->NewGaussBase(2);
	Gauss* gaussup = element->NewGaussTop(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminantBase(&Jdet,xyz_list_base,gauss);
		element->NodalFunctions(basis,gauss);

		enthalpy_input->GetInputValue(&enthalpy,gauss);
		pressure_input->GetInputValue(&pressure,gauss);
		watercolumn_input->GetInputValue(&watercolumn,gauss);

		if((watercolumn<=0.) && (enthalpy < PureIceEnthalpy(element,pressure))){
			/* the above check is equivalent to 
			 NOT ((watercolumn>0.) AND (enthalpy<PIE)) AND (enthalpy<PIE)*/
			geothermalflux_input->GetInputValue(&geothermalflux,gauss);

			friction->GetAlpha2(&alpha2,gauss,vx_input,vy_input,vz_input);
			vx_input->GetInputValue(&vx,gauss);
			vy_input->GetInputValue(&vy,gauss);
			vz_input->GetInputValue(&vz,gauss);
			basalfriction = alpha2*(vx*vx + vy*vy + vz*vz);
			heatflux      = (basalfriction+geothermalflux)/(rho_ice);

			scalar = gauss->weight*Jdet*heatflux;
			if(dt!=0.) scalar=dt*scalar;

			for(int i=0;i<numnodes;i++) pe->values[i]+=scalar*basis[i];
		}
		else if(enthalpy >= PureIceEnthalpy(element,pressure)){
			/* check positive thickness of temperate basal ice layer */
			enthalpy_input->GetInputValue(&enthalpyup,gaussup);
			pressure_input->GetInputValue(&pressureup,gaussup);
			if(enthalpyup >= PureIceEnthalpy(element,pressureup)){
				// TODO: temperate ice has positive thickness: grad enthalpy*n=0.
			}
			else{
				// only base temperate, set Dirichlet BCs in Penta::UpdateBasalConstraintsEnthalpy()
			}
		}
		else{
			// base cold, but watercolumn positive. Set base to h_pmp.
		}
	}

	/*Clean up and return*/
	delete gauss;
	delete gaussup;
	delete friction;
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(xyz_list_base);
	return pe;

}/*}}}*/
ElementVector* EnthalpyAnalysis::CreatePVectorShelf(Element* element){/*{{{*/

	/*Get basal element*/
	if(!element->IsOnBed() || !element->IsFloating()) return NULL;

	IssmDouble  h_pmp,dt,Jdet,scalar_ocean,pressure;
	IssmDouble *xyz_list_base = NULL;

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Initialize vectors*/
	ElementVector* pe    = element->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinatesBase(&xyz_list_base);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	Input*      pressure_input=element->GetInput(PressureEnum); _assert_(pressure_input);
	IssmDouble  gravity             = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble  rho_water           = element->GetMaterialParameter(MaterialsRhoWaterEnum);
	IssmDouble  rho_ice             = element->GetMaterialParameter(MaterialsRhoIceEnum);
	IssmDouble  heatcapacity        = element->GetMaterialParameter(MaterialsHeatcapacityEnum);
	IssmDouble  mixed_layer_capacity= element->GetMaterialParameter(MaterialsMixedLayerCapacityEnum);
	IssmDouble  thermal_exchange_vel= element->GetMaterialParameter(MaterialsThermalExchangeVelocityEnum);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGaussBase(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminantBase(&Jdet,xyz_list_base,gauss);
		element->NodalFunctions(basis,gauss);

		pressure_input->GetInputValue(&pressure,gauss);
		h_pmp=element->PureIceEnthalpy(pressure);

		scalar_ocean=gauss->weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_vel*h_pmp/(heatcapacity*rho_ice);
		if(reCast<bool,IssmDouble>(dt)) scalar_ocean=dt*scalar_ocean;

		for(int i=0;i<numnodes;i++) pe->values[i]+=scalar_ocean*basis[i];
	}

	/*Clean up and return*/
	delete gauss;
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(xyz_list_base);
	return pe;
}/*}}}*/
void EnthalpyAnalysis::GetBConduct(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF1. 
	 * For node i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_conduct=[ dh/dx ]
	 *                  [ dh/dy ]
	 *                  [ dh/dz ]
	 * where h is the interpolation function for node i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF1*numnodes)
	 */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* dbasis=xNew<IssmDouble>(3*numnodes);
	element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[numnodes*0+i] = dbasis[0*numnodes+i];
		B[numnodes*1+i] = dbasis[1*numnodes+i];
		B[numnodes*2+i] = dbasis[2*numnodes+i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(dbasis);
}/*}}}*/
void EnthalpyAnalysis::GetBAdvec(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF1. 
	 * For node i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_advec =[ h ]
	 *                 [ h ]
	 *                 [ h ]
	 * where h is the interpolation function for node i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF1*NUMNODESP1)
	 */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions*/
	IssmDouble* basis=xNew<IssmDouble>(numnodes);
	element->NodalFunctions(basis,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[numnodes*0+i] = basis[i];
		B[numnodes*1+i] = basis[i];
		B[numnodes*2+i] = basis[i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(basis);
}/*}}}*/
void EnthalpyAnalysis::GetBAdvecprime(IssmDouble* B,Element* element,IssmDouble* xyz_list,Gauss* gauss){/*{{{*/
	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF1. 
	 * For node i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Biprime_advec=[ dh/dx ]
	 *                     [ dh/dy ]
	 *                     [ dh/dz ]
	 * where h is the interpolation function for node i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF1*numnodes)
	 */

	/*Fetch number of nodes for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Get nodal functions derivatives*/
	IssmDouble* dbasis=xNew<IssmDouble>(3*numnodes);
	element->NodalFunctionsDerivatives(dbasis,xyz_list,gauss);

	/*Build B: */
	for(int i=0;i<numnodes;i++){
		B[numnodes*0+i] = dbasis[0*numnodes+i];
		B[numnodes*1+i] = dbasis[1*numnodes+i];
		B[numnodes*2+i] = dbasis[2*numnodes+i];
	}

	/*Clean-up*/
	xDelete<IssmDouble>(dbasis);
}/*}}}*/
void EnthalpyAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	element->GetSolutionFromInputsOneDof(solution,EnthalpyEnum);
}/*}}}*/
void EnthalpyAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	bool        converged;
	int         i,rheology_law;
	IssmDouble  B_average,s_average,T_average=0.,P_average=0.;
	int        *doflist   = NULL;
	IssmDouble *xyz_list  = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes    = element->GetNumberOfNodes();

	/*Fetch dof list and allocate solution vector*/
	element->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values        = xNew<IssmDouble>(numnodes);
	IssmDouble* pressure      = xNew<IssmDouble>(numnodes);
	IssmDouble* surface       = xNew<IssmDouble>(numnodes);
	IssmDouble* B             = xNew<IssmDouble>(numnodes);
	IssmDouble* temperature   = xNew<IssmDouble>(numnodes);
	IssmDouble* waterfraction = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numnodes;i++){
		values[i]=solution[doflist[i]];

		/*Check solution*/
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
	}

	/*Get all inputs and parameters*/
	element->GetInputValue(&converged,ConvergedEnum);
	element->GetInputListOnNodes(&pressure[0],PressureEnum);
	if(converged){
		for(i=0;i<numnodes;i++){
			element->EnthalpyToThermal(&temperature[i],&waterfraction[i],values[i],pressure[i]);
			if(waterfraction[i]<0.) _error_("Negative water fraction found in solution vector");
			if(waterfraction[i]>1.) _error_("Water fraction >1 found in solution vector");
		}
		element->AddInput(EnthalpyEnum,values,P1Enum);
		element->AddInput(WaterfractionEnum,waterfraction,P1Enum);
		element->AddInput(TemperatureEnum,temperature,P1Enum);

		/*Update Rheology only if converged (we must make sure that the temperature is below melting point
		 * otherwise the rheology could be negative*/
		element->FindParam(&rheology_law,MaterialsRheologyLawEnum);
		element->GetInputListOnNodes(&surface[0],SurfaceEnum);
		switch(rheology_law){
			case NoneEnum:
				/*Do nothing: B is not temperature dependent*/
				break;
			case PatersonEnum:
				for(i=0;i<numnodes;i++) B[i]=Paterson(temperature[i]);
				element->AddMaterialInput(MaterialsRheologyBEnum,&B[0],P1Enum);
				break;
			case ArrheniusEnum:
				element->GetVerticesCoordinates(&xyz_list);
				for(i=0;i<numnodes;i++) B[i]=Arrhenius(temperature[i],surface[i]-xyz_list[i*3+2],element->GetMaterialParameter(MaterialsRheologyNEnum));
				element->AddMaterialInput(MaterialsRheologyBEnum,&B[0],P1Enum);
				break;
			case LliboutryDuvalEnum:
				for(i=0;i<numnodes;i++) B[i]=LliboutryDuval(values[i],pressure[i],element->GetMaterialParameter(MaterialsRheologyNEnum),element->GetMaterialParameter(MaterialsBetaEnum),element->GetMaterialParameter(ConstantsReferencetemperatureEnum),element->GetMaterialParameter(MaterialsHeatcapacityEnum),element->GetMaterialParameter(MaterialsLatentheatEnum)); 
				element->AddMaterialInput(MaterialsRheologyBEnum,&B[0],P1Enum); 
				break; 
			default: _error_("Rheology law " << EnumToStringx(rheology_law) << " not supported yet");
		}
	}
	else{
		element->AddInput(EnthalpyPicardEnum,values,P1Enum);
	}

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(pressure);
	xDelete<IssmDouble>(surface);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(temperature);
	xDelete<IssmDouble>(waterfraction);
	xDelete<IssmDouble>(xyz_list);
	xDelete<int>(doflist);
}/*}}}*/

/*Intermediaries*/
IssmDouble EnthalpyAnalysis::EnthalpyDiffusionParameter(Element* element,IssmDouble enthalpy,IssmDouble pressure){/*{{{*/

	IssmDouble heatcapacity             = element->GetMaterialParameter(MaterialsHeatcapacityEnum);
	IssmDouble temperateiceconductivity = element->GetMaterialParameter(MaterialsTemperateiceconductivityEnum);
	IssmDouble thermalconductivity      = element->GetMaterialParameter(MaterialsThermalconductivityEnum);

	if(enthalpy < PureIceEnthalpy(element,pressure)){
		return thermalconductivity/heatcapacity;
	}
	else{
		return temperateiceconductivity/heatcapacity;
	}
}/*}}}*/
IssmDouble EnthalpyAnalysis::EnthalpyDiffusionParameterVolume(Element* element,int enthalpy_enum){/*{{{*/

	int         iv;
	IssmDouble  lambda;                   /* fraction of cold ice    */
	IssmDouble  kappa   ,kappa_c,kappa_t; /* enthalpy conductivities */
	IssmDouble  Hc,Ht;


	/*Get pressures and enthalpies on vertices*/
	int         numvertices = element->GetNumberOfVertices();
	IssmDouble* pressures   = xNew<IssmDouble>(numvertices);
	IssmDouble* enthalpies  = xNew<IssmDouble>(numvertices);
	IssmDouble* PIE         = xNew<IssmDouble>(numvertices);
	IssmDouble* dHpmp       = xNew<IssmDouble>(numvertices);
	element->GetInputListOnVertices(pressures,PressureEnum);
	element->GetInputListOnVertices(enthalpies,enthalpy_enum);
	for(iv=0;iv<numvertices;iv++){
		PIE[iv]   = PureIceEnthalpy(element,pressures[iv]);
		dHpmp[iv] = enthalpies[iv]-PIE[iv];
	}

	bool allequalsign = true;
	if(dHpmp[0]<0.){
		for(iv=1; iv<numvertices;iv++) allequalsign=(allequalsign && (dHpmp[iv]<0.));
	}
	else{
		for(iv=1; iv<numvertices;iv++) allequalsign=(allequalsign && (dHpmp[iv]>=0.));
	}

	if(allequalsign){
		kappa = EnthalpyDiffusionParameter(element,enthalpies[0],pressures[0]);
	}
	else{
		/* return harmonic mean of thermal conductivities, weighted by fraction of cold/temperate ice,
			cf Patankar 1980, pp44 */
		kappa_c = EnthalpyDiffusionParameter(element,PureIceEnthalpy(element,0.)-1.,0.);
		kappa_t = EnthalpyDiffusionParameter(element,PureIceEnthalpy(element,0.)+1.,0.);
		Hc=0.; Ht=0.;
		for(iv=0; iv<numvertices;iv++){
			if(enthalpies[iv]<PIE[iv])
			 Hc+=(PIE[iv]-enthalpies[iv]);
			else
			 Ht+=(enthalpies[iv]-PIE[iv]);
		}
		_assert_((Hc+Ht)>0.);
		lambda = Hc/(Hc+Ht);
		kappa  = 1./(lambda/kappa_c + (1.-lambda)/kappa_t);
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(PIE);
	xDelete<IssmDouble>(dHpmp);
	xDelete<IssmDouble>(pressures);
	xDelete<IssmDouble>(enthalpies);
	return kappa;
}
/*}}}*/
IssmDouble EnthalpyAnalysis::PureIceEnthalpy(Element* element,IssmDouble pressure){/*{{{*/

	IssmDouble heatcapacity         = element->GetMaterialParameter(MaterialsHeatcapacityEnum);
	IssmDouble referencetemperature = element->GetMaterialParameter(ConstantsReferencetemperatureEnum);

	return heatcapacity*(TMeltingPoint(element,pressure)-referencetemperature);
}/*}}}*/
IssmDouble EnthalpyAnalysis::TMeltingPoint(Element* element,IssmDouble pressure){/*{{{*/

	IssmDouble meltingpoint = element->GetMaterialParameter(MaterialsMeltingpointEnum);
	IssmDouble beta         = element->GetMaterialParameter(MaterialsBetaEnum);

	return meltingpoint-beta*pressure;
}/*}}}*/
