/*! \file Penta.h 
 *  \brief: header file for penta object
 */

#ifndef _PENTA_H_
#define _PENTA_H_

/*Headers:*/
/*{{{*/
#include "./Element.h"
#include "./ElementHook.h"
#include "./PentaRef.h"
class Object;
class Parameters;
class Results;
class Inputs;
class Input;
class IoModel;
class Node;
class Material;
class Matpar;
class Tria;
class ElementMatrix;
class ElementVector;
class GaussPenta;
#include "../../shared/Exceptions/exceptions.h"
#include "../../shared/Enum/Enum.h"
/*}}}*/

class Penta: public Element,public ElementHook,public PentaRef{

	public:

		int          id;
		int          sid;

		Node       **nodes;                       // set of nodes
		Vertex     **vertices;                    // 6 vertices
		Material    *material;                    // 1 material ice
		Matpar      *matpar;                      // 1 material parameter
		Penta      **verticalneighbors;           // 2 neighbors: first one under, second one above

		Parameters  *parameters;                  //pointer to solution parameters
		Inputs      *inputs;

		/*Penta constructors and destructor: {{{*/
		Penta();
		Penta(int penta_id,int penta_sid,int i, IoModel* iomodel,int nummodels);
		~Penta();
		/*}}}*/
		/*Object virtual functions definitions: {{{*/
		Object *copy();
		void    DeepEcho();
		void    Echo();
		int     ObjectEnum();
		int     Id();
		/*}}}*/
		/*Update virtual functions definitions: {{{*/
		void  InputUpdateFromConstant(bool constant, int name);
		void  InputUpdateFromConstant(IssmDouble constant, int name);
		void  InputUpdateFromConstant(int constant, int name);
		void  InputUpdateFromVector(IssmDouble* vector, int name, int type);
		#ifdef _HAVE_DAKOTA_
		void  InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type);
		void  InputUpdateFromMatrixDakota(IssmDouble* matrix, int nows, int ncols, int name, int type);
		#endif
		void  InputUpdateFromIoModel(int index, IoModel* iomodel);
		/*}}}*/
		/*Element virtual functions definitions: {{{*/
		bool        AllActive(void){_error_("not implemented yet");};
		bool        AnyActive(void){_error_("not implemented yet");};
		void   BasalFrictionCreateInput(void);
		IssmDouble CharacteristicLength(void){_error_("not implemented yet");};
		void   ComputeBasalStress(Vector<IssmDouble>* sigma_b);
		void   ComputeStrainRate(Vector<IssmDouble>* eps);
		void   ComputeStressTensor();
		void   Configure(Elements* elements,Loads* loads,Nodes* nodes,Vertices* vertices,Materials* materials,Parameters* parameters);
		void   DeleteMaterials(void){_error_("not implemented yet");};
		void   ElementSizes(IssmDouble* hx,IssmDouble* hy,IssmDouble* hz);
		void   FindParam(int* pvalue,int paramenum);
		void   FindParam(IssmDouble* pvalue,int paramenum);
		void   FindParam(int** pvalues,int* psize,int paramenum);
		int    FiniteElement(void);
		void   SetCurrentConfiguration(Elements* elements,Loads* loads,Nodes* nodes,Materials* materials,Parameters* parameters);
		void   SetwiseNodeConnectivity(int* d_nz,int* o_nz,Node* node,bool* flags,int* flagsindices,int set1_enum,int set2_enum);
		void   CreateDVector(Vector<IssmDouble>* df);
		void   CreateJacobianMatrix(Matrix<IssmDouble>* Jff);
		void   Delta18oParameterization(void);
		void   EnthalpyToThermal(IssmDouble* ptemperature,IssmDouble* pwaterfraction,IssmDouble enthalpy,IssmDouble pressure);
		Element* GetBasalElement(void);
		void	 GetDofList(int** pdoflist,int approximation_enum,int setenum);
		void	 GetDofListVelocity(int** pdoflist,int setenum);
		void	 GetDofListPressure(int** pdoflist,int setenum);
		int    GetNodeIndex(Node* node);
		void   GetNodesSidList(int* sidlist);
		void   GetNodesLidList(int* lidlist);
		int    GetNumberOfNodes(void);
		int    GetNumberOfNodesPressure(void);
		int    GetNumberOfNodesVelocity(void);
		int    GetNumberOfVertices(void);
		IssmDouble GetMaterialParameter(int enum_in);
		void   GetSolutionFromInputsOneDof(Vector<IssmDouble>* solution,int enum_type);
		IssmDouble GetZcoord(Gauss* gauss);
		void   GetVectorFromInputs(Vector<IssmDouble>* vector,int name_enum);
		void   GetVerticesCoordinates(IssmDouble** pxyz_list);
		void   GetVerticesCoordinatesBase(IssmDouble** pxyz_list);
		void   GetVerticesCoordinatesTop(IssmDouble** pxyz_list);

		int    Sid();
		void   InputCreate(IssmDouble* vector,IoModel* iomodel,int M,int N,int vector_type,int vector_enum,int code);
		void   InputDepthAverageAtBase(int enum_type,int average_enum_type,int object_enum=MeshElementsEnum);
		void   InputDuplicate(int original_enum,int new_enum);
		void   InputScale(int enum_type,IssmDouble scale_factor);
		int    NumberofNodesVelocity(void);
		int    NumberofNodesPressure(void);
		int    VelocityInterpolation();
		IssmDouble PureIceEnthalpy(IssmDouble pressure);
		int    PressureInterpolation();
		bool   IsZeroLevelset(int levelset_enum);
		void   ZeroLevelsetCoordinates(IssmDouble** pxyz_zero,IssmDouble* xyz_list,int levelsetenum);

		void   ResultInterpolation(int* pinterpolation,int output_enum);
		void   ResultToVector(Vector<IssmDouble>* vector,int output_enum);
		void   PositiveDegreeDay(IssmDouble* pdds,IssmDouble* pds,IssmDouble signorm);
		void   ReduceMatrices(ElementMatrix* Ke,ElementVector* pe);
		void   ResetCoordinateSystem(void);
		void   SmbGradients();
	   Element* SpawnBasalElement(void);
		IssmDouble  TMeltingPoint(IssmDouble pressure);
		IssmDouble SurfaceArea(void);
		void   Update(int index, IoModel* iomodel,int analysis_counter,int analysis_type,int finitelement);
		int    NodalValue(IssmDouble* pvalue, int index, int natureofdataenum);
		IssmDouble TimeAdapt();
		void   ValueP1OnGauss(IssmDouble* pvalue,IssmDouble* values,Gauss* gauss);
		void   ValueP1DerivativesOnGauss(IssmDouble* dvalue,IssmDouble* values,IssmDouble* xyz_list,Gauss* gauss);
		int    VertexConnectivity(int vertexindex);
		void   VerticalSegmentIndices(int** pindices,int* pnumseg);
		void   ViscousHeatingCreateInput(void);
		void   ViscousHeating(IssmDouble* pphi,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input);

		 #ifdef _HAVE_RESPONSES_
		void   AverageOntoPartition(Vector<IssmDouble>* partition_contributions,Vector<IssmDouble>* partition_areas,IssmDouble* vertex_response,IssmDouble* qmu_part);
		IssmDouble IceVolume(void);
		IssmDouble IceVolumeAboveFloatation(void);
		IssmDouble TotalSmb(void);
		void   MinVel(IssmDouble* pminvel);
		void   MinVx(IssmDouble* pminvx);
		void   MinVy(IssmDouble* pminvy);
		void   MinVz(IssmDouble* pminvz);
		IssmDouble MassFlux(IssmDouble* segment);
		IssmDouble MassFlux(IssmDouble x1,IssmDouble y1, IssmDouble x2, IssmDouble y2,int segment_id);
		void   MaxAbsVx(IssmDouble* pmaxabsvx);
		void   MaxAbsVy(IssmDouble* pmaxabsvy);
		void   MaxAbsVz(IssmDouble* pmaxabsvz);
		void   MaxVel(IssmDouble* pmaxvel);
		void   ElementResponse(IssmDouble* presponse,int response_enum);
		void   MaxVx(IssmDouble* pmaxvx);
		void   MaxVy(IssmDouble* pmaxvy);
		void   MaxVz(IssmDouble* pmaxvz);
		#endif

		#ifdef _HAVE_GIA_
		void   GiaDeflection(Vector<IssmDouble>* wg,Vector<IssmDouble>* dwgdt,IssmDouble* x,IssmDouble* y);
		#endif

		#ifdef _HAVE_CONTROL_
		IssmDouble DragCoefficientAbsGradient(void);
		void   GradientIndexing(int* indexing,int control_index);
		void   Gradj(Vector<IssmDouble>* gradient,int control_type,int control_index);
		void   GradjDragSSA(Vector<IssmDouble>* gradient,int control_index);
		void   GradjDragHO(Vector<IssmDouble>* gradient,int control_index);
		void   GradjDragFS(Vector<IssmDouble>* gradient,int control_index);
		void   GradjBbarSSA(Vector<IssmDouble>* gradient,int control_index);
		void   GradjBbarHO(Vector<IssmDouble>* gradient,int control_index);
		void   GradjBbarFS(Vector<IssmDouble>* gradient,int control_index);
		void   GetVectorFromControlInputs(Vector<IssmDouble>* gradient,int control_enum,int control_index,const char* data);
		void   SetControlInputsFromVector(IssmDouble* vector,int control_enum,int control_index);
		void   ControlInputGetGradient(Vector<IssmDouble>* gradient,int enum_type,int control_index);
		void   ControlInputScaleGradient(int enum_type,IssmDouble scale);
		void   ControlInputSetGradient(IssmDouble* gradient,int enum_type,int control_index);
		void   ControlToVectors(Vector<IssmPDouble>* vector_control, Vector<IssmPDouble>* vector_gradient,int control_enum);
		IssmDouble RheologyBbarAbsGradient(void);
		IssmDouble ThicknessAbsMisfit(void);
		IssmDouble SurfaceAbsVelMisfit(void);
		IssmDouble SurfaceRelVelMisfit(void);
		IssmDouble SurfaceLogVelMisfit(void);
		IssmDouble SurfaceLogVxVyMisfit(void);
		IssmDouble SurfaceAverageVelMisfit(void);
		IssmDouble ThicknessAbsGradient(void);
		IssmDouble ThicknessAlongGradient(void){_error_("not supported");};
		IssmDouble ThicknessAcrossGradient(void){_error_("not supported");};
		IssmDouble BalancethicknessMisfit(void){_error_("not supported");};
		void   InputControlUpdate(IssmDouble scalar,bool save_parameter);
		IssmDouble Misfit(int modelenum,int observationenum,int weightsenum){_error_("not implemented yet");};
		#endif

		#ifdef _HAVE_GROUNDINGLINE_
		void   MigrateGroundingLine(IssmDouble* sheet_ungrounding);
		void   PotentialUngrounding(Vector<IssmDouble>* potential_sheet_ungrounding);
		int    UpdatePotentialUngrounding(IssmDouble* potential_sheet_ungrounding,Vector<IssmDouble>* vec_nodes_on_iceshelf,IssmDouble* nodes_on_iceshelf);
		#endif

		/*}}}*/
		/*Penta specific routines:{{{*/
		void           AddBasalInput(int input_enum, IssmDouble* values, int interpolation_enum);
		void           AddInput(int input_enum, IssmDouble* values, int interpolation_enum);
		void           AddMaterialInput(int input_enum, IssmDouble* values, int interpolation_enum);
		void	         NormalBase(IssmDouble* bed_normal, IssmDouble* xyz_list);
		void           NormalSection(IssmDouble* normal,IssmDouble* xyz_list);
		void	         NormalTop(IssmDouble* bed_normal, IssmDouble* xyz_list);
		ElementMatrix* CreateBasalMassMatrix(void);
		ElementMatrix* CreateKMatrix(void);
		ElementMatrix* CreateKMatrixMasstransport(void);
		ElementMatrix* CreateKMatrixFreeSurfaceTop(void);
		ElementMatrix* CreateKMatrixFreeSurfaceBase(void);
		ElementVector* CreatePVector(void);
		ElementVector* CreatePVectorMasstransport(void);
		ElementVector* CreatePVectorFreeSurfaceTop(void);
		ElementVector* CreatePVectorFreeSurfaceBase(void);
		ElementVector* CreatePVectorL2ProjectionBase(void);
		IssmDouble     EnthalpyDiffusionParameter(IssmDouble enthalpy,IssmDouble pressure);
		IssmDouble     EnthalpyDiffusionParameterVolume(int numvertices,IssmDouble* enthalpy,IssmDouble* pressure);
		void           GetAreaCoordinates(IssmDouble *area_coordinates,IssmDouble* xyz_zero,IssmDouble* xyz_list,int numpoints);

		void	         GetVertexPidList(int* doflist);
		void           GetVertexSidList(int* sidlist);
		void           GetConnectivityList(int* connectivity);
		void           GetGroundedPart(int* point1,IssmDouble* fraction1, IssmDouble* fraction2,bool* mainlyfloating);
		IssmDouble     GetGroundedPortion(IssmDouble* xyz_list);
		int            GetElementType(void);
		Input*         GetInput(int inputenum);
		Input*         GetMaterialInput(int inputenum);
		void           GetInputListOnVertices(IssmDouble* pvalue,int enumtype);
		void           GetInputListOnVertices(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue);
		void           GetInputListOnNodes(IssmDouble* pvalue,int enumtype);
		void           GetInputListOnNodes(IssmDouble* pvalue,int enumtype,IssmDouble defaultvalue);
		void           GetInputValue(IssmDouble* pvalue,Node* node,int enumtype);
		void           GetInputValue(bool* pvalue,int enum_type);
		void           GetInputValue(int* pvalue,int enum_type);
		void           GetInputValue(IssmDouble* pvalue,int enum_type);
		void           GetMaterialInputValue(IssmDouble* pvalue,Node* node,int enumtype);
		void	         GetPhi(IssmDouble* phi, IssmDouble*  epsilon, IssmDouble viscosity);
		void           GetStrainRate3dHO(IssmDouble* epsilon,IssmDouble* xyz_list,Gauss* gauss, Input* vx_input, Input* vy_input);
		void           GetStrainRate3d(IssmDouble* epsilon,IssmDouble* xyz_list, Gauss* gauss, Input* vx_input, Input* vy_input, Input* vz_input);
		Penta*         GetUpperElement(void);
		Penta*         GetLowerElement(void);
		void           InputChangeName(int input_enum, int enum_type_old);
		void	         InputExtrude(int enum_type,int object_type);
		void           InputUpdateFromSolutionOneDof(IssmDouble* solutiong,int enum_type);
		void           InputUpdateFromSolutionOneDofCollapsed(IssmDouble* solutiong,int enum_type);
		bool	         IsInput(int name);
		bool	         IsOnSurface(void);
		bool	         IsOnBed(void);
		bool           IsFloating(void); 
		bool           IsNodeOnShelfFromFlags(IssmDouble* flags);
		void           JacobianDeterminant(IssmDouble*  Jdet, IssmDouble* xyz_list,Gauss* gauss);
		void           JacobianDeterminantLine(IssmDouble* Jdet, IssmDouble* xyz_list,Gauss* gauss);
		void           JacobianDeterminantSurface(IssmDouble*  pJdet, IssmDouble* xyz_list,Gauss* gauss);
		void           JacobianDeterminantBase(IssmDouble* pJdet,IssmDouble* xyz_list_base,Gauss* gauss);
		void           JacobianDeterminantTop(IssmDouble* pJdet,IssmDouble* xyz_list_base,Gauss* gauss);
		bool           NoIceInElement(void); 
		Gauss*         NewGauss(void);
		Gauss*         NewGauss(int order);
      Gauss*         NewGauss(IssmDouble* xyz_list, IssmDouble* xyz_list_front,int order){_error_("not implemented yet");};
      Gauss*         NewGauss(IssmDouble* xyz_list, IssmDouble* xyz_list_front,int order_horiz,int order_vert);
		Gauss*         NewGaussBase(int order);
		Gauss*         NewGaussLine(int vertex1,int vertex2,int order);
		Gauss*         NewGaussTop(int order);
		ElementVector* NewElementVector(int approximation_enum);
		ElementMatrix* NewElementMatrix(int approximation_enum);
		void           NodalFunctions(IssmDouble* basis,Gauss* gauss);
		void           NodalFunctionsDerivatives(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss);
		void           NodalFunctionsDerivativesVelocity(IssmDouble* dbasis,IssmDouble* xyz_list,Gauss* gauss);
		void           NodalFunctionsVelocity(IssmDouble* basis,Gauss* gauss);
		void           NodalFunctionsPressure(IssmDouble* basis,Gauss* gauss);
		IssmDouble     MinEdgeLength(IssmDouble* xyz_list);
		void	         SetClone(int* minranks);
		Tria*	         SpawnTria(int location);
		IssmDouble     StabilizationParameter(IssmDouble u, IssmDouble v, IssmDouble w, IssmDouble diameter, IssmDouble kappa);
		void           TransformLoadVectorCoord(ElementVector* pe,int transformenum);
		void           TransformLoadVectorCoord(ElementVector* pe,int* transformenum_list);
		void           TransformLoadVectorCoord(ElementVector* pe,int numnodes,int transformenum){_error_("not implemented yet");};      /*Tiling only*/
		void           TransformLoadVectorCoord(ElementVector* pe,int numnodes,int* transformenum_list){_error_("not implemented yet");};/*Tiling only*/
		void           TransformSolutionCoord(IssmDouble* values,int transformenum);
		void           TransformSolutionCoord(IssmDouble* values,int* transformenum_list);
		void           TransformSolutionCoord(IssmDouble* values,int numnodes,int transformenum);      /*Tiling only*/
		void           TransformSolutionCoord(IssmDouble* values,int numnodes,int* transformenum_list);/*Tiling only*/
		void           TransformStiffnessMatrixCoord(ElementMatrix* Ke,int transformenum);
		void           TransformStiffnessMatrixCoord(ElementMatrix* Ke,int* transformenum_list);
		void           TransformStiffnessMatrixCoord(ElementMatrix* Ke,int numnodes,int transformenum){_error_("not implemented yet");};
		void           TransformStiffnessMatrixCoord(ElementMatrix* Ke,int numnodes,int* transformenum_list){_error_("not implemented yet");};
		void           ViscosityFS(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input);
		void           ViscosityHO(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input);
		void           ViscositySSA(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){_error_("not implemented");};

		#ifdef _HAVE_STRESSBALANCE_
		ElementMatrix* CreateKMatrixCouplingSSAHO(void);
		ElementMatrix* CreateKMatrixCouplingSSAHOViscous(void);
		ElementMatrix* CreateKMatrixCouplingSSAHOFriction(void);
		ElementMatrix* CreateKMatrixCouplingSSAFS(void);
		ElementMatrix* CreateKMatrixCouplingSSAFSViscous(void);
		ElementMatrix* CreateKMatrixCouplingSSAFSFriction(void);
		ElementMatrix* CreateKMatrixCouplingHOFS(void);
		ElementMatrix* CreateKMatrixStressbalanceHoriz(void);
		ElementMatrix* CreateKMatrixAdjointHoriz(void);
		ElementVector* CreateDVectorStressbalanceHoriz(void);
		ElementVector* CreateDVectorStressbalanceFS(void);
		ElementMatrix* CreateKMatrixStressbalanceSIA(void);
		ElementMatrix* CreateKMatrixStressbalanceSSA2d(void);
		ElementMatrix* CreateKMatrixStressbalanceSSA3d(void);
		ElementMatrix* CreateKMatrixStressbalanceSSA3dViscous(void);
		ElementMatrix* CreateKMatrixStressbalanceSSA3dFriction(void);
		ElementMatrix* CreateKMatrixStressbalanceSSAHO(void);
		ElementMatrix* CreateKMatrixStressbalanceSSAFS(void);
		ElementMatrix* CreateKMatrixStressbalanceL1L2(void);
		ElementMatrix* CreateKMatrixStressbalanceL1L2Viscous(void);
		ElementMatrix* CreateKMatrixStressbalanceL1L2Friction(void);
		ElementMatrix* CreateKMatrixStressbalanceHO(void);
		ElementMatrix* CreateKMatrixStressbalanceHOViscous(void);
		ElementMatrix* CreateKMatrixStressbalanceHOFriction(void);
		ElementMatrix* CreateKMatrixStressbalanceHOFS(void);
		ElementMatrix* CreateKMatrixStressbalanceFS(void);
		ElementMatrix* CreateKMatrixStressbalanceFSViscous(void);
		void           KMatrixGLSstabilization(ElementMatrix* Ke);
		ElementMatrix* CreateKMatrixStressbalanceFSFriction(void);
		ElementMatrix* CreateKMatrixStressbalanceVert(void);
		ElementMatrix* CreateKMatrixStressbalanceVertVolume(void);
		ElementMatrix* CreateKMatrixStressbalanceVertSurface(void);
		ElementMatrix* CreateJacobianStressbalanceHoriz(void);
		ElementMatrix* CreateJacobianStressbalanceSSA2d(void);
		ElementMatrix* CreateJacobianStressbalanceHO(void);
		ElementMatrix* CreateJacobianStressbalanceFS(void);
		ElementVector* CreatePVectorCouplingSSAFS(void);
		ElementVector* CreatePVectorCouplingSSAFSViscous(void);
		ElementVector* CreatePVectorCouplingSSAFSFriction(void);
		ElementVector* CreatePVectorCouplingHOFS(void);
		ElementVector* CreatePVectorCouplingHOFSViscous(void);
		ElementVector* CreatePVectorCouplingHOFSFriction(void);
		ElementVector* CreatePVectorStressbalanceHoriz(void);
		ElementVector* CreatePVectorStressbalanceSIA(void);
		ElementVector* CreatePVectorStressbalanceSSA(void);
		ElementVector* CreatePVectorStressbalanceSSAHO(void);
		ElementVector* CreatePVectorStressbalanceSSAFS(void);
		ElementVector* CreatePVectorStressbalanceL1L2(void);
		ElementVector* CreatePVectorStressbalanceHO(void);
		ElementVector* CreatePVectorStressbalanceHODrivingStress(void);
		ElementVector* CreatePVectorStressbalanceHOFront(void);
		ElementVector* CreatePVectorStressbalanceHOFS(void);
		ElementVector* CreatePVectorStressbalanceFS(void);
		ElementVector* CreatePVectorStressbalanceFSFront(void);
		ElementVector* CreatePVectorStressbalanceFSViscous(void);
		void           PVectorGLSstabilization(ElementVector* pe);
		ElementVector* CreatePVectorStressbalanceFSShelf(void);
		ElementVector* CreatePVectorStressbalanceVert(void);
		ElementVector* CreatePVectorStressbalanceVertVolume(void);
		ElementVector* CreatePVectorStressbalanceVertBase(void);
		void           GetL1L2Viscosity(IssmDouble*, IssmDouble*, GaussPenta*, Input*, Input*, Input*);
		#endif

		#ifdef _HAVE_CONTROL_
		ElementVector* CreatePVectorAdjointHoriz(void);
		ElementMatrix* CreateKMatrixAdjointSSA2d(void);
		ElementMatrix* CreateKMatrixAdjointHO(void);
		ElementMatrix* CreateKMatrixAdjointFS(void);
		ElementVector* CreatePVectorAdjointSSA(void);
		ElementVector* CreatePVectorAdjointHO(void);
		ElementVector* CreatePVectorAdjointFS(void);
		#endif

		#ifdef _HAVE_HYDROLOGY_
		ElementMatrix* CreateKMatrixHydrologyDCInefficient(void);
		ElementMatrix* CreateKMatrixHydrologyDCEfficient(void);
		ElementMatrix* CreateEPLDomainMassMatrix(void);
		ElementVector* CreatePVectorHydrologyDCInefficient(void);
		ElementVector* CreatePVectorHydrologyDCEfficient(void);
		ElementVector* CreatePVectorL2ProjectionEPL(void);
		void           GetHydrologyDCInefficientHmax(IssmDouble* ph_max, Node* innode);
		void           GetHydrologyDCInefficientHmax(IssmDouble* ph_max,int index){_error_("not implemented yet");};
		void           GetHydrologyTransfer(Vector<IssmDouble>* transfer);
		void           HydrologyEPLGetActive(Vector<IssmDouble>* active_vec);
		void           HydrologyEPLGetMask(Vector<IssmDouble>* vec_mask);
		void           ComputeEPLThickness(void);
		#endif

		void           UpdateConstraintsExtrudeFromBase(void){_error_("not implemented yet");};
		void           UpdateConstraintsExtrudeFromTop(void){_error_("not implemented yet");};
		#ifdef _HAVE_THERMAL_
		ElementMatrix* CreateKMatrixEnthalpy(void);
		ElementMatrix* CreateKMatrixEnthalpyVolume(void);
		ElementMatrix* CreateKMatrixEnthalpyShelf(void);
		ElementMatrix* CreateKMatrixThermal(void);
		ElementMatrix* CreateKMatrixMelting(void);
		ElementMatrix* CreateKMatrixThermalVolume(void);
		ElementMatrix* CreateKMatrixThermalShelf(void);
		ElementVector* CreatePVectorEnthalpy(void);
		ElementVector* CreatePVectorEnthalpyVolume(void);
		ElementVector* CreatePVectorEnthalpyShelf(void);
		ElementVector* CreatePVectorEnthalpySheet(void);
		ElementVector* CreatePVectorMelting(void);
		ElementVector* CreatePVectorThermal(void);
		ElementVector* CreatePVectorThermalVolume(void);
		ElementVector* CreatePVectorThermalShelf(void);
		ElementVector* CreatePVectorThermalSheet(void);
		void           UpdateBasalConstraintsEnthalpy(void);
		void           ComputeBasalMeltingrate(void);
		void           DrainWaterfraction(IssmDouble* drainrate_element);
		#endif

		#ifdef _HAVE_BALANCED_
		ElementMatrix* CreateKMatrixBalancethickness(void);
		ElementVector* CreatePVectorBalancethickness(void);
		#endif
		/*}}}*/
};
#endif  /* _PENTA_H */
