#include "./HydrologyDCInefficientAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  HydrologyDCInefficientAnalysis::DofsPerNode(int** doflist,int meshtype,int approximation){/*{{{*/
	return 1;
}/*}}}*/
void HydrologyDCInefficientAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	int         hydrology_model;
	int         sedimentlimit_flag;
	int         transfer_flag;
	int         penalty_lock;
	bool        isefficientlayer;
	IssmDouble  sedimentlimit;
	IssmDouble  penalty_factor;
	IssmDouble  leakagefactor;
	IssmDouble  rel_tol;

	/*retrieve some parameters: */
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	iomodel->FetchData(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	iomodel->FetchData(&sedimentlimit_flag,HydrologydcSedimentlimitFlagEnum);
	iomodel->FetchData(&transfer_flag,HydrologydcTransferFlagEnum);
	iomodel->FetchData(&penalty_factor,HydrologydcPenaltyFactorEnum);
	iomodel->FetchData(&rel_tol,HydrologydcRelTolEnum);
	iomodel->FetchData(&penalty_lock,HydrologydcPenaltyLockEnum);

	if(sedimentlimit_flag==1){
		iomodel->FetchData(&sedimentlimit,HydrologydcSedimentlimitEnum);
		parameters->AddObject(new DoubleParam(HydrologydcSedimentlimitEnum,sedimentlimit));
	}

	if(transfer_flag==1){
		iomodel->FetchData(&leakagefactor,HydrologydcLeakageFactorEnum);
		parameters->AddObject(new DoubleParam(HydrologydcLeakageFactorEnum,leakagefactor));
	}

	parameters->AddObject(new DoubleParam(HydrologydcPenaltyFactorEnum,penalty_factor));
	parameters->AddObject(new IntParam(HydrologyModelEnum,hydrology_model));
	parameters->AddObject(new BoolParam(HydrologydcIsefficientlayerEnum,isefficientlayer));
	parameters->AddObject(new IntParam(HydrologydcSedimentlimitFlagEnum,sedimentlimit_flag));
	parameters->AddObject(new IntParam(HydrologydcTransferFlagEnum,transfer_flag));
	parameters->AddObject(new DoubleParam(HydrologydcRelTolEnum,rel_tol));
	parameters->AddObject(new IntParam(HydrologydcPenaltyLockEnum,penalty_lock));

}/*}}}*/
void HydrologyDCInefficientAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	bool   isefficientlayer;
	int    hydrology_model;

	/*Fetch data needed: */
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	/*Fetch data needed: */
	iomodel->Constant(&isefficientlayer,HydrologydcIsefficientlayerEnum);

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	iomodel->FetchDataToInput(elements,ThicknessEnum);
	iomodel->FetchDataToInput(elements,SurfaceEnum);
	iomodel->FetchDataToInput(elements,BedEnum);
	iomodel->FetchDataToInput(elements,MaskIceLevelsetEnum);
	iomodel->FetchDataToInput(elements,MeshElementonbedEnum);
	iomodel->FetchDataToInput(elements,MeshElementonsurfaceEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsMeltingRateEnum);
	iomodel->FetchDataToInput(elements,SedimentHeadEnum);
	if(isefficientlayer)iomodel->FetchDataToInput(elements,HydrologydcMaskEplactiveNodeEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	/*Fetch parameters: */
	int  hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);

	/*Now, do we really want DC?*/
	if(hydrology_model!=HydrologydcEnum) return;

	if(iomodel->meshtype==Mesh3DEnum) iomodel->FetchData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
	::CreateNodes(nodes,iomodel,HydrologyDCInefficientAnalysisEnum,P1Enum);
	iomodel->DeleteData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	/*retrieve some parameters: */
	int hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);
	if(hydrology_model!=HydrologydcEnum) return;

	IoModelToConstraintsx(constraints,iomodel,HydrologydcSpcsedimentHeadEnum,HydrologyDCInefficientAnalysisEnum,P1Enum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	/*Fetch parameters: */
	int hydrology_model;
	iomodel->Constant(&hydrology_model,HydrologyModelEnum);
	if(hydrology_model!=HydrologydcEnum) return;

	iomodel->FetchData(1,MeshVertexonbedEnum);

	//create penalties for nodes: no node can have a temperature over the melting point
	CreateSingleNodeToElementConnectivity(iomodel);
	for(int i=0;i<iomodel->numberofvertices;i++){
		if (iomodel->meshtype==Mesh3DEnum){
			/*keep only this partition's nodes:*/
			if(iomodel->my_vertices[i]){
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,HydrologyDCInefficientAnalysisEnum));
			}
		}
		else if(reCast<int>(iomodel->Data(MeshVertexonbedEnum)[i])){
			if(iomodel->my_vertices[i]){
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,HydrologyDCInefficientAnalysisEnum));
			}	
		}
	}
	iomodel->DeleteData(1,MeshVertexonbedEnum);
}/*}}}*/

/*Finite Element Analysis*/
ElementMatrix* HydrologyDCInefficientAnalysis::CreateKMatrix(Element* element){/*{{{*/
	_error_("not implemented yet");
}/*}}}*/
ElementVector* HydrologyDCInefficientAnalysis::CreatePVector(Element* element){/*{{{*/

	/*Intermediaries*/
	int      meshtype;
	Element* basalelement;

	/*Get basal element*/
	element->FindParam(&meshtype,MeshTypeEnum);
	switch(meshtype){
		case Mesh2DhorizontalEnum:
			basalelement = element;
			break;
		case Mesh3DEnum:
			if(!element->IsOnBed()) return NULL;
			basalelement = element->SpawnBasalElement();
			break;
		default: _error_("mesh "<<EnumToStringx(meshtype)<<" not supported yet");
	}

	/*Intermediaries */
	IssmDouble dt,scalar,water_head;
	IssmDouble water_load,transfer;
	IssmDouble Jdet;
	IssmDouble *xyz_list  = NULL;
	Input*      old_wh_input      = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = basalelement->GetNumberOfNodes();

	/*Initialize Element vector*/
	ElementVector* pe    = basalelement->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	basalelement->GetVerticesCoordinates(&xyz_list);
	IssmDouble sediment_storing = SedimentStoring(basalelement);
	basalelement->FindParam(&dt,TimesteppingTimeStepEnum);
	Input* water_input       = element->GetInput(BasalforcingsMeltingRateEnum); _assert_(water_input);
	Input* transfer_input    = element->GetInput(WaterTransferEnum);            _assert_(transfer_input);
	if(dt!= 0.){old_wh_input = element->GetInput(SedimentHeadOldEnum);          _assert_(old_wh_input);}

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=basalelement->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		basalelement->JacobianDeterminant(&Jdet,xyz_list,gauss);
		basalelement->NodalFunctions(basis,gauss);

		/*Loading term*/
		water_input->GetInputValue(&water_load,gauss);
		transfer_input->GetInputValue(&transfer,gauss);
		scalar = Jdet*gauss->weight*(water_load+transfer);
		if(dt!=0.) scalar = scalar*dt;
		for(int i=0;i<numnodes;i++) pe->values[i]+=scalar*basis[i];

		/*Transient term*/
		if(dt!=0.){
			old_wh_input->GetInputValue(&water_head,gauss);
			scalar = Jdet*gauss->weight*water_head*sediment_storing;
			for(int i=0;i<numnodes;i++) pe->values[i]+=scalar*basis[i];
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	delete gauss;
	if(meshtype!=Mesh2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
	return pe;
}/*}}}*/
void HydrologyDCInefficientAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	element->GetSolutionFromInputsOneDof(solution,SedimentHeadEnum);
}/*}}}*/
void HydrologyDCInefficientAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	int        meshtype;
	bool       converged;
	int*       doflist=NULL;
	Element*   basalelement=NULL;

	element->FindParam(&meshtype,MeshTypeEnum);
	if(meshtype!=Mesh2DhorizontalEnum){
		if(!element->IsOnBed()) return;
		basalelement=element->SpawnBasalElement();
	}
	else{
		basalelement = element;
	}

	/*Fetch number of nodes for this finite element*/
	int numnodes = basalelement->GetNumberOfNodes();

	/*Fetch dof list and allocate solution vector*/
	basalelement->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values   = xNew<IssmDouble>(numnodes);
	IssmDouble* residual = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numnodes;i++){
		values[i] =solution[doflist[i]];
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
	}

	/*If converged keep the residual in mind*/
	element->GetInputValue(&converged,ConvergedEnum);

	/*Get inputs*/
	if(converged){
		IssmDouble penalty_factor,kmax,kappa,h_max;
		element->FindParam(&kmax,HydrologySedimentKmaxEnum);
		element->FindParam(&penalty_factor,HydrologydcPenaltyFactorEnum);

		kappa=kmax*pow(10.,penalty_factor);

		for(int i=0;i<numnodes;i++){
			basalelement->GetHydrologyDCInefficientHmax(&h_max,i);
			if(values[i]>h_max) residual[i] = kappa*(values[i]-h_max);
			else                residual[i] = 0.;
		}
	}

	/*Add input to the element: */
	element->AddBasalInput(SedimentHeadEnum,values,P1Enum);
	element->AddBasalInput(SedimentHeadResidualEnum,residual,P1Enum);

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(residual);
	xDelete<int>(doflist);
	if(meshtype!=Mesh2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
}/*}}}*/

/*Intermediaries*/
IssmDouble HydrologyDCInefficientAnalysis::SedimentStoring(Element* element){/*{{{*/
	IssmDouble rho_freshwater           = element->GetMaterialParameter(MaterialsRhoFreshwaterEnum);
	IssmDouble g                        = element->GetMaterialParameter(ConstantsGEnum);
	IssmDouble sediment_porosity        = element->GetMaterialParameter(HydrologydcSedimentPorosityEnum);
	IssmDouble sediment_thickness       = element->GetMaterialParameter(HydrologydcSedimentThicknessEnum);
	IssmDouble sediment_compressibility = element->GetMaterialParameter(HydrologydcSedimentCompressibilityEnum);
	IssmDouble water_compressibility    = element->GetMaterialParameter(HydrologydcWaterCompressibilityEnum);
	return rho_freshwater*g*sediment_porosity*sediment_thickness*(water_compressibility+(sediment_compressibility/sediment_porosity));		 
}/*}}}*/
