#include "./DamageEvolutionAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  DamageEvolutionAnalysis::DofsPerNode(int** doflist,int meshtype,int approximation){/*{{{*/
	return 1;
}/*}}}*/
void DamageEvolutionAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	/*retrieve some parameters: */
	parameters->AddObject(iomodel->CopyConstantObject(DamageLawEnum));
	parameters->AddObject(iomodel->CopyConstantObject(DamageStabilizationEnum));

	parameters->AddObject(iomodel->CopyConstantObject(DamagePenaltyThresholdEnum));
	parameters->AddObject(iomodel->CopyConstantObject(DamagePenaltyLockEnum));
	parameters->AddObject(iomodel->CopyConstantObject(DamagePenaltyFactorEnum));
	parameters->AddObject(iomodel->CopyConstantObject(DamageMaxiterEnum));
	parameters->AddObject(iomodel->CopyConstantObject(DamageMaxDamageEnum));

	/*Retrieve law dependent parameters: */
	char* law  = NULL;
	iomodel->Constant(&law,DamageLawEnum);
	if (strcmp(law,"pralong")==0){
		parameters->AddObject(iomodel->CopyConstantObject(DamageC1Enum));
		parameters->AddObject(iomodel->CopyConstantObject(DamageC2Enum));
		parameters->AddObject(iomodel->CopyConstantObject(DamageC3Enum));
		parameters->AddObject(iomodel->CopyConstantObject(DamageC4Enum));
		parameters->AddObject(iomodel->CopyConstantObject(DamageStressThresholdEnum));
		parameters->AddObject(iomodel->CopyConstantObject(DamageHealingEnum));
	}
	xDelete<char>(law);

}/*}}}*/
void DamageEvolutionAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	/*What input do I need to run my damage evolution model?*/
	iomodel->FetchDataToInput(elements,VxEnum);
	iomodel->FetchDataToInput(elements,VyEnum);
	iomodel->FetchDataToInput(elements,VzEnum);
	iomodel->FetchDataToInput(elements,DamageDEnum);
	iomodel->FetchDataToInput(elements,MaskIceLevelsetEnum);
	iomodel->FetchDataToInput(elements,PressureEnum);

	bool dakota_analysis;
	iomodel->Constant(&dakota_analysis,QmuIsdakotaEnum);
	if(dakota_analysis){
		elements->InputDuplicate(DamageDEnum, QmuDamageDEnum);
	}
}/*}}}*/
void DamageEvolutionAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	iomodel->FetchData(1,MeshVertexonbedEnum);
	::CreateNodes(nodes,iomodel,DamageEvolutionAnalysisEnum,P1Enum);
	iomodel->DeleteData(1,MeshVertexonbedEnum);
}/*}}}*/
void DamageEvolutionAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	int stabilization;
	iomodel->Constant(&stabilization,DamageStabilizationEnum);

	IoModelToConstraintsx(constraints,iomodel,DamageSpcdamageEnum,DamageEvolutionAnalysisEnum,P1Enum);

}/*}}}*/
void DamageEvolutionAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	/*create penalties for nodes: no node can have a damage > 1*/
	iomodel->FetchData(1,DamageSpcdamageEnum);
	CreateSingleNodeToElementConnectivity(iomodel);

	for(int i=0;i<iomodel->numberofvertices;i++){

		/*keep only this partition's nodes:*/
		if(iomodel->my_vertices[i]){
			if (xIsNan<IssmDouble>(iomodel->Data(DamageSpcdamageEnum)[i])){ //No penalty applied on spc nodes!
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,DamageEvolutionAnalysisEnum));
			}
		}
	}
	iomodel->DeleteData(1,DamageSpcdamageEnum);

}/*}}}*/

/*Finite Element Analysis*/
ElementMatrix* DamageEvolutionAnalysis::CreateKMatrix(Element* element){/*{{{*/
	_error_("not implemented yet");
}/*}}}*/
ElementVector* DamageEvolutionAnalysis::CreatePVector(Element* element){/*{{{*/

	/*Intermediaries*/
	int      meshtype;
	Element* basalelement;

	/*Get basal element*/
	element->FindParam(&meshtype,MeshTypeEnum);
	switch(meshtype){
		case Mesh2DhorizontalEnum:
			basalelement = element;
			break;
		case Mesh3DEnum:
			if(!element->IsOnBed()) return NULL;
			basalelement = element->SpawnBasalElement();
			break;
		default: _error_("mesh "<<EnumToStringx(meshtype)<<" not supported yet");
	}

	/*Intermediaries */
	IssmDouble  Jdet,dt;
	IssmDouble  f,damage;
	IssmDouble* xyz_list = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Initialize Element vector and other vectors*/
	ElementVector* pe    = element->NewElementVector();
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	element->GetVerticesCoordinates(&xyz_list);
	element->FindParam(&dt,TimesteppingTimeStepEnum);
	this->CreateDamageFInput(element);
	Input* damaged_input = element->GetMaterialInput(DamageDbarEnum); _assert_(damaged_input);
	Input* damagef_input = element->GetMaterialInput(DamageFEnum);    _assert_(damagef_input);

	/* Start  looping on the number of gaussian points: */
	Gauss* gauss=element->NewGauss(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){
		gauss->GaussPoint(ig);

		element->JacobianDeterminant(&Jdet,xyz_list,gauss);
		element->NodalFunctions(basis,gauss);

		damaged_input->GetInputValue(&damage,gauss);
		damagef_input->GetInputValue(&f,gauss);

		for(int i=0;i<numnodes;i++) pe->values[i]+=Jdet*gauss->weight*(damage+dt*f)*basis[i];
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(xyz_list);
	xDelete<IssmDouble>(basis);
	if(meshtype!=Mesh2DhorizontalEnum){basalelement->DeleteMaterials(); delete basalelement;};
	delete gauss;
	return pe;
}/*}}}*/
void DamageEvolutionAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	   _error_("not implemented yet");
}/*}}}*/
void DamageEvolutionAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	IssmDouble  max_damage;
	int			*doflist = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Fetch dof list and allocate solution vector*/
	element->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numnodes);

	/*Get user-supplied max_damage: */
	element->FindParam(&max_damage,DamageMaxDamageEnum);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numnodes;i++){
		values[i]=solution[doflist[i]];
		/*Check solution*/
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
		/*Enforce D < max_damage and D > 0 */
		if(values[i]>max_damage) values[i]=max_damage;
		else if(values[i]<0.)    values[i]=0.;
	}

	/*Get all inputs and parameters*/
	element->AddMaterialInput(DamageDbarEnum,values,P1Enum);

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<int>(doflist);
}/*}}}*/

/*Intermediaries*/
void DamageEvolutionAnalysis::CreateDamageFInput(Element* element){/*{{{*/

	/*Intermediaries */
	IssmDouble c1,c2,c3,healing,stress_threshold;
	IssmDouble s_xx,s_xy,s_yy;
	IssmDouble J2s,Xis,Psi,PosPsi,NegPsi;
	IssmDouble damage,sigma_xx,sigma_xy,sigma_yy;

	/*Fetch number of vertices and allocate output*/
	int numvertices = element->GetNumberOfVertices();
	IssmDouble* f   = xNew<IssmDouble>(numvertices);

	/*retrieve parameters:*/
	element->FindParam(&c1,DamageC1Enum);
	element->FindParam(&c2,DamageC2Enum);
	element->FindParam(&c3,DamageC3Enum);
	element->FindParam(&healing,DamageHealingEnum);
	element->FindParam(&stress_threshold,DamageStressThresholdEnum);

	/*Compute stress tensor: */
	element->ComputeStressTensor();

	/*retrieve what we need: */
	Input* sigma_xx_input  = element->GetInput(StressTensorxxEnum);     _assert_(sigma_xx_input);
	Input* sigma_xy_input  = element->GetInput(StressTensorxyEnum);     _assert_(sigma_xy_input);
	Input* sigma_yy_input  = element->GetInput(StressTensoryyEnum);     _assert_(sigma_yy_input);
	Input* damage_input    = element->GetMaterialInput(DamageDbarEnum); _assert_(damage_input);

	/*Damage evolution z mapping: */
	Gauss* gauss=element->NewGauss();
	for (int iv=0;iv<numvertices;iv++){
		gauss->GaussVertex(iv);
		
		damage_input->GetInputValue(&damage,gauss);
		sigma_xx_input->GetInputValue(&sigma_xx,gauss);
		sigma_xy_input->GetInputValue(&sigma_xy,gauss);
		sigma_yy_input->GetInputValue(&sigma_yy,gauss);

		s_xx=sigma_xx/(1.-damage);
		s_xy=sigma_xy/(1.-damage);
		s_yy=sigma_yy/(1.-damage);

		J2s=1./sqrt(2.)*sqrt(s_xx*s_xx + s_yy*s_yy + s_xy*s_xy);
		Xis=sqrt(3.0)*J2s;
		Psi=Xis-stress_threshold;
		PosPsi=max(Psi,0.);
		NegPsi=max(-Psi,0.);

		f[iv]= c1*(pow(PosPsi,c2) - healing*pow(NegPsi,c2))*pow((1.-damage),-c3);
	}

	/*Add input*/
	element->AddMaterialInput(DamageFEnum,f,P1Enum);
	
	/*Clean up and return*/
	delete gauss;
}/*}}}*/
