function [elements,grids,loads,constraints,materials]=ModelProcessorPrognostic(md);
%MODELPROCESSORPROGNOSTIC - process model for a prognostic solution
%
%   This routine uses all the informations in the model md and put them
%   into different structures (grids, elements, loads, constrained,materials)
%   that will be used to create the stiffness matrix and load vector.
%   After this routine, the model md should not be called until the end of the
%   solution sequence.
%
%   Usage:
%      [elements,grids,loads,constraints,materials]=ModelProcessorPrognostic(md)

%Allocate grids and elements
elements=struct('element',cell(md.numberofelements,1));
materials=struct('material',cell(0,0));

%Build elements
if strcmpi(md.type,'2d'),
	pos=[1:md.numberofelements]';
	[elements(pos).element]=deal(triaelem);

	elements(pos)=SetStructureField(elements(pos),'element','type','triaelem');
	elements(pos)=SetStructureField(elements(pos),'element','id',pos);
	elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,:));
	elements(pos)=SetStructureField(elements(pos),'element','h',md.thickness(md.elements(pos,1:3)));
	elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:3)));
	elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:3)));
	elements(pos)=SetStructureField(elements(pos),'element','shelf',md.elementoniceshelf(pos));
	elements(pos)=SetStructureField(elements(pos),'element','artificial_diffusivity',md.artificial_diffusivity);

else
	pos=[1:md.numberofelements]';
	[elements(pos).element]=deal(pentaelem);

	elements(pos)=SetStructureField(elements(pos),'element','type','pentaelem');
	elements(pos)=SetStructureField(elements(pos),'element','id',pos);
	elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,1:6));
	elements(pos)=SetStructureField(elements(pos),'element','h',md.thickness(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','shelf',md.elementoniceshelf(pos));
	elements(pos)=SetStructureField(elements(pos),'element','onbed',md.elementonbed(pos));
	elements(pos)=SetStructureField(elements(pos),'element','onsurface',md.elementonsurface(pos));
	elements(pos)=SetStructureField(elements(pos),'element','collapse',1);
	elements(pos)=SetStructureField(elements(pos),'element','artificial_diffusivity',md.artificial_diffusivity);
end

%Get the grids set up:
grids=struct('grid',cell(md.numberofgrids,1));

pos=[1:md.numberofgrids]';
[grids(pos).grid]=deal(node);
grids(pos)=SetStructureField(grids(pos),'grid','id',pos);
grids(pos)=SetStructureField(grids(pos),'grid','x',md.x(pos));
grids(pos)=SetStructureField(grids(pos),'grid','y',md.y(pos));
grids(pos)=SetStructureField(grids(pos),'grid','z',md.z(pos));
grids(pos)=SetStructureField(grids(pos),'grid','s',(md.z(pos)-md.bed(pos))./md.thickness(pos));
grids(pos)=SetStructureField(grids(pos),'grid','surface',md.surface(pos));
grids(pos)=SetStructureField(grids(pos),'grid','onbed',md.gridonbed(pos));

%spc degrees of freedom:
%	 for each grid, 6 degrees of freedom are allowed. These dofs are numbered from 1 to 6. The first 3
%    deal with the (x,y,z) velocities, or deformations. The last 3 deal with the (x,y,z) rotations.
%    If a certain degree of freedom i (1<=i<=6) is constrained to the value 0, the number i should be added to the
%    gridset field of a grid.
%    The gridset field holds all the numbers corresponding to the dofs that have been constrained to 0 value. Because
%    we do not know firshand how many dofs have been constrained for a certain grid, we need a flexible way
%    of keeping track of these constraints. Hence gridset is a string array, of no given size, with no given
%    numerical order.
%    Ex: if a grid has 0 values for the x and z deformations, and 0 values for the y rotation, we could add any of the
%    following strings to the gridset: '135', '153', '315', etc ...
if strcmpi(md.type,'3d')
		%Here, block all dof for every grid
	grids(pos)=SetStructureField(grids(pos),'grid','gridset','123456');
else
	grids(pos)=SetStructureField(grids(pos),'grid','gridset','23456');
end

if strcmpi(md.type,'3d'),
	%for bed elements, free up the 1'st dof for the first 3 grids 
	for n=1:length(elements),
		%all dofs associated to this element are frozen, except if the element is on the bed, and acts as a 'fake' pentaelem, 
		%and a true 'macayeal' tria element.
		if elements(n).element.onbed,
			for j=1:3,
				grids(elements(n).element.g(j)).grid.gridset='23456';
			end
		end
	end
end

%Boundary conditions: penalties between two 3d meshes
loads=struct('load',cell([0,1]));


%Initialize constraints structure
constraints=struct('constraint',cell(length(find(md.gridondirichlet_prog)),1));

pos=find(md.gridondirichlet_prog);
count=1;
for i=1:length(find(md.gridondirichlet_prog)),

		%constrain the thickness
		constraints(count).constraint=spc;
		constraints(count).constraint.grid=pos(i);
		constraints(count).constraint.dof=1;
		constraints(count).constraint.value=md.dirichletvalues_prog(pos(i)); %in m
		count=count+1;
end

end %end function
