%SPHEREMESH class definition
%
%   Usage:
%      spheremesh=spheremesh();

classdef spheremesh
	properties (SetAccess=public) 
		x                           = NaN;
		y                           = NaN;
		z                           = NaN;
		r                           = NaN;
		theta                       = NaN;
		phi                         = NaN
		elements                    = NaN
		dimension                   = 0;
		numberoflayers              = 0;
		numberofelements            = 0;
		numberofvertices            = 0;

		vertexconnectivity          = NaN
		elementconnectivity         = NaN
		average_vertex_connectivity = 0;
	end
	methods
		function obj = spheremesh(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%the connectivity is the avergaded number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			obj.average_vertex_connectivity=25;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','spheremesh.x','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.y','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.z','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.r','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.theta','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.phi','NaN',1,'size',[md.spheremesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','spheremesh.elements','NaN',1,'>',0,'values',1:md.spheremesh.numberofvertices);
			if(md.spheremesh.dimension==2),
				md = checkfield(md,'fieldname','spheremesh.elements','size',[md.spheremesh.numberofelements 3]);
			else
				md = checkfield(md,'fieldname','spheremesh.elements','size',[md.spheremesh.numberofelements 6]);
			end
			if any(~ismember(1:md.spheremesh.numberofvertices,sort(unique(md.spheremesh.elements(:)))));
				md = checkmessage(md,'orphan nodes have been found. Check the spheremesh outline');
			end
			md = checkfield(md,'fieldname','spheremesh.dimension','values',[2 3]);
			md = checkfield(md,'fieldname','spheremesh.numberoflayers','>=',0);
			md = checkfield(md,'fieldname','spheremesh.numberofelements','>',0);
			md = checkfield(md,'fieldname','spheremesh.numberofvertices','>',0);
			if (md.spheremesh.dimension==2),
				md = checkfield(md,'fieldname','spheremesh.average_vertex_connectivity','>=',9,'message','''spheremesh.average_vertex_connectivity'' should be at least 9 in 2d');
			else
				md = checkfield(md,'fieldname','spheremesh.average_vertex_connectivity','>=',24,'message','''spheremesh.average_vertex_connectivity'' should be at least 24 in 3d');
			end
			md = checkfield(md,'fieldname','spheremesh.elementconnectivity','size',[md.spheremesh.numberofelements 3],'NaN',1);

			%Solution specific checks
			switch(solution),
				case MasstransportSolutionEnum(),
					if md.masstransport.stabilization==3,
						md = checkfield(md,'fieldname','spheremesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d spheremeshes');
					end
				case TransientSolutionEnum(),
					if md.transient.ismasstransport & md.masstransport.stabilization==3,
						md = checkfield(md,'fieldname','spheremesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d spheremeshes');
					end
				case ThermalSolutionEnum(),
					md = checkfield(md,'fieldname','spheremesh.dimension','values',3,'message','thermal solution only supported on 3d spheremeshes');
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Mesh:')); 

			disp(sprintf('\n      Elements and vertices:'));
			fielddisplay(obj,'numberofelements','number of elements');
			fielddisplay(obj,'numberofvertices','number of vertices');
			fielddisplay(obj,'elements','vertex indices of the mesh elements');
			fielddisplay(obj,'x','vertices x coordinate [m]');
			fielddisplay(obj,'y','vertices y coordinate [m]');
			fielddisplay(obj,'z','vertices z coordinate [m]');
			fielddisplay(obj,'r','vertices r coordinate [m]');
			fielddisplay(obj,'theta','vertices theta coordinate [degrees]');
			fielddisplay(obj,'phi','vertices phi coordinate [degrees]');

			disp(sprintf('\n      Properties:'));
			fielddisplay(obj,'dimension','spheremesh dimension (2d or 3d)');
			fielddisplay(obj,'numberoflayers','number of extrusion layers');

			fielddisplay(obj,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(obj,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(obj,'average_vertex_connectivity','average number of vertices connected to one vertex');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','z','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','r','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','theta','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','phi','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','dimension','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberoflayers','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',obj,'fieldname','elementconnectivity','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','average_vertex_connectivity','format','Integer');
		end % }}}
	end
end
