#include "./ThermalAnalysis.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

/*Model processing*/
int  ThermalAnalysis::DofsPerNode(int** doflist,int meshtype,int approximation){/*{{{*/
	return 1;
}/*}}}*/
void ThermalAnalysis::UpdateParameters(Parameters* parameters,IoModel* iomodel,int solution_enum,int analysis_enum){/*{{{*/

	int     numoutputs;
	char**  requestedoutputs = NULL;

	parameters->AddObject(iomodel->CopyConstantObject(ThermalMaxiterEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalStabilizationEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalPenaltyFactorEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalPenaltyThresholdEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalPenaltyLockEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalIsenthalpyEnum));
	parameters->AddObject(iomodel->CopyConstantObject(ThermalIsdynamicbasalspcEnum));

	iomodel->FetchData(&requestedoutputs,&numoutputs,ThermalRequestedOutputsEnum);
	parameters->AddObject(new IntParam(ThermalNumRequestedOutputsEnum,numoutputs));
	if(numoutputs)parameters->AddObject(new StringArrayParam(ThermalRequestedOutputsEnum,requestedoutputs,numoutputs));
	iomodel->DeleteData(&requestedoutputs,numoutputs,ThermalRequestedOutputsEnum);

}/*}}}*/
void ThermalAnalysis::UpdateElements(Elements* elements,IoModel* iomodel,int analysis_counter,int analysis_type){/*{{{*/

	/*Now, is the model 3d? otherwise, do nothing: */
	if(iomodel->meshtype==Mesh2DhorizontalEnum)return;

	/*Update elements: */
	int counter=0;
	for(int i=0;i<iomodel->numberofelements;i++){
		if(iomodel->my_elements[i]){
			Element* element=(Element*)elements->GetObjectByOffset(counter);
			element->Update(i,iomodel,analysis_counter,analysis_type,P1Enum);
			counter++;
		}
	}

	bool dakota_analysis;
	iomodel->Constant(&dakota_analysis,QmuIsdakotaEnum);

	iomodel->FetchDataToInput(elements,ThicknessEnum);
	iomodel->FetchDataToInput(elements,SurfaceEnum);
	iomodel->FetchDataToInput(elements,BedEnum);
	iomodel->FetchDataToInput(elements,FrictionCoefficientEnum);
	iomodel->FetchDataToInput(elements,FrictionPEnum);
	iomodel->FetchDataToInput(elements,FrictionQEnum);
	iomodel->FetchDataToInput(elements,MaskIceLevelsetEnum);
	iomodel->FetchDataToInput(elements,MaskGroundediceLevelsetEnum);
	iomodel->FetchDataToInput(elements,MeshElementonbedEnum);
	iomodel->FetchDataToInput(elements,MeshElementonsurfaceEnum);
	iomodel->FetchDataToInput(elements,FlowequationElementEquationEnum);
	iomodel->FetchDataToInput(elements,MaterialsRheologyBEnum);
	iomodel->FetchDataToInput(elements,MaterialsRheologyNEnum);
	iomodel->FetchDataToInput(elements,PressureEnum);
	iomodel->FetchDataToInput(elements,TemperatureEnum);
	iomodel->FetchDataToInput(elements,BasalforcingsGeothermalfluxEnum);
	iomodel->FetchDataToInput(elements,VxEnum);
	iomodel->FetchDataToInput(elements,VyEnum);
	iomodel->FetchDataToInput(elements,VzEnum);
	InputUpdateFromConstantx(elements,0.,VxMeshEnum);
	InputUpdateFromConstantx(elements,0.,VyMeshEnum);
	InputUpdateFromConstantx(elements,0.,VzMeshEnum);
	if(dakota_analysis){
		elements->InputDuplicate(TemperatureEnum,QmuTemperatureEnum);
		elements->InputDuplicate(BasalforcingsMeltingRateEnum,QmuMeltingEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVxMeshEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVyMeshEnum);
		elements->InputDuplicate(VxMeshEnum,QmuVzMeshEnum);
	}
}/*}}}*/
void ThermalAnalysis::CreateNodes(Nodes* nodes,IoModel* iomodel){/*{{{*/

	if(iomodel->meshtype==Mesh3DEnum) iomodel->FetchData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
	::CreateNodes(nodes,iomodel,ThermalAnalysisEnum,P1Enum);
	iomodel->DeleteData(2,MeshVertexonbedEnum,MeshVertexonsurfaceEnum);
}/*}}}*/
void ThermalAnalysis::CreateConstraints(Constraints* constraints,IoModel* iomodel){/*{{{*/

	/*Only 3d mesh supported*/
	if(iomodel->meshtype==Mesh3DEnum){
		IoModelToConstraintsx(constraints,iomodel,ThermalSpctemperatureEnum,ThermalAnalysisEnum,P1Enum);
	}

}/*}}}*/
void ThermalAnalysis::CreateLoads(Loads* loads, IoModel* iomodel){/*{{{*/

	if(iomodel->meshtype==Mesh2DhorizontalEnum) _error_("2d meshes not supported yet");

	/*create penalties for nodes: no node can have a temperature over the melting point*/
	iomodel->FetchData(1,ThermalSpctemperatureEnum);
	CreateSingleNodeToElementConnectivity(iomodel);

	for(int i=0;i<iomodel->numberofvertices;i++){

		/*keep only this partition's nodes:*/
		if(iomodel->my_vertices[i]){
			if (xIsNan<IssmDouble>(iomodel->Data(ThermalSpctemperatureEnum)[i])){ //No penalty applied on spc nodes!
				loads->AddObject(new Pengrid(iomodel->loadcounter+i+1,i,iomodel,ThermalAnalysisEnum));
			}
		}
	}
	iomodel->DeleteData(1,ThermalSpctemperatureEnum);

}/*}}}*/

/*Numerics*/
void ThermalAnalysis::GetSolutionFromInputs(Vector<IssmDouble>* solution,Element* element){/*{{{*/
	element->GetSolutionFromInputsOneDof(solution,TemperatureEnum);
}/*}}}*/
void ThermalAnalysis::InputUpdateFromSolution(IssmDouble* solution,Element* element){/*{{{*/

	bool        converged;
	int         i,rheology_law;
	IssmDouble  B_average,s_average,T_average=0.;
	int        *doflist   = NULL;
	IssmDouble *xyz_list  = NULL;
	bool        hack      = false;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = element->GetNumberOfNodes();

	/*Fetch dof list and allocate solution vector*/
	element->GetDofList(&doflist,NoneApproximationEnum,GsetEnum);
	IssmDouble* values    = xNew<IssmDouble>(numnodes);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numnodes;i++){
		values[i]=solution[doflist[i]];

		/*Check solution*/
		if(xIsNan<IssmDouble>(values[i])) _error_("NaN found in solution vector");
		//if(values[i]<0)      _printf_("temperature < 0°K found in solution vector\n");
		//if(values[i]>275)    _printf_("temperature > 275°K found in solution vector (Paterson's rheology associated is negative)\n");
	}

	/*Force temperature between [Tpmp-50 Tpmp] to disable penalties*/
	if(hack){
		IssmDouble* pressure = xNew<IssmDouble>(numnodes);
		element->GetInputListOnVertices(pressure,PressureEnum);
		for(i=0;i<numnodes;i++){
			if(values[i]>element->TMeltingPoint(pressure[i]))     values[i]=element->TMeltingPoint(pressure[i]);
			if(values[i]<element->TMeltingPoint(pressure[i])-50.) values[i]=element->TMeltingPoint(pressure[i])-50.;
		}
		xDelete<IssmDouble>(pressure);
	}

	/*Get all inputs and parameters*/

	element->GetInputValue(&converged,ConvergedEnum);
	if(converged){
		element->AddInput(TemperatureEnum,values,P1Enum);

		/*Update Rheology only if converged (we must make sure that the temperature is below melting point
		 * otherwise the rheology could be negative*/
		element->FindParam(&rheology_law,MaterialsRheologyLawEnum);
		switch(rheology_law){
			case NoneEnum:
				/*Do nothing: B is not temperature dependent*/
				break;
			case PatersonEnum:
				for(i=0;i<numnodes;i++) T_average+=values[i]/reCast<IssmDouble>(numnodes);
				B_average=Paterson(T_average);
				element->AddMaterialInput(MaterialsRheologyBEnum,&B_average,P0Enum);
				break;
			case ArrheniusEnum:{
				Input* surface_input=element->GetInput(SurfaceEnum); _assert_(surface_input);
				surface_input->GetInputAverage(&s_average);
				element->GetVerticesCoordinates(&xyz_list);
				for(i=0;i<numnodes;i++) T_average+=values[i]/reCast<IssmDouble>(numnodes);
				//B_average=Arrhenius(T_average,
							//s_average-((xyz_list[0][2]+xyz_list[1][2]+xyz_list[2][2]+xyz_list[3][2]+xyz_list[4][2]+xyz_list[5][2])/6.0),
							//element->GetMaticeParameter(MaterialsRheologyNEnum));
				element->AddMaterialInput(MaterialsRheologyBEnum,&B_average,P0Enum);
				break;
				}
			default:
				_error_("Rheology law " << EnumToStringx(rheology_law) << " not supported yet");

		}
	}
	else{
		element->AddInput(TemperaturePicardEnum,values,P1Enum);
	}

	/*Free ressources:*/
	xDelete<IssmDouble>(values);
	xDelete<IssmDouble>(xyz_list);
	xDelete<int>(doflist);
}/*}}}*/
