/*!\file Seg.cpp
 * \brief: implementation of the Segment object
 */
/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../classes.h"
#include "../../shared/shared.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES 2
/*Constructors/destructor/copy*/
/*FUNCTION Seg::Seg(){{{*/
Seg::Seg(){
	this->nodes      = NULL;
	this->vertices   = NULL;
	this->material   = NULL;
	this->matpar     = NULL;
	this->inputs     = NULL;
	this->parameters = NULL;
}
/*}}}*/
/*FUNCTION Seg::Seg(int id, int sid,int index, IoModel* iomodel,int nummodels){{{*/
Seg::Seg(int seg_id, int seg_sid, int index, IoModel* iomodel,int nummodels)
		:SegRef(nummodels),ElementHook(nummodels,index+1,2,iomodel){

			/*id: */
			this->id  = seg_id;
			this->sid = seg_sid;

			//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
			this->parameters = NULL;

			/*intialize inputs: */
			this->inputs  = new Inputs();

			/*initialize pointers:*/
			this->nodes    = NULL;
			this->vertices = NULL;
			this->material = NULL;
			this->matpar   = NULL;

		}
/*}}}*/
/*FUNCTION Seg::~Seg(){{{*/
Seg::~Seg(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Seg::copy {{{*/
Object* Seg::copy() {
	_error_("not implemented yet");
}
/*}}}*/

/*FUNCTION Seg::Echo{{{*/
void Seg::Echo(void){
	_printf_("Seg:\n");
	_printf_("   id: " << id << "\n");
	if(nodes){
		nodes[0]->Echo();
		nodes[1]->Echo();
	}
	else _printf_("nodes = NULL\n");

	if (material) material->Echo();
	else _printf_("material = NULL\n");

	if (matpar) matpar->Echo();
	else _printf_("matpar = NULL\n");

	_printf_("   parameters\n");
	if (parameters) parameters->Echo();
	else _printf_("parameters = NULL\n");

	_printf_("   inputs\n");
	if (inputs) inputs->Echo();
	else _printf_("inputs=NULL\n");
}
/*}}}*/
/*FUNCTION Seg::FindParam(int* pvalue,int paramenum){{{*/
void Seg::FindParam(int* pvalue,int paramenum){
	this->parameters->FindParam(pvalue,paramenum);
}
/*}}}*/
/*FUNCTION Seg::FindParam(IssmDouble* pvalue,int paramenum){{{*/
void Seg::FindParam(IssmDouble* pvalue,int paramenum){
	this->parameters->FindParam(pvalue,paramenum);
}
/*}}}*/
/*FUNCTION Seg::FiniteElement{{{*/
int Seg::FiniteElement(void){
	return this->element_type;
}
/*}}}*/
/*FUNCTION Seg::DeepEcho{{{*/
void Seg::DeepEcho(void){

	_printf_("Seg:\n");
	_printf_("   id: " << id << "\n");
	if(nodes){
		nodes[0]->DeepEcho();
		nodes[1]->DeepEcho();
	}
	else _printf_("nodes = NULL\n");

	if (material) material->DeepEcho();
	else _printf_("material = NULL\n");

	if (matpar) matpar->DeepEcho();
	else _printf_("matpar = NULL\n");

	_printf_("   parameters\n");
	if (parameters) parameters->DeepEcho();
	else _printf_("parameters = NULL\n");

	_printf_("   inputs\n");
	if (inputs) inputs->DeepEcho();
	else _printf_("inputs=NULL\n");

	return;
}
/*}}}*/
/*FUNCTION Seg::ObjectEnum{{{*/
int Seg::ObjectEnum(void){

	return SegEnum;

}
/*}}}*/
/*FUNCTION Seg::Id {{{*/
int    Seg::Id(){

	return id;

}
/*}}}*/

/*FUNCTION Seg::GetMaterialParameter{{{*/
IssmDouble Seg::GetMaterialParameter(int enum_in){

	_assert_(this->matpar);
	return this->matpar->GetMaterialParameter(enum_in);
}
/*}}}*/
/*FUNCTION Seg::GetSize{{{*/
IssmDouble Seg::GetSize(void){

	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble x1,y1,x2,y2;

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];

	return sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1));
}
/*}}}*/
/*FUNCTION Seg::CreateKMatrixFreeSurfaceTop {{{*/
ElementMatrix* Seg::CreateKMatrixFreeSurfaceTop(void){

	/*Intermediaries */
	int        stabilization;
	IssmDouble Jdet,D_scalar,dt,h;
	IssmDouble vx,vel;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*Fetch number of nodes for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element matrix and vectors*/
	ElementMatrix* Ke     = new ElementMatrix(nodes,numnodes,this->parameters,NoneApproximationEnum);
	IssmDouble*    basis  = xNew<IssmDouble>(numnodes);
	IssmDouble*    B      = xNew<IssmDouble>(1*numnodes);
	IssmDouble*    Bprime = xNew<IssmDouble>(1*numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	this->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	this->parameters->FindParam(&stabilization,MasstransportStabilizationEnum);
	Input* vx_input=inputs->GetInput(VxEnum); _assert_(vx_input);
	h=this->GetSize();

	/* Start  looping on the number of gaussian points: */
	GaussSeg *gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);

		vx_input->GetInputValue(&vx,gauss);

		D_scalar=gauss->weight*Jdet;

		TripleMultiply(basis,1,numnodes,1,
					&D_scalar,1,1,0,
					basis,1,numnodes,0,
					&Ke->values[0],1);

		GetNodalFunctions(B,gauss);
		GetBprimeMasstransport(Bprime,&xyz_list[0][0],gauss);

		D_scalar=dt*gauss->weight*Jdet*vx;
		TripleMultiply(B,1,numnodes,1,
					&D_scalar,1,1,0,
					Bprime,1,numnodes,0,
					&Ke->values[0],1);

		if(stabilization==2){
			/*Streamline upwinding*/
			vel=fabs(vx)+1.e-8;
			D_scalar=dt*gauss->weight*Jdet*h/(2.*vel)*vx;
		}
		else if(stabilization==1){
			/*SSA*/
			vx_input->GetInputAverage(&vx);
			D_scalar=dt*gauss->weight*Jdet*h/2.*fabs(vx);
		}
		if(stabilization==1 || stabilization==2){
			TripleMultiply(Bprime,1,numnodes,1,
						&D_scalar,1,1,0,
						Bprime,1,numnodes,0,
						&Ke->values[0],1);
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(Bprime);
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Seg::CreateKMatrixFreeSurfaceBase {{{*/
ElementMatrix* Seg::CreateKMatrixFreeSurfaceBase(void){

	/*Intermediaries */
	int        stabilization;
	IssmDouble Jdet,D_scalar,dt,h;
	IssmDouble vx,vel;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*Fetch number of nodes for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element matrix and vectors*/
	ElementMatrix* Ke     = new ElementMatrix(nodes,numnodes,this->parameters,NoneApproximationEnum);
	IssmDouble*    basis  = xNew<IssmDouble>(numnodes);
	IssmDouble*    B      = xNew<IssmDouble>(1*numnodes);
	IssmDouble*    Bprime = xNew<IssmDouble>(1*numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	this->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	this->parameters->FindParam(&stabilization,MasstransportStabilizationEnum);
	Input* vx_input=inputs->GetInput(VxEnum); _assert_(vx_input);
	h=this->GetSize();

	/* Start  looping on the number of gaussian points: */
	GaussSeg *gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);

		vx_input->GetInputValue(&vx,gauss);

		D_scalar=gauss->weight*Jdet;

		TripleMultiply(basis,1,numnodes,1,
					&D_scalar,1,1,0,
					basis,1,numnodes,0,
					&Ke->values[0],1);

		GetNodalFunctions(B,gauss);
		GetBprimeMasstransport(Bprime,&xyz_list[0][0],gauss);

		D_scalar=dt*gauss->weight*Jdet*vx;
		TripleMultiply(B,1,numnodes,1,
					&D_scalar,1,1,0,
					Bprime,1,numnodes,0,
					&Ke->values[0],1);

		if(stabilization==2){
			/*Streamline upwinding*/
			vel=fabs(vx)+1.e-8;
			D_scalar=dt*gauss->weight*Jdet*h/(2.*vel)*vx;
		}
		else if(stabilization==1){
			/*SSA*/
			vx_input->GetInputAverage(&vx);
			D_scalar=dt*gauss->weight*Jdet*h/2.*fabs(vx);
		}
		if(stabilization==1 || stabilization==2){
			TripleMultiply(Bprime,1,numnodes,1,
						&D_scalar,1,1,0,
						Bprime,1,numnodes,0,
						&Ke->values[0],1);
		}
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	xDelete<IssmDouble>(B);
	xDelete<IssmDouble>(Bprime);
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Seg::CreateMassMatrix {{{*/
ElementMatrix* Seg::CreateMassMatrix(void){

	/* Intermediaries */
	IssmDouble  D,Jdet;
	IssmDouble  xyz_list[NUMVERTICES][3];

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element matrix and vectors*/
	ElementMatrix* Ke    = new ElementMatrix(nodes,numnodes,this->parameters,NoneApproximationEnum);
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);

	/* Start looping on the number of gaussian points: */
	GaussSeg* gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetNodalFunctions(basis,gauss);
		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		D=gauss->weight*Jdet;

		TripleMultiply(basis,1,numnodes,1,
					&D,1,1,0,
					basis,1,numnodes,0,
					&Ke->values[0],1);
	}

	/*Clean up and return*/
	delete gauss;
	xDelete<IssmDouble>(basis);
	return Ke;
}
/*}}}*/
/*FUNCTION Seg::CreatePVectorL2Projection {{{*/
ElementVector* Seg::CreatePVectorL2Projection(void){

	/*Intermediaries */
	int        i,input_enum;
	IssmDouble Jdet;
	IssmDouble xyz_list[NUMVERTICES][3];
	IssmDouble value;
	Input*     input  = NULL;
	Input*     input2 = NULL;

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element vector*/
	ElementVector* pe    = new ElementVector(nodes,numnodes,this->parameters);
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	this->parameters->FindParam(&input_enum,InputToL2ProjectEnum);
	switch(input_enum){
		case SurfaceSlopeXEnum: input2 = inputs->GetInput(SurfaceEnum); _assert_(input2); break;
		case BedSlopeXEnum:     input2 = inputs->GetInput(BedEnum);     _assert_(input2); break;
		default: input = inputs->GetInput(input_enum);
	}

	/* Start  looping on the number of gaussian points: */
	GaussSeg* gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);

		if(input2){
			input2->GetInputDerivativeValue(&value,&xyz_list[0][0],gauss);
		}
		else{
			input->GetInputValue(&value,gauss);
		}

		for(i=0;i<numnodes;i++) pe->values[i]+=Jdet*gauss->weight*value*basis[i];
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Seg::CreatePVectorFreeSurfaceTop {{{*/
ElementVector* Seg::CreatePVectorFreeSurfaceTop(void){

	/*Intermediaries */
	IssmDouble Jdet,dt;
	IssmDouble ms,surface,vy;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element vector and other vectors*/
	ElementVector* pe    = new ElementVector(nodes,numnodes,this->parameters);
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	this->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	Input* vy_input     = inputs->GetInput(VyEnum);                         _assert_(vy_input);
	Input* ms_input     = inputs->GetInput(SurfaceforcingsMassBalanceEnum); _assert_(ms_input);
	Input* surface_input= inputs->GetInput(SurfaceEnum);                    _assert_(surface_input);

	/*Initialize mb_correction to 0, do not forget!:*/
	/* Start  looping on the number of gaussian points: */
	GaussSeg* gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);

		vy_input->GetInputValue(&vy,gauss);
		ms_input->GetInputValue(&ms,gauss);
		surface_input->GetInputValue(&surface,gauss);

		for(int i=0;i<numnodes;i++) pe->values[i]+=Jdet*gauss->weight*(surface + dt*ms + dt*vy)*basis[i];
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Seg::CreatePVectorFreeSurfaceBase {{{*/
ElementVector* Seg::CreatePVectorFreeSurfaceBase(void){

	/*Intermediaries */
	IssmDouble Jdet,dt;
	IssmDouble mb,mb_correction,bed,vy;
	IssmDouble xyz_list[NUMVERTICES][3];

	/*Fetch number of nodes and dof for this finite element*/
	int numnodes = this->NumberofNodes();

	/*Initialize Element vector and other vectors*/
	ElementVector* pe    = new ElementVector(nodes,numnodes,this->parameters);
	IssmDouble*    basis = xNew<IssmDouble>(numnodes);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0],vertices,NUMVERTICES);
	this->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	Input* vy_input            = inputs->GetInput(VyEnum);                         _assert_(vy_input);
	Input* mb_input            = inputs->GetInput(BasalforcingsMeltingRateEnum);   _assert_(mb_input);
	Input* mb_correction_input = inputs->GetInput(BasalforcingsMeltingRateCorrectionEnum);
	Input* bed_input           = inputs->GetInput(BedEnum);                        _assert_(bed_input);

	/*Initialize mb_correction to 0, do not forget!:*/
	/* Start  looping on the number of gaussian points: */
	GaussSeg* gauss=new GaussSeg(2);
	for(int ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);

		vy_input->GetInputValue(&vy,gauss);
		mb_input->GetInputValue(&mb,gauss);
		bed_input->GetInputValue(&bed,gauss);
		if(mb_correction_input)
		 mb_correction_input->GetInputValue(&mb_correction,gauss);
		else
		 mb_correction=0.;

		for(int i=0;i<numnodes;i++) pe->values[i]+=Jdet*gauss->weight*(bed+dt*(mb-mb_correction) + dt*vy)*basis[i];
	}

	/*Clean up and return*/
	xDelete<IssmDouble>(basis);
	delete gauss;
	return pe;
}
/*}}}*/
