/*!\file ModelProcessorx
 * \brief: create datasets using input binary file and a set of requested analyses
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "./ModelProcessorx.h"

void ModelProcessorx(Elements** pelements, Nodes** pnodes, Vertices** pvertices, Materials** pmaterials, Constraints** pconstraints, Loads** ploads, Parameters** pparameters, FILE* IOMODEL,FILE* toolkitfile, char* rootpath,const int solution_enum,const int nummodels,const int* analysis_type_list){

	int   i,analysis_type,verbose;
	bool  isthermal,ismasstransport,isstressbalance,isgroundingline,isenthalpy;

	/*Initialize datasets*/
	Elements    *elements    = new Elements();
	Nodes       *nodes       = new Nodes();
	Vertices    *vertices    = new Vertices();
	Materials   *materials   = new Materials();
	Constraints *constraints = new Constraints();
	Loads       *loads       = new Loads();
	Parameters  *parameters  = new Parameters();

	/*Initialize IoModel from input file*/
	IoModel* iomodel = new IoModel(IOMODEL);

	/*Fetch parameters: */
	iomodel->Constant(&verbose,VerboseEnum);
	iomodel->Constant(&isthermal,TransientIsthermalEnum);
	iomodel->Constant(&isenthalpy,ThermalIsenthalpyEnum);
	iomodel->Constant(&ismasstransport,TransientIsmasstransportEnum);
	iomodel->Constant(&isstressbalance,TransientIsstressbalanceEnum);
	iomodel->Constant(&isgroundingline,TransientIsgroundinglineEnum);

	SetVerbosityLevel(verbose);

	if(VerboseMProcessor()) _printf0_("   starting model processor \n");

	/*Partition Elements and Nodes*/
	ElementsAndVerticesPartitioning(&iomodel->my_elements,&iomodel->my_vertices,iomodel);

	/*Create elements, vertices and materials, independent of analysis_enum: */
	CreateElementsVerticesAndMaterials(elements,vertices,materials,iomodel,nummodels);

	/*Create Parameters*/
	CreateParameters(parameters,iomodel,rootpath,toolkitfile,solution_enum);

	for(i=0;i<nummodels;i++){

		analysis_type=analysis_type_list[i];
		parameters->AddObject(new IntParam(AnalysisCounterEnum,i));

		/*Hack for trasient runs (FIXME: to be improved)*/
		if(solution_enum==TransientSolutionEnum && analysis_type==ThermalAnalysisEnum  && iomodel->meshtype==Mesh2DhorizontalEnum) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==MeltingAnalysisEnum  && iomodel->meshtype==Mesh2DhorizontalEnum) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==EnthalpyAnalysisEnum && iomodel->meshtype==Mesh2DhorizontalEnum) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==ThermalAnalysisEnum && isthermal==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==MeltingAnalysisEnum && isthermal==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==EnthalpyAnalysisEnum && isthermal==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==ThermalAnalysisEnum && isenthalpy==true) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==MeltingAnalysisEnum && isenthalpy==true) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==EnthalpyAnalysisEnum && isenthalpy==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==MasstransportAnalysisEnum && ismasstransport==false && isgroundingline==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==StressbalanceAnalysisEnum && isstressbalance==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==StressbalanceVerticalAnalysisEnum && isstressbalance==false) continue;
		if(solution_enum==TransientSolutionEnum && analysis_type==StressbalanceSIAAnalysisEnum && isstressbalance==false) continue;
		if(solution_enum==SteadystateSolutionEnum && analysis_type==ThermalAnalysisEnum && isenthalpy==true) continue;
		if(solution_enum==SteadystateSolutionEnum && analysis_type==MeltingAnalysisEnum && isenthalpy==true) continue;
		if(solution_enum==SteadystateSolutionEnum && analysis_type==EnthalpyAnalysisEnum && isenthalpy==false) continue;

		if(VerboseMProcessor()) _printf0_("   creating datasets for analysis " << EnumToStringx(analysis_type) << "\n");
		CreateDataSets(elements,nodes,vertices,materials,constraints,loads,parameters,iomodel,toolkitfile,rootpath,solution_enum,analysis_type,nummodels,i);
	}
	if(VerboseMProcessor()) _printf0_("   done with model processor \n");

	/*Free resources:*/
	delete iomodel;

	/*Assign output pointers:*/
	*pelements    = elements;
	*pnodes       = nodes;
	*pvertices    = vertices;
	*pmaterials   = materials;
	*pconstraints = constraints;
	*ploads       = loads;
	*pparameters  = parameters;
}
