/*!\file: CreateDataSets
 * \brief general driver for creating all datasets that make a finite element iomodel
 */ 

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "../MeshPartitionx/MeshPartitionx.h"
#include "./ModelProcessorx.h"

void CreateDataSets(Elements** pelements,Nodes** pnodes, Vertices** pvertices, Materials** pmaterials, Constraints** pconstraints, Loads** ploads,Parameters** pparameters,IoModel* iomodel,FILE* toolkitfile,char* rootpath,const int solution_enum,const int analysis_enum,const int nummodels,int analysis_counter){

	Elements   *elements   = NULL;
	Materials  *materials  = NULL;
	Parameters *parameters = NULL;

	/*Process Finite Element Mesh*/

	/*Partition Elements and Nodes*/
	ElementsAndVerticesPartitioning(&iomodel->my_elements,&iomodel->my_vertices,iomodel);

	/*Create elements, vertices and materials, independent of analysis_enum: */
	CreateElementsVerticesAndMaterials(pelements, pvertices, pmaterials, iomodel,nummodels);

	/*Create Parameters*/
	CreateParameters(pparameters,iomodel,rootpath,toolkitfile,solution_enum,analysis_enum,analysis_counter);

	/*Recover elements and materials, for future update: */
	elements  = *pelements;
	materials = *pmaterials;

	/*Creates Nodes and constraints datasets if empty*/
	if(!*pnodes)       *pnodes       = new Nodes();
	if(!*pconstraints) *pconstraints = new Constraints();
	if(!*ploads)       *ploads       = new Loads();

	/*Now, branch onto analysis dependent model generation: */
	Analysis* analysis = EnumToAnalysis(analysis_enum);
	analysis->UpdateParameters(pparameters,iomodel,solution_enum,analysis_enum);
	analysis->CreateNodes(pnodes,iomodel);
	analysis->CreateConstraints(pconstraints,iomodel);
	analysis->CreateLoads(ploads,iomodel);
	analysis->UpdateElements(elements,iomodel,analysis_counter,analysis_enum);
	delete analysis;

	/*Update Elements and Materials For Inversions*/
	#ifdef _HAVE_CONTROL_
	UpdateElementsAndMaterialsControl(elements,materials,iomodel);
	#endif

	/*Update Elements and Materials For Dakota*/
	#ifdef _HAVE_DAKOTA_
	UpdateElementsAndMaterialsDakota(elements,materials,iomodel);
	#endif


	/*Update Elements in case we are running a transient solution: */
	#ifdef _HAVE_TRANSIENT_
	parameters=*pparameters;
	if(analysis_counter==(nummodels-1)&& solution_enum==TransientSolutionEnum){
		UpdateElementsTransient(elements,parameters,iomodel,analysis_counter,analysis_enum);
	}
	#endif

	/*Sort datasets: */
	SortDataSets(pelements,pnodes,pvertices, ploads, pmaterials, pconstraints, pparameters);

	/* Update counters, because we have created more nodes, loads and
	 * constraints, and ids for objects created in next call to CreateDataSets
	 * will need to start at the end of the updated counters: */
	UpdateCounters(iomodel,pnodes,ploads,pconstraints);
}
