function [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorThermal(md,solutiontype);
%MODELPROCESSORTHERMAL - process model for a thermal solution
%
%   This routine uses all the informations in the model md and put them
%   into different structures (grids, elements, loads, constrained,materials)
%   that will be used to create the stiffness matrix and load vector.
%   After this routine, the model md should not be called until the end of the
%   solution sequence.
%
%   Usage:
%      [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorThermal(md,solutiontype)

%Allocate grids and elements
elements=struct('element',cell(md.numberofelements,1));
materials=struct('material',cell(md.numberofelements+1,1));

%Build 3d elements
pos=[1:md.numberofelements]';
[elements(pos).element]=deal(pentaelem);

elements(pos)=SetStructureField(elements(pos),'element','type','pentaelem');
elements(pos)=SetStructureField(elements(pos),'element','id',pos);
elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,1:6));
elements(pos)=SetStructureField(elements(pos),'element','h',md.thickness(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','friction_type',md.drag_type);
elements(pos)=SetStructureField(elements(pos),'element','k',md.drag(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','p',md.p(pos));
elements(pos)=SetStructureField(elements(pos),'element','q',md.q(pos));
elements(pos)=SetStructureField(elements(pos),'element','shelf',md.elementoniceshelf(pos));
elements(pos)=SetStructureField(elements(pos),'element','onbed',md.elementonbed(pos));
elements(pos)=SetStructureField(elements(pos),'element','onsurface',md.elementonsurface(pos));
elements(pos)=SetStructureField(elements(pos),'element','meanvel',md.meanvel);
elements(pos)=SetStructureField(elements(pos),'element','epsvel',md.epsvel);
elements(pos)=SetStructureField(elements(pos),'element','collapse',0);
elements(pos)=SetStructureField(elements(pos),'element','matid',pos);
elements(pos)=SetStructureField(elements(pos),'element','geothermalflux',md.geothermalflux(md.elements(pos,1:6)));
%For penta elements where we want to implement MacAyeal's element, we need to collapse 
%the formulation into trias: 
el6pos=find(md.elements_type(:,1)==MacAyealFormulationEnum | md.elements_type(:,1)==PattynFormulationEnum);
pos2=find(ismember(el6pos,find(md.elements_type(:,1)==MacAyealFormulationEnum())));
elements(pos2)=SetStructureField(elements(pos2),'element','collapse',ones(length(pos2),1));

if strcmpi(solutiontype,'thermalsteady'),
	elements(pos)=SetStructureField(elements(pos),'element','thermal_steadystate',1);
else
	elements(pos)=SetStructureField(elements(pos),'element','thermal_steadystate',0);
end

[materials(pos).material]=deal(matice);

materials(pos)=SetStructureField(materials(pos),'material','id',pos);
materials(pos)=SetStructureField(materials(pos),'material','B',md.B(md.elements(pos,1:6))*[1;1;1;1;1;1]/6);
materials(pos)=SetStructureField(materials(pos),'material','n',md.n(pos));

%Add physical constants in materials
[materials(end).constants]=deal(matpar);
materials(end)=SetStructureField(materials(end),'constants','g',md.g);
materials(end)=SetStructureField(materials(end),'constants','rho_ice',md.rho_ice);
materials(end)=SetStructureField(materials(end),'constants','rho_water',md.rho_water);
materials(end)=SetStructureField(materials(end),'constants','thermalconductivity',md.thermalconductivity);
materials(end)=SetStructureField(materials(end),'constants','heatcapacity',md.heatcapacity);
materials(end)=SetStructureField(materials(end),'constants','latentheat',md.latentheat);
materials(end)=SetStructureField(materials(end),'constants','beta',md.beta);
materials(end)=SetStructureField(materials(end),'constants','meltingpoint',md.meltingpoint);
materials(end)=SetStructureField(materials(end),'constants','mixed_layer_capacity',md.mixed_layer_capacity);
materials(end)=SetStructureField(materials(end),'constants','thermal_exchange_velocity',md.thermal_exchange_velocity);

%Get the grids set up:
grids=struct('grid',cell(md.numberofgrids,1));

pos=[1:md.numberofgrids]';
[grids(pos).grid]=deal(node);
grids(pos)=SetStructureField(grids(pos),'grid','id',pos);
grids(pos)=SetStructureField(grids(pos),'grid','x',md.x(pos));
grids(pos)=SetStructureField(grids(pos),'grid','y',md.y(pos));
grids(pos)=SetStructureField(grids(pos),'grid','z',md.z(pos));
grids(pos)=SetStructureField(grids(pos),'grid','s',(md.z(pos)-md.bed(pos))./md.thickness(pos));
grids(pos)=SetStructureField(grids(pos),'grid','surface',md.surface(pos));
grids(pos)=SetStructureField(grids(pos),'grid','onbed',md.gridonbed(pos));

%spc degrees of freedom:
%	 for each grid, 6 degrees of freedom are allowed. These dofs are numbered from 1 to 6. The first 3
%    deal with the (x,y,z) velocities, or deformations. The last 3 deal with the (x,y,z) rotations.
%    If a certain degree of freedom i (1<=i<=6) is constrained to the value 0, the number i should be added to the
%    gridset field of a grid.
%    The gridset field holds all the numbers corresponding to the dofs that have been constrained to 0 value. Because
%    we do not know firshand how many dofs have been constrained for a certain grid, we need a flexible way
%    of keeping track of these constraints. Hence gridset is a string array, of no given size, with no given
%    numerical order.
%    Ex: if a grid has 0 values for the x and z deformations, and 0 values for the y rotation, we could add any of the
%    following strings to the gridset: '135', '153', '315', etc ...
grids(pos)=SetStructureField(grids(pos),'grid','gridset','23456');

%Boundary conditions:

%penalties
loads=struct('load',cell(0,1));

%create thermal penalties for every grid. Temperatures can never go over the melting point. 
count=1;
for i=1:md.numberofgrids,
	if ~md.gridondirichlet_thermal(i),  %No penalty applied on spc grids!
		pengridobject=pengrid;
		pengridobject.id=count;
		pengridobject.dof=1;
		pengridobject.grid=i;
		pengridobject.active=0;
		if strcmpi(solutiontype,'thermalsteady'),
			pengridobject.thermal_steadystate=1;
		else
			pengridobject.thermal_steadystate=0;
		end
		pengridobject.penalty_offset=md.penalty_offset;

		loads(count).load=pengridobject;
		count=count+1;
	end
end

%Single point constraints:
spcs=find(md.gridondirichlet_thermal);
constraints=struct('constraint',cell(length(spcs),1));

count=1;
for i=1:md.numberofgrids,
	if md.gridondirichlet_thermal(i),
		constraints(count).constraint=spc;
		constraints(count).constraint.grid=i;
		constraints(count).constraint.dof=1;
		constraints(count).constraint.value=md.dirichletvalues_thermal(i,1);
		count=count+1;
	end
end


%Last thing, return a partitioning vector, and its transpose.
[part,tpart]=PartitioningVector(md,grids);


end %end function
